package com.artfess.cqxy.contract.controller;

import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.controller.BaseController;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.util.BeanUtils;
import com.artfess.cqxy.contract.manager.ContractManager;
import com.artfess.cqxy.contract.model.Contract;
import com.artfess.cqxy.contract.vo.ContractVo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.util.Assert;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.Arrays;
import java.util.List;

/**
 * 合同管理表(BizContract)表控制层
 *
 * @author 黎沐华
 * @since 2022-02-26 10:35:03
 */
@Slf4j
@RestController
@Api(tags = "合同管理接口")
@ApiGroup(group = {ApiGroupConsts.GROUP_BIZ})
@RequestMapping("/biz/contract/v1")
public class ContractController extends BaseController<ContractManager, Contract> {

    /**
     * 保存或修改数据
     * @param contractVo 实体对象
     * @return 新增或更新结果
     */
    @PostMapping("/save")
    @ApiOperation(value = "S-保存或修改数据",httpMethod = "POST")
    public CommonResult insert(@RequestBody ContractVo contractVo) {
        log.debug("合同管理-保存或修改数据接口请求参数：{}", contractVo);
        Assert.notNull(contractVo.getContract().getProjectId(),"项目ID不能为空");
        boolean add = StringUtils.isEmpty(contractVo.getContract().getId());
        boolean save = add ? baseService.saveContract(contractVo) : baseService.updateContract(contractVo);
        String msg = (add ? "添加" : "修改")+(save ? "成功" : "失败");
        log.debug("合同管理-保存或修改数据接口响应结果：{}", msg);
        return new CommonResult(save, msg);
    }

    /**
     * 批量导入Excel数据
     * @param file Excel文件
     */
    @PostMapping("/importExcelData")
    @ApiOperation(value = "S-批量导入Excel数据")
    public CommonResult importExcelData (@ApiParam("上传文件") @RequestParam("file") MultipartFile file, String projectId){
        log.debug("批量导入Excel数据接口请求参数：{}",file, projectId);
        baseService.importExcelData(file, projectId);
        return new CommonResult<>("导入成功");
    }

    /**
     * 导出数据到Excel
     * @param queryFilter 通用查询器
     * @param response 响应对象
     * @return 通用结果
     * @throws IOException IO异常
     */
    @PostMapping("/exportDataToExcel")
    @ApiOperation(value = "S-导出数据到Excel",notes = "参数说明：查询器只使到了query组传参，其他都可以不要，查询条件要带上表别名：主表 bc、副表 pm；传入pm.ID_导出该项目下的数据，否则将导出所有数据")
    public CommonResult exportDataToExcel (@ApiParam(name = "queryFilter",value = "通用查询器")@RequestBody QueryFilter<Contract> queryFilter, HttpServletResponse response) throws IOException {
        log.debug("导出数据到Excel接口请求参数：{}",queryFilter,response);
        baseService.exportDatatoExcel(queryFilter,response);
        return new CommonResult<>("导出成功");
    }

    /**
     * 删除数据
     * @param ids 主键集合
     * @return 删除结果
     */
    @PostMapping("/delete")
    @ApiOperation(value = "S-删除数据",httpMethod = "POST")
    public CommonResult delete(@ApiParam(name = "ids",value = "ID集合") @RequestParam("ids") String ids) {
        log.debug("合同管理-删除数据接口请求参数：{}", ids);
        Assert.notNull(ids,"主键集合不能为空");
        boolean state = baseService.removeContractByIds(Arrays.asList(ids.split(",")));
        String msg = state ? "删除成功" : "删除失败";
        log.debug("合同管理-删除数据接口响应结果：{}", msg);
        return new CommonResult<String>(state, msg);
    }

    /**
     * 通过主键查询单条数据
     * @param id 主键
     * @return 单条数据
     */
    @PostMapping("/detail/{id}")
    @ApiOperation(value = "S-根据ID查询详情",httpMethod = "POST")
    public CommonResult<ContractVo> selectOne(@ApiParam(name = "id",value = "主键ID") @PathVariable String id) {
        log.debug("合同管理-根据ID查询详情 接口请求参数：{}", id);
        Assert.notNull(id,"主键不能为空");
        ContractVo result = baseService.getVoById(id);
        boolean state = !BeanUtils.isEmpty(result);
        log.debug("合同管理-根据ID查询详情 接口响应结果：{}", result);
        return new CommonResult<ContractVo>(state, state?"操作成功":"该ID没有对应数据", result);
    }

    /**
     * 分页查询所有数据
     * @param queryFilter 通用查询器
     * @return 分页数据
     */
    @PostMapping("/page")
    @ApiOperation(value = "S-分页查询所有数据",notes = "参数说明：查询器只使到了query组传参，其他都可以不要，查询条件要带上表别名：主表 bc、副表 pm；传入pm.ID_切换项目")
    public CommonResult<PageList<ContractVo>> selectAll(@ApiParam(name = "queryFilter",value = "通用查询器")@RequestBody QueryFilter<Contract> queryFilter) {
        log.debug("合同管理-分页查询所有数据接口请求参数：{}", queryFilter);
        PageList<ContractVo> result = baseService.queryAllByPage(queryFilter);
        log.debug("合同管理-分页查询所有数据接口响应结果：{}", result);
        return new CommonResult<>(true,"操作成功",result);
    }


    @PostMapping("/downloadBatch")
    @ApiOperation(value = "S-批量下载合同",notes = "必须传入项目ID，合同id可选，按指定的合同ID批量下载附件，若合同id为空，则默认打包下载该项目下所有合同的附件")
    public CommonResult downloadBatch (HttpServletResponse response,
            @ApiParam(name = "projectId",value = "项目ID")@RequestParam String projectId,
            @ApiParam(name = "contractIds",value = "合同IDs")@RequestParam String contractIds){
        log.debug("批量下载合同接口请求参数：{}",projectId,contractIds);
        baseService.downloadBatch(projectId,Arrays.asList(contractIds.split(",")),response);
        return new CommonResult<>("导出成功");
    }

}

