package com.artfess.cqxy.documentLibrary.controller;

import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.controller.BaseController;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.util.BeanUtils;
import com.artfess.cqxy.documentLibrary.model.FileLibraryType;
import com.artfess.cqxy.documentLibrary.manager.FileLibraryTypeManager;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.util.Assert;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.io.Serializable;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;

/**
 * 文件库-分类表(FileLibraryType)表控制层
 *
 * @author 黎沐华
 * @since 2022-04-02 09:21:22
 */
@Slf4j
@RestController
@Api(tags = "文件库-分类接口")
@ApiGroup(group = {ApiGroupConsts.GROUP_BIZ})
@RequestMapping("/biz/fileLibrary/type/v1")
public class FileLibraryTypeController extends BaseController<FileLibraryTypeManager, FileLibraryType> {

    /**
     * 保存或修改数据
     *
     * @param fileLibraryType 实体对象
     * @return 新增结果
     */
    @PostMapping("/batchSave")
    @ApiOperation(value = "S-批量保存或修改数据", httpMethod = "POST")
    public CommonResult saveBatch(@ApiParam(name = "fileLibraryType", value = "文件库-分类信息表对象") @RequestBody FileLibraryType fileLibraryType) {
        Assert.notNull(fileLibraryType, "数据不能为空");
        String msg = "数据添加成功！";
//        boolean save = baseService.saveOrUpdateBatch(fileLibraryType);
        //判断编码是否重复
        String code = fileLibraryType.getCode();
        String id = fileLibraryType.getId();
        boolean isKeyExist = baseService.isKeyExist(id, code);
        if (isKeyExist) {
            msg = "输入的分类编码【" + code + "】已存在!";
            return new CommonResult<String>(false, msg);
        } else {
            if (StringUtils.isNotBlank(id)) {
                baseService.update(fileLibraryType);
                msg = "数据修改成功！";
            } else {
                baseService.save(fileLibraryType);
            }
        }
        return new CommonResult(true, msg);
    }

    /**
     * 删除数据
     *
     * @param ids 主键集合
     * @return 删除结果
     */
    @PostMapping("/delete")
    @ApiOperation(value = "S-删除数据", httpMethod = "POST")
    public CommonResult delete(@ApiParam(name = "ids", value = "ID集合") @RequestParam("ids") String ids) {
        log.debug("文件库-分类-删除数据接口请求参数：{}", ids);
        Assert.notNull(ids, "主键集合不能为空");
        boolean state = baseService.removeByIds(Arrays.asList(ids.split(",")));
        String msg = state ? "删除成功" : "删除失败";
        log.debug("文件库-分类-删除数据接口响应结果：{}", msg);
        return new CommonResult(state, msg);
    }

    /**
     * 通过主键查询单条数据
     *
     * @param id 主键
     * @return 单条数据
     */
    @PostMapping("/detail/{id}")
    @ApiOperation(value = "S-根据ID查询详情", httpMethod = "POST")
    public CommonResult<FileLibraryType> selectOne(@ApiParam(name = "id", value = "主键ID") @PathVariable String id) {
        log.debug("文件库-分类-根据ID查询详情 接口请求参数：{}", id);
        Assert.notNull(id, "主键不能为空");
        FileLibraryType result = baseService.getById(id);
        boolean state = !BeanUtils.isEmpty(result);
        log.debug("文件库-分类-根据ID查询详情 接口响应结果：{}", result);
        return new CommonResult<FileLibraryType>(state, state ? "操作成功" : "该ID没有对应数据", result);
    }


    @PostMapping("/getByParentId")
    @ApiOperation(value = "S-根据父节点编码查询其子节点", notes = "根据父节点编码查询子节点，URL传参pCode，根节点为root", httpMethod = "POST")
    public CommonResult<List<FileLibraryType>> getByParentId(@RequestParam(value = "pCode") String pCode) {
        log.debug("文件库-分类-获取树状数据接口请求参数：{}", pCode);
        List<FileLibraryType> result = baseService.getByParentCode(pCode);
        log.debug("文件库-分类-获取树状数据接口响应结果：{}", result);
        return new CommonResult<List<FileLibraryType>>(true, "操作成功", result);
    }

    /**
     * 分页查询所有数据
     *
     * @param queryFilter 通用查询器
     * @return 分页数据
     */
    @PostMapping("/queryChildNodeByPage")
    @ApiOperation(value = "S-分页查询所有数据", notes = "参数说明：在通用查询器中的query对象组传入查询参数，必须传入 PCODE_ 父节点名，所有参数都需要带上表别名——bflt，不分页则传入pageSize为-1", httpMethod = "POST")
    public CommonResult<PageList<FileLibraryType>> queryChildNodeByPage(@ApiParam(name = "queryFilter", value = "通用查询器") @RequestBody QueryFilter<FileLibraryType> queryFilter) {
        log.debug("文件库-分类-分页查询所有数据接口请求参数：{}", queryFilter);
        PageList<FileLibraryType> result = baseService.queryChildNodeByPage(queryFilter);
        log.debug("文件库-分类-分页查询所有数据接口响应结果：{}", result);
        return new CommonResult<PageList<FileLibraryType>>(true, "操作成功", result);
    }

}

