package com.artfess.workflow.runtime.controller;

import com.artfess.base.controller.BaseController;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.FieldRelation;
import com.artfess.base.query.PageBean;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.query.QueryOP;
import com.artfess.base.util.BeanUtils;
import com.artfess.base.util.StringUtil;
import com.artfess.bpm.persistence.manager.BpmDefinitionManager;
import com.artfess.bpm.persistence.model.DefaultBpmDefinition;
import com.artfess.workflow.runtime.manager.BpmPrintRecordManager;
import com.artfess.workflow.runtime.model.BpmPrintRecord;
import com.artfess.workflow.runtime.params.BpmPrintRecordObject;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

/**
 * <pre>
 * 描述：流程打印文件管理表 控制器类
 * 构建组：x7
 * 作者:zhangxw
 * 邮箱:zhangxw@jee-soft.cn
 * 日期:2020-05-23 19:58:35
 * 版权：广州宏天软件股份有限公司
 * </pre>
 */
@RestController
@RequestMapping(value = "/runtime/bpmPrintRecord/v1")
@Api(tags = "bpmPrintRecordController")
public class BpmPrintRecordController extends BaseController<BpmPrintRecordManager, BpmPrintRecord> {
    @Resource
    BpmPrintRecordManager bpmPrintRecordManager;
    @Resource
    BpmDefinitionManager bpmDefinitionManager;

    /**
     * 流程打印文件管理表列表(分页条件查询)数据
     *
     * @param request
     * @return
     * @throws Exception PageJson
     * @throws
     */
    @PostMapping("/listJson")
    @ApiOperation(value = "流程打印文件管理表数据列表", httpMethod = "POST", notes = "获取流程打印文件管理表列表")
    public PageList<BpmPrintRecord> list(@ApiParam(name = "queryFilter", value = "查询对象") @RequestBody QueryFilter queryFilter) throws Exception {
        queryFilter.addFilter("LAST_VERSION_", 1, QueryOP.EQUAL, FieldRelation.AND, "isMain");
        return bpmPrintRecordManager.query(queryFilter);
    }

    @RequestMapping(value = "/getByProInstId", method = RequestMethod.GET, produces = {
            "application/json; charset=utf-8"})
    @ApiOperation(value = "获取实例下的打印合同", httpMethod = "GET", notes = "获取实例下的打印合同")
    public List<BpmPrintRecord> getByProInstId(
            @ApiParam(name = "procInstId", value = "procInstId", required = true) @RequestParam String procInstId,
            @ApiParam(name = "nodeId", value = "nodeId", required = false) @RequestParam Optional<String> nodeId)
            throws Exception {
        QueryFilter queryFilter = QueryFilter.build().withPage(new PageBean(1, 1000));
        queryFilter.addFilter("LAST_VERSION_", 1, QueryOP.EQUAL, FieldRelation.AND, "isMain");
        queryFilter.addFilter("PROC_INST_ID_", procInstId, QueryOP.EQUAL, FieldRelation.AND, "isMain");
        if (StringUtil.isNotEmpty(nodeId.orElse(null))) {
            queryFilter.addFilter("NODE_ID_", nodeId.get(), QueryOP.EQUAL, FieldRelation.AND, "isMain");
        }
        PageList<BpmPrintRecord> pageList = bpmPrintRecordManager.query(queryFilter);
        if (BeanUtils.isNotEmpty(pageList)) {
            return pageList.getRows();
        } else {
            return new ArrayList<BpmPrintRecord>();
        }
    }

    /**
     * 流程打印文件管理表明细页面
     *
     * @param id
     * @return
     * @throws Exception ModelAndView
     */
    @GetMapping(value = "/getJson")
    @ApiOperation(value = "流程打印文件管理表数据详情", httpMethod = "GET", notes = "流程打印文件管理表数据详情")
    public BpmPrintRecord get(@ApiParam(name = "id", value = "业务对象主键", required = true) @RequestParam String id) throws Exception {
        return bpmPrintRecordManager.get(id);
    }

    /**
     * 新增流程打印文件管理表
     *
     * @param bpmPrintRecord
     * @return
     * @throws Exception
     * @throws
     */
    @PostMapping(value = "save")
    @ApiOperation(value = "新增,更新流程打印文件管理表数据", httpMethod = "POST", notes = "新增,更新流程打印文件管理表数据")
    public CommonResult<String> save(@ApiParam(name = "bpmPrintRecord", value = "流程打印文件管理表业务对象", required = true) @RequestBody BpmPrintRecord bpmPrintRecord) throws Exception {
        String msg = "添加流程打印文件管理表成功";
        if (StringUtil.isEmpty(bpmPrintRecord.getId())) {
            bpmPrintRecordManager.create(bpmPrintRecord);
        } else {
            bpmPrintRecordManager.update(bpmPrintRecord);
            msg = "更新流程打印文件管理表成功";
        }
        return new CommonResult<String>(msg);
    }

    /**
     * 批量删除流程打印文件管理表记录
     *
     * @param ids
     * @return
     * @throws Exception
     * @throws
     */
    @DeleteMapping(value = "/remove")
    @ApiOperation(value = "批量删除流程打印文件管理表记录", httpMethod = "DELETE", notes = "批量删除流程打印文件管理表记录")
    public CommonResult<String> removes(@ApiParam(name = "ids", value = "业务主键数组,多个业务主键之间用逗号分隔", required = true) @RequestParam String... ids) throws Exception {
        bpmPrintRecordManager.removeByIds(ids);
        return new CommonResult<String>(true, "批量删除成功");
    }

    @PostMapping(value = "printLog")
    @ApiOperation(value = "新增打印记录", httpMethod = "POST", notes = "新增打印记录")
    public CommonResult<String> printLog(@ApiParam(name = "printRecordObject", value = "添加打印记录参数对象", required = true) @RequestBody BpmPrintRecordObject printRecordObject) throws Exception {
        if (StringUtil.isEmpty(printRecordObject.getProcInstId())) {
            return new CommonResult<String>(false, "新增打印记录失败，实例id不能为空！");
        }
        if (StringUtil.isEmpty(printRecordObject.getProcDefId())) {
            return new CommonResult<String>(false, "新增打印记录失败，流程定义id不能为空！");
        }
        DefaultBpmDefinition def = bpmDefinitionManager.get(printRecordObject.getProcDefId());
        if (BeanUtils.isEmpty(def)) {
            return new CommonResult<String>(false, "新增打印记录失败，流程定义id不存在！");
        }
        printRecordObject.setProcDefKey(def.getDefKey());
        printRecordObject.setProcDefName(def.getName());
        return bpmPrintRecordManager.addPrintLog(printRecordObject);
    }
}
