package com.artfess.xqxt.meeting.controller;


import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.controller.BaseController;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.poi.util.ExcelUtil;
import com.artfess.xqxt.meeting.dto.MeetingDTO;
import com.artfess.xqxt.meeting.m900.bean.ConferenceStatus;
import com.artfess.xqxt.meeting.m900.bean.ParticipantStatusV2;
import com.artfess.xqxt.meeting.manager.BizTerminalManager;
import com.artfess.xqxt.meeting.manager.MeetingManager;
import com.artfess.xqxt.meeting.manager.MeetingUserManager;
import com.artfess.xqxt.meeting.manager.ZTEManager;
import com.artfess.xqxt.meeting.model.*;
import com.artfess.xqxt.meeting.vo.*;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.springframework.beans.BeanUtils;
import org.springframework.util.Assert;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

@RestController
@RequestMapping("/meeting/v1/")
@Api(tags = "会议模块")
@ApiGroup(group = {ApiGroupConsts.GROUP_BIZ})
public class MeetingController extends BaseController<MeetingManager, Meeting> {

    @Resource
    MeetingManager meetingManager;

    @Resource
    private ZTEManager zteManager;

    @Resource
    private MeetingUserManager meetingUserManager;

    @PostMapping(value = "/export")
    @ApiOperation("会议导出")
    public void exportMeeting(@ApiParam(name = "id", value = "会议ID") @RequestParam String id, HttpServletResponse response) throws Exception {
        Assert.notNull(id, "会议ID不能为空");
        meetingManager.exportMeeting(id,response);
    }

    @RequestMapping(value = "/getHistory", method = RequestMethod.POST, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "历史会议", httpMethod = "POST")
    public CommonResult<PageList<MeetingVO>> getAllHistory(@RequestParam Integer type,@ApiParam(name = "queryFilter", value = "分页查询信息") @RequestBody QueryFilter<Meeting> queryFilter) {
        PageList<MeetingVO> result =  meetingManager.getHistory(type,queryFilter);
        return new CommonResult<>(true, "数据查询成功", result);
    }

    @RequestMapping(value = "/listCount", method = RequestMethod.GET)
    @ApiOperation(value = "会议数量", httpMethod = "GET")
    public CommonResult<MeetingCountVO> listCount() {
        return new CommonResult<>(true, "数据查询成功", meetingManager.getMeetingCount());
    }

    @RequestMapping(value = "/getJson", method = RequestMethod.POST, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "会议查询", httpMethod = "POST")
    public CommonResult<PageList<MeetingVO>> getJson(@ApiParam(name = "queryFilter", value = "分页查询信息") @RequestBody QueryFilter<Meeting> queryFilter) {
        PageList<MeetingVO> result =  meetingManager.findByUserId(queryFilter);
        return new CommonResult<>(true, "数据查询成功", result);
    }

    @RequestMapping(value = "/getSiteListByMeetingId", method = RequestMethod.GET, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "根据会议ID查询会场列表及状态", httpMethod = "GET")
    public CommonResult<List<BizSiteParamExVO>> getSiteListByMeetingId(@ApiParam(name = "meetingId", value = "会议ID") @RequestParam String meetingId,
                                                                     @ApiParam(name = "terminalIdentifier", value = "终端ID，多个使用逗号分割ID") @RequestParam String terminalIdentifier) throws Exception {
        String[] terminalIdentifiers = null;
        if(StringUtils.isNotBlank(terminalIdentifier)){
            terminalIdentifiers = terminalIdentifier.split(",");
        }
        //测试环境下注释中兴调用，测试环境调用会只查询本地并模拟终端状态，不查询终端实际状态
//        List<BizSiteParamExVO> result = meetingManager.getSiteListByMeetingId(meetingId);
        //生产环境调用
        List<BizSiteParamExVO> result =  zteManager.getParticipantStatusV2(meetingId, terminalIdentifiers);
        return new CommonResult<>(true, "数据查询成功", result);
    }

    @GetMapping(value = "get/{id}")
    @ApiOperation(value = "根据ID查询会议信息", httpMethod = "GET", notes = "根据ID查询会议信息")
    public CommonResult<MeetingVO> get(@ApiParam(name = "id", value = "业务对象主键", required = true) @PathVariable String id)  throws Exception{
        return new CommonResult<MeetingVO>(true,"成功",meetingManager.findById(id));
    }

    @GetMapping(value = "getSha1")
    @ApiOperation(value = "获取微信签名", httpMethod = "GET", notes = "获取微信签名")
    public CommonResult<Sha1VO> getSha1(@RequestParam String url){
        return new CommonResult<Sha1VO>(true,"成功",meetingManager.getSha1(url));
    }








    @RequestMapping(value = "/saveMeeting", method = RequestMethod.POST, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "保存会议信息加分组", httpMethod = "POST")
    public CommonResult<String> saveMeetingVo(@ApiParam(name = "MeetingDTO", value = "会议实体") @RequestBody MeetingDTO dto){
        String result = meetingManager.insertMeetingVo(dto);
        if ("1".equals(result)) {
            return new CommonResult<>(true,"会议创建成功！");
        }
        return new CommonResult<>(false,"会场创建失败，"+ ZTEErrorCode.getDescByCode(result));
    }



    @RequestMapping(value = "/updateMeeting", method = RequestMethod.POST, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "修改会议信息加分组", httpMethod = "POST")
    public CommonResult<String> updateMeetingVo(@ApiParam(name = "MeetingDTO", value = "会议实体") @RequestBody MeetingDTO dto)  {
        int result = meetingManager.updateMeetingVo(dto);
        if (result<=0) return new CommonResult<>(false,"修改失败！");
        return new CommonResult<String>(true,"修改成功！");
    }


    @RequestMapping(value = "/delayMeeting", method = RequestMethod.POST, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "延长会议时间", httpMethod = "POST")
    public CommonResult<String> delayMeeting(@ApiParam(name = "meetingId", value = "会议ID") @RequestParam String meetingId,
                                             @ApiParam(name = "meetingType", value = "会议类型") @RequestParam Integer meetingType,
                                             @ApiParam(name = "minutes", value = "延长时间，分钟") @RequestParam Integer minutes) throws Exception {
        String result = meetingManager.delayMeeting(meetingId,meetingType,minutes);
        if ("1".equals(result)) {
            return new CommonResult<>(false,"延长会议时间成功！");
        }
        return new CommonResult<String>(true,"延长会议时间失败！");
    }








    @DeleteMapping(value = "/removes")
    @ApiOperation(value = "批量删除会议", httpMethod = "DELETE", notes = "批量删除会议")
    public CommonResult<String> removes(
            @ApiParam(name = "ids", value = "ID集合以，隔开", required = true)
            @RequestParam(value = "ids", required = true) String ids) throws Exception {
        if (StringUtils.isNotBlank(ids)) {
            List<String> idsList = Arrays.asList(ids.split(","));
            UpdateWrapper<Meeting> updateWrapper = new UpdateWrapper<>();
            updateWrapper.in("ID_", idsList);
            updateWrapper.set("IS_DELE_", "1");
            meetingManager.update(null, updateWrapper);
        }
        return new CommonResult<>(true, "批量删除成功");
    }


    @RequestMapping(value = "/updateStatus", method = RequestMethod.GET)
    @ApiOperation(value = "修改会议状态", httpMethod = "GET")
    public CommonResult<String> updateStatus( @ApiParam("会议ID")@RequestParam String id,
                                              @ApiParam("会议状态，1：未开始，2：进行中，3：已结束，4：取消")@RequestParam Integer status)  {
        // 会议状态，1：未开始，2：进行中，3：已结束，4：取消会议
        String result = meetingManager.updateStatus(id,status);
        if ("1".equals(result)) {
            return new CommonResult<String>(true,"修改成功！");
        }
        return new CommonResult<>(false,"修改失败，"+ ZTEErrorCode.getDescByCode(result));
    }

    @RequestMapping(value = "/endMeeting", method = RequestMethod.GET)
    @ApiOperation(value = "结束会议", httpMethod = "GET")
    public CommonResult<String> endMeeting(@RequestParam String id)  {
        String result = meetingManager.endMeeting(id);
        if ("1".equals(result)) {
            return new CommonResult<String>(true,"会议结束成功！");
        }
        return new CommonResult<>(false,"会议结束失败，"+ ZTEErrorCode.getDescByCode(result));
    }

    @RequestMapping(value = "/quiet", method = RequestMethod.GET)
    @ApiOperation(value = "会场麦克风静音（一键静音时不传终端ID）", httpMethod = "GET")
    public CommonResult<String> quietMeeting(@ApiParam("会议ID") @RequestParam String id,
                                             @ApiParam("终端ID，如果不传则关闭会议中所有会场麦克风") @RequestParam String placeId)  {
        String result = zteManager.muteParticipant(id, placeId, true);
        if ("1".equals(result)) {
            return new CommonResult<String>(true,"静音成功！");
        }
        return new CommonResult<>(false,"静音失败，"+ ZTEErrorCode.getDescByCode(result));
    }

    @RequestMapping(value = "/cancelQuiet", method = RequestMethod.GET)
    @ApiOperation(value = "会场麦克风取消静音（打开麦克风输入）", httpMethod = "GET")
    public CommonResult<String> cancelQuiet(@ApiParam("会议ID") @RequestParam String id,
                                            @ApiParam("终端ID，如果不传则打开会议中所有会场麦克风") @RequestParam String placeId)  {

        String result = zteManager.cancelMuteParticipant(id, placeId, true);
        if ("1".equals(result)) {
            return new CommonResult<String>(true,"取消静音成功！");
        }
        return new CommonResult<>(false,"取消静音失败，"+ ZTEErrorCode.getDescByCode(result));
    }

    @RequestMapping(value = "/quietParticipant", method = RequestMethod.GET)
    @ApiOperation(value = "会场扬声器静音（关闭扬声器输入）", httpMethod = "GET")
    public CommonResult<String> quietParticipant(@ApiParam("会议ID") @RequestParam String id,
                                                 @ApiParam("终端ID，如果不传则关闭会议中所有会场扬声器") @RequestParam String placeId)  {
        String result = meetingManager.quiet(id, placeId);
        if ("1".equals(result)) {
            return new CommonResult<String>(true,"静音成功！");
        }
        return new CommonResult<>(false,"静音失败，"+ ZTEErrorCode.getDescByCode(result));
    }

    @RequestMapping(value = "/cancelQuietParticipant", method = RequestMethod.GET)
    @ApiOperation(value = "会场扬声器取消静音（打开扬声器输入）", httpMethod = "GET")
    public CommonResult<String> cancelQuietParticipant(@ApiParam("会议ID") @RequestParam String id,
                                                       @ApiParam("终端ID，如果不传则打开会议中所有会场扬声器") @RequestParam String placeId)  {
      String result = meetingManager.cancelQuiet(id,placeId);
        if ("1".equals(result)) {
            return new CommonResult<String>(true,"取消静音成功！");
        }
        return new CommonResult<>(false,"取消静音失败，"+ ZTEErrorCode.getDescByCode(result));
    }

    @RequestMapping(value = "/connect", method = RequestMethod.POST)
    @ApiOperation(value = "会场呼叫（一键呼叫时不传终端ID）", httpMethod = "POST")
    public CommonResult<String> conect(@ApiParam(name = "meetParamVO", value = "会议参数Vo")  @RequestBody MeetParamVO meetParamVO)  {
        String id = meetParamVO.getId();
        String placeIds = meetParamVO.getPlaceIds();
        String result = meetingManager.connect(id,placeIds);
        if ("1".equals(result)) {
            return new CommonResult<String>(true,"会场呼叫成功！");
        }
        return new CommonResult<>(false,"会场呼叫失败，"+ ZTEErrorCode.getDescByCode(result));
    }

    @RequestMapping(value = "/disconnect", method = RequestMethod.GET)
    @ApiOperation(value = "会场断开", httpMethod = "GET")
    public CommonResult<String> disconnect(@ApiParam("会议ID") @RequestParam String id,
                                           @ApiParam("终端ID，多个使用逗号分隔") @RequestParam String placeIds)  {
        String result = meetingManager.disconnect(id,placeIds);
        if ("1".equals(result)) {
            return new CommonResult<String>(true,"会场断开成功！");
        }
        return new CommonResult<>(false,"会场断开失败，"+ ZTEErrorCode.getDescByCode(result));
    }


    @RequestMapping(value = "/invitePar", method = RequestMethod.GET)
    @ApiOperation(value = "会场添加", httpMethod = "GET")
    public CommonResult<String> invitePar(@ApiParam("会议ID") @RequestParam String id,@ApiParam("终端ID") @RequestParam String placeId)  {
        String result = meetingManager.inviteParticipant(id, placeId);
        if ("1".equals(result)) {
            return new CommonResult<String>(true,"会场添加成功！");
        }
        return new CommonResult<>(false,"会场添加失败，"+ ZTEErrorCode.getDescByCode(result));
    }

    @RequestMapping(value = "/parStatus", method = RequestMethod.GET)
    @ApiOperation(value = "会场状态", httpMethod = "GET")
    public CommonResult<ParticipantStatusV2[]> parStatus(@RequestParam String id) throws Exception {
        return new CommonResult<ParticipantStatusV2[]>
                (true, "数据查询成功", meetingManager.participantStatusV2(id));
    }

    @RequestMapping(value = "/setChairman", method = RequestMethod.GET)
    @ApiOperation(value = "设置主席", httpMethod = "GET")
    public CommonResult<String> setChairman(@ApiParam("会议ID") @RequestParam String id, @ApiParam("终端ID") @RequestParam String placeId)  {
        String result = zteManager.setChairman(id, placeId);
        if("-1".equalsIgnoreCase(placeId)){
            // 如果是取消主席，则关闭麦克风
            zteManager.quiet(id, placeId, false);
        }else {
            // 如果是设置主席，则开启麦克风
            zteManager.cancelQuiet(id, placeId,false);
        }
        if ("1".equals(result)) {
            return new CommonResult<>(true,"设置成功！");
        }
        return new CommonResult<>(false,"设置失败，"+ ZTEErrorCode.getDescByCode(result));
    }

    @RequestMapping(value = "/call", method = RequestMethod.GET)
    @ApiOperation(value = "点名", httpMethod = "GET")
    public CommonResult<String> call(@ApiParam("会议ID")@RequestParam String id, @ApiParam("被点名终端ID")@RequestParam String placeId,
                                    @ApiParam("主席会场终端ID，如果有主席必须传入，否则会全员静音")@RequestParam String chairmanId)  {
        Assert.notNull(chairmanId, "主席ID不能为空，请先设置主席！");
        // 1.关闭轮训
        String s = zteManager.browseParticipant(id, null, 0);
        // 2.关闭所有麦克风
//        meetingManager.quiet(id,null);
        String quiet = zteManager.muteParticipant(id, null, false);
        // 3.开启被点名会场
        String s1 = zteManager.cancelMuteParticipant(id, placeId, false);
        //  开启主席会场麦克风
        if(StringUtils.isNotBlank(chairmanId)){
            String s2 = zteManager.cancelMuteParticipant(id, chairmanId, false);
        }
        // 4.广播被点名会场
        String result = zteManager.boardcastParticipant(id, placeId);
        if ("1".equals(result)) {
            // 修改当前会议广播
            Meeting meeting = meetingManager.get(id);
            if(null != meeting){
                meeting.setBoardcaster(placeId);
                meetingManager.update(meeting);
            }
            return new CommonResult<>(true,"点名成功");
        }
        return new CommonResult<>(false,"点名失败，"+ ZTEErrorCode.getDescByCode(result));
    }

    /**
     * 取消点名，目前设置为，取消点名后关闭广播会场，然后广播主席会场，所以主席ID必转
     * @param id
     * @param placeId
     * @param chairmanId
     * @return
     */
    @RequestMapping(value = "/cancelCall", method = RequestMethod.GET)
    @ApiOperation(value = "取消点名", httpMethod = "GET")
    public CommonResult<String> cancelCall(@ApiParam("会议ID")@RequestParam String id, @ApiParam("被取消点名终端ID")@RequestParam String placeId,
                                     @ApiParam("主席会场终端ID，如果有主席必须传入，否则会全员静音")@RequestParam String chairmanId)  {
        Assert.notNull(chairmanId, "主席ID不能为空！");
        // 1.关闭轮训
        String s = zteManager.browseParticipant(id, null, 0);
        // 2.关闭所有麦克风
        String quiet = zteManager.muteParticipant(id, null, false);
        // 3.只开启主席会场麦克风
        String s2 = zteManager.cancelMuteParticipant(id, chairmanId, false);
        // 4.广播主席会场
        String result = zteManager.boardcastParticipant(id, chairmanId);
        // 如果取消主席会场的点名，则进入轮训
        boolean allCancel = false;
        if(placeId.equalsIgnoreCase(chairmanId)){
            zteManager.browseParticipant(id, null, 1);
            allCancel = true;
        }
        if ("1".equals(result)) {
            // 修改当前会议广播
            Meeting meeting = meetingManager.get(id);
            if(null != meeting){
                if(allCancel){
                    meeting.setBoardcaster("-1");
                }else {
                    meeting.setBoardcaster(placeId);
                }
                meetingManager.update(meeting);
            }
            return new CommonResult<>(true,"取消点名成功");
        }
        return new CommonResult<>(false,"取消点名失败，"+ ZTEErrorCode.getDescByCode(result));
    }


    @RequestMapping(value = "/browseParticipant", method = RequestMethod.POST)
    @ApiOperation(value = "会场轮询", httpMethod = "POST")
    public CommonResult<String> browseParticipant(@ApiParam(name = "meetBrowseParamVO", value = "会议轮训参数Vo")
                                                      @RequestBody MeetBrowseParamVO meetBrowseParamVO)  {
        String id = meetBrowseParamVO.getId();
        String placeId = meetBrowseParamVO.getPlaceId();
        Integer cmdType = meetBrowseParamVO.getCmdType();
        String result = zteManager.browseParticipant(id, placeId, cmdType);
        result = "1";
        if ("1".equals(result)) {
            // 修改当前会议轮训状态
            Meeting meeting = meetingManager.get(id);
           if(null != meeting){
               if(null != cmdType && cmdType == 0){

                   meeting.setRotationStatus(1);
                   meetingManager.update(new UpdateWrapper<Meeting>().set("ROTATION_STATUS_",1).eq("ID_", id));
               }else {
                   meeting.setRotationStatus(0);
                   meetingManager.update(new UpdateWrapper<Meeting>().set("ROTATION_STATUS_",0).eq("ID_", id));
               }
               meetingManager.update(meeting);
           }
            return new CommonResult<>(true,"会场轮询设置成功！");
        }
        return new CommonResult<>(false,"会场轮询设置失败，"+ ZTEErrorCode.getDescByCode(result));
    }

    @RequestMapping(value = "/updateMeetingUser", method = RequestMethod.POST)
    @ApiOperation(value = "会议用户信息修改", httpMethod = "POST")
    public CommonResult<String> updateMeetingUser(@RequestBody MeetingUser meetingUser)  {
        if (meetingManager.updateMeetingUser(meetingUser)>0){
            return new CommonResult<>(true,"修改成功！");
        }
        return new CommonResult<>(false,"修改失败！");
    }

    @RequestMapping(value = "/signIn", method = RequestMethod.GET)
    @ApiOperation(value = "签到", httpMethod = "GET")
    public CommonResult<String> signIn(@RequestParam String meetingId,@RequestParam String userId)  {
        Assert.notNull(meetingId, "会议ID不能为空");
        Assert.notNull(userId, "用户ID不能为空");
        meetingManager.signIn(meetingId,userId);
        return new CommonResult<>(true,"签到成功！");
    }

    @RequestMapping(value = "/disSignIn", method = RequestMethod.GET)
    @ApiOperation(value = "取消签到", httpMethod = "GET")
    public CommonResult<String> disSignIn(@RequestParam String meetingId,@RequestParam String userId)  {
        Assert.notNull(meetingId, "会议ID不能为空");
        Assert.notNull(userId, "用户ID不能为空");
        meetingManager.disSignIn(meetingId,userId);
        return new CommonResult<>(true,"取消签到成功！");
    }

    @RequestMapping(value = "/signOut", method = RequestMethod.GET)
    @ApiOperation(value = "签退", httpMethod = "GET")
    public CommonResult<String> signOut(@RequestParam String meetingId,@RequestParam String userId)  {
        Assert.notNull(meetingId, "会议ID不能为空");
        Assert.notNull(userId, "用户ID不能为空");
        meetingManager.signOut(meetingId,userId);
        return new CommonResult<>(true,"签退成功！");
    }

    @RequestMapping(value = "/disSignOut", method = RequestMethod.GET)
    @ApiOperation(value = "取消签退", httpMethod = "GET")
    public CommonResult<String> disSignOut(@RequestParam String meetingId,@RequestParam String userId)  {
        Assert.notNull(meetingId, "会议ID不能为空");
        Assert.notNull(userId, "用户ID不能为空");
        meetingManager.disSignOut(meetingId,userId);
        return new CommonResult<>(true,"取消签退成功！");
    }

    @RequestMapping(value = "/signInUserList", method = RequestMethod.GET)
    @ApiOperation(value = "签到人员列表", httpMethod = "GET")
    public CommonResult<List<MeetingUser>> signInUserList(@RequestParam String meetingId,@RequestParam Integer type)  {
        Assert.notNull(meetingId, "会议ID不能为空");
        return new CommonResult<>(true,"成功！",meetingManager.signInUserList(meetingId,type));
    }

    @RequestMapping(value = "/signOutUserList", method = RequestMethod.GET)
    @ApiOperation(value = "签退人员列表", httpMethod = "GET")
    public CommonResult<List<MeetingUser>> signOutUserList(@RequestParam String meetingId,@RequestParam Integer type)  {
        Assert.notNull(meetingId, "会议ID不能为空");
        return new CommonResult<>(true,"成功！",meetingManager.signOutUserList(meetingId,type));
    }


    @RequestMapping(value = "/signInAndOutCount", method = RequestMethod.GET)
    @ApiOperation(value = "签到/签退人员数量统计", httpMethod = "GET")
    public CommonResult<SignInAndOutCountVo> signInAndOutCount(@RequestParam String meetingId)  {
        Assert.notNull(meetingId, "会议ID不能为空");
        return new CommonResult<SignInAndOutCountVo>(true,"数据查询成功！", meetingManager.signInAndOutCount(meetingId));
    }


    @RequestMapping(value = "/getMeetingUserByMeetingIdAndUserId", method = RequestMethod.GET)
    @ApiOperation(value = "根据会议ID、用户id查询用户信息", httpMethod = "GET")
    public CommonResult<MeetingUser> getMeetingUserByMeetingIdAndUserId(@RequestParam String meetingId, @RequestParam String userId)  {
        Assert.notNull(meetingId, "会议ID不能为空");
//        Assert.notNull(userId, "用户ID不能为空");
        return new CommonResult<MeetingUser>(true,"数据查询成功！", meetingManager.getMeetingUserByMeetingIdAndUserId(meetingId, userId));
    }


    @RequestMapping(value = "/deleteParticipant", method = RequestMethod.GET)
    @ApiOperation(value = "删除会场", httpMethod = "GET")
    public CommonResult<String> deleteParticipant(@ApiParam("会议ID") @RequestParam String id,
                                                  @ApiParam("终端ID,多个使用逗号分隔") @RequestParam String placeId)  {
        Assert.notNull(placeId, "终端ID不能为空！");
        String result = zteManager.deleteParticipant(id, placeId);
        if ("1".equals(result)) {
            return new CommonResult<>(true,"会场删除成功！");
        }
        return new CommonResult<>(false,"会场删除失败，"+ ZTEErrorCode.getDescByCode(result));
    }


    @RequestMapping(value = "/feedBack", method = RequestMethod.POST)
    @ApiOperation(value = "反馈-入会反馈", httpMethod = "POST")
    public CommonResult<String> feedBack(@RequestBody MeetFeedBackVO meetFeedBackVO)  {
        meetingUserManager.feedBack(meetFeedBackVO);
        return new CommonResult<>(true,"入会反馈成功！");
    }

    @RequestMapping(value = "/synConferenceInfo", method = RequestMethod.POST)
    @ApiOperation(value = "同步中兴正在进行的会议", httpMethod = "POST")
    public CommonResult<String> synConferenceInfo()  {
        meetingManager.synConferenceInfo();
        return new CommonResult<>(true,"同步中兴正在进行的会议成功！", "");
    }

    @RequestMapping(value = "/synConferenceReserved", method = RequestMethod.POST)
    @ApiOperation(value = "同步中兴将来预约的会议", httpMethod = "POST")
    public CommonResult<String> synConferenceReserved()  {
        meetingManager.synConferenceReserved();
        return new CommonResult<>(true,"同步中兴将来预约的会议成功！", "");
    }


    @RequestMapping(value = "/getHistoryConferenceList", method = RequestMethod.POST)
    @ApiOperation(value = "从中兴获取历史会议", httpMethod = "POST")
    public CommonResult<List<ConferenceStatus>> getHistoryConferenceList(@RequestBody HistoryConferenceVO historyConferenceVO)  {
        List<ConferenceStatus> result = zteManager.getHistoryConferenceList(historyConferenceVO);
        return new CommonResult<>(true,"从中兴获取历史会议查询成功！", result);
    }

}
