package com.artfess.xqxt.meeting.manager.impl;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.artfess.base.constants.WebsocketConst;
import com.artfess.base.context.BaseContext;
import com.artfess.base.manager.impl.BaseManagerImpl;
import com.artfess.base.query.*;
import com.artfess.base.util.BeanUtils;
import com.artfess.base.util.JsonUtil;
import com.artfess.base.util.StringUtil;
import com.artfess.base.webSocket.PushService;
import com.artfess.poi.util.ExcelUtil;
import com.artfess.uc.manager.UserManager;
import com.artfess.uc.model.User;
import com.artfess.uc.util.ContextUtil;
import com.artfess.xqxt.meeting.dao.BizMeetingTopicUserDao;
import com.artfess.xqxt.meeting.dao.BizMeetingVoteDao;
import com.artfess.xqxt.meeting.manager.*;
import com.artfess.xqxt.meeting.model.*;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.gexin.fastjson.JSON;
import com.google.api.client.util.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.poi.xssf.usermodel.*;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.interceptor.TransactionAspectSupport;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 会议议题投票表 服务实现类
 *
 * @author 管理员
 * @company 阿特菲斯信息技术有限公司
 * @since 2022-11-03
 */
@Slf4j
@Service
public class BizMeetingVoteManagerImpl extends BaseManagerImpl<BizMeetingVoteDao, BizMeetingVote> implements BizMeetingVoteManager {

    @Autowired
    private BizMeetingTopicManager topicManager;

    @Autowired
    private BizMeetingQuestionManager questionManager;

    @Autowired
    private BizMeetingOptionManager optionManager;

    @Autowired
    private PushService pushService;

    @Autowired
    private MeetingManager meetingManager;

    @Autowired
    private MeetingUserManager meetingUserManager;

    @Resource
    private BizMeetingTopicUserManager bizMeetingTopicUserManager;

    @Resource
    private BizMeetingTopicUserDao bizMeetingTopicUserDao;

    @Resource
    private BaseContext baseContext;

    @Resource
    private UserManager userManager;
    @Resource
    private BizMeetingVoteManager bizMeetingVoteManager;

    @Override
    public BizMeetingTopic detail(String id) {
        BizMeetingTopic bizMeetingTopic  = topicManager.getById(id);
        if (bizMeetingTopic!=null){
            QueryWrapper<BizMeetingTopicUser> queryWrapper = new QueryWrapper<>();
            queryWrapper.eq("TOPIC_ID_",id);
            List<BizMeetingTopicUser> topicUsers = bizMeetingTopicUserManager.list(queryWrapper);
            //投票人员总数
            bizMeetingTopic.setVotedTotalNum(topicUsers.size());
            List<String> userids =
                    topicUsers.stream().map(BizMeetingTopicUser::getUserId).collect(Collectors.toList());
            QueryWrapper<User> userWrapper = new QueryWrapper<>();
            userWrapper.in("ID_",userids);
            bizMeetingTopic.setUsers(userManager.list(userWrapper));
        }
        for (BizMeetingQuestion bizMeetingQuestion : bizMeetingTopic.getQuestionList()) {
            String questionId = bizMeetingQuestion.getId();
            //查询这个问题已有多少人投票
            QueryWrapper<BizMeetingVote> voteQueryWrapper = new QueryWrapper<>();
            voteQueryWrapper.eq("QUESTION_ID_",questionId);

            int count = bizMeetingVoteManager.count(voteQueryWrapper);
            bizMeetingQuestion.setVotedNum(count);
            bizMeetingQuestion.setVotedNumTotal( count+"/"+bizMeetingTopic.getVotedTotalNum());
        }




        return bizMeetingTopic;
    }

    @Override
    public Integer checkTheme(String meetingId, String theme) {
        QueryWrapper<BizMeetingTopic> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("MEETING_ID_",meetingId);
        queryWrapper.eq("TOPIC_CONTENT_",theme);
        return topicManager.list(queryWrapper).size();
    }

    @Override
    public PageList<BizMeetingTopic> selectByUserId(QueryFilter<BizMeetingTopic> queryFilter) {
        QueryWrapper<BizMeetingTopicUser> queryWrapper = new QueryWrapper<>();
        List<QueryField> listQueryField = queryFilter.getQuerys();
        List<BizMeetingTopicUser> topicUsers = new ArrayList<>();
        for (QueryField queryField : listQueryField) {
            if ("meeting_id_".equals(queryField.getProperty())) {
                List<String> meetings = Arrays.asList(queryField.getValue().toString().split(","));
                queryWrapper.in("MEETING_ID_",meetings);
                queryWrapper.eq("USER_ID_",baseContext.getCurrentUserId());
                topicUsers = bizMeetingTopicUserManager.list(queryWrapper);
                break;
            }
        }
        if (topicUsers.size()>0){
            queryFilter.addFilter
                    ("ID_",topicUsers.stream().map(BizMeetingTopicUser::getTopicId).collect(Collectors.toList()), QueryOP.IN);
        }
        queryFilter.addFilter("CREATE_BY_",baseContext.getCurrentUserId(),QueryOP.EQUAL, FieldRelation.OR);
        if (topicManager.query(queryFilter).getRows().size()>0){
            return topicManager.query(queryFilter);
        }else {
            PageList<BizMeetingTopic> nullList = new PageList();
            nullList.setRows(new ArrayList());
            return nullList;
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean createVote(BizMeetingTopic topic) {
        boolean result = false;
        try {
            Assert.notNull(topic, "参数不能为空！");
            Assert.notNull(topic.getMeetingId(), "会议ID不能为空!");
            Assert.notEmpty(topic.getQuestionList(), "问题列表不能为空!");
            topic.getQuestionList().forEach(ql -> {
                Assert.notEmpty(ql.getOptionList(), "问题【" + ql.getQuestionContent() + "】的选项列表不能为空！");
            });

            // 先删再加
            List<String> deleteQuestion = Lists.newArrayList();
            List<String> deleteOption = Lists.newArrayList();
            questionManager.getBaseMapper().selectList(new QueryWrapper<BizMeetingQuestion>().eq("topic_id_", topic.getId())).forEach(ql->{
                deleteQuestion.add(ql.getId());
            });
            if (BeanUtils.isNotEmpty(deleteQuestion)){
                questionManager.removeByIds(deleteQuestion);
                optionManager.getBaseMapper().selectList(new QueryWrapper<BizMeetingOption>().in("QUESTION_ID_", deleteQuestion)).forEach(ol->{
                    deleteOption.add(ol.getId());
                });
            }
            if (BeanUtils.isNotEmpty(deleteOption)){
                optionManager.removeByIds(deleteOption);
            }

            if (StringUtil.isEmpty(topic.getId()) && null == topic.getStatus()) {
                topic.setStatus(1); // 创建时默认已保存状态
            }else {
                bizMeetingTopicUserDao.deleteById(topic.getMeetingId(),topic.getId());
            }
            result = topicManager.saveOrUpdate(topic);
//            roleBackCheck(result);
            int a = 0;
            // 回填ID
            topic.getQuestionList().forEach(ql -> {
                ql.setTopicId(topic.getId());
                ql.setAnonymous(Integer.valueOf(topic.getAnonymous()));
                ql.setMeetingId(topic.getMeetingId());
            });

            result = questionManager.saveOrUpdateBatch(topic.getQuestionList());
//            roleBackCheck(result);
            List<BizMeetingOption> options = Lists.newArrayList();
            topic.getQuestionList().forEach(ql -> {
                ql.getOptionList().forEach(ol -> {
                    if (!ql.getQuestionType().equals(ol.getOptionType())) {
                        throw new RuntimeException("问题【" + ql.getQuestionContent() + "】和选项【" + ol.getOptionValue() + "】的类型不相符!");
                    }
                    ol.setQuestionId(ql.getId());
                    options.add(ol);
                });
            });

            result = optionManager.saveOrUpdateBatch(options);
//            roleBackCheck(result);
            //添加投票人

            List<String> userIds = Arrays.asList(topic.getUserIds().split(","));
            for (String userId:userIds){
                BizMeetingTopicUser bizMeetingTopicUser = new BizMeetingTopicUser();
                bizMeetingTopicUser.setMeetingId(topic.getMeetingId());
                bizMeetingTopicUser.setTopicId(topic.getId());
                bizMeetingTopicUser.setUserId(userId);
                bizMeetingTopicUser.setCreateBy(baseContext.getCurrentUserId());
                bizMeetingTopicUser.setCreateName(baseContext.getCurrentUserName());
                bizMeetingTopicUserManager.create(bizMeetingTopicUser);
            }
            pushTopic(topic);
        } catch (Exception e) {
            TransactionAspectSupport.currentTransactionStatus().setRollbackOnly();
            throw new RuntimeException(e.getMessage());// 捕获异常后回滚事务，再将其抛出报错给前端
        }
        return result;
    }

    @Override
    @Transactional
    public boolean vote(BizMeetingVote vote) {
        Assert.notNull(vote, "参数不能为空!");
        Assert.notNull(vote.getQuestionId(), "问题ID不能为空!");
        Assert.notNull(vote.getOptionId(), "选项ID不能为空!");
        User currentUser = ContextUtil.getCurrentUser();
        // 用户关联检查，默认为当前登录用户，都不存在则抛出异常
        if (BeanUtils.isEmpty(vote.getUserId()) && BeanUtils.isEmpty(vote.getUserName()) && BeanUtils.isEmpty(vote.getUserAccount()) && BeanUtils.isEmpty(currentUser)) {
            throw new RuntimeException("用户关联异常，没有可关联的用户。");
        }
        if (BeanUtils.isEmpty(vote.getUserId()) || BeanUtils.isEmpty(vote.getUserName()) || BeanUtils.isEmpty(vote.getUserAccount())) {
            log.error("投票用户关联，传入参数userId、userAccount、userName不完全，默认使用当前登录用户......");
            vote.setUserId(currentUser.getUserId());
            vote.setUserAccount(currentUser.getAccount());
            vote.setUserName(currentUser.getFullname());
        }

        BizMeetingQuestion question = questionManager.get(vote.getQuestionId());
        // 检查用户是否已经投过票
        int repeatCheck = baseMapper.selectCount(new QueryWrapper<BizMeetingVote>()
                .eq("QUESTION_ID_", vote.getQuestionId()).eq("USER_ID_", vote.getUserId()));
        if (repeatCheck >= 1) {
            throw new RuntimeException("用户【" + vote.getUserName() + "】已经为【" + question.getQuestionContent() + "】投过票了!");
        }

        // 检查问题类型是否和传入的选项长度匹配
        Assert.notNull(question, "questionId【" + vote.getQuestionId() + "】没有数据!");
        if (("1".equals(question.getQuestionType()) && vote.getOptionId().split(",").length != 1)) {
            throw new RuntimeException("传入的选项长度与问题类型不匹配!");
        }
        boolean result = save(vote);
        BizMeetingTopic pushData = topicManager.getById(question.getTopicId());
        pushTopicId(pushData);
        return result;
    }

    @Override
    public boolean changeVoteStatus(BizMeetingTopic topic) {
        Assert.notNull(topic, "参数不能为空!");
        Assert.notNull(topic.getId(), "主键 ID 不能为空!");
        Assert.notNull(topic.getStatus(), "状态 Status 不能为空!");
        UpdateWrapper<BizMeetingTopic> updateWrapper = new UpdateWrapper<>();
        updateWrapper.set("STATUS_", topic.getStatus()).eq("ID_", topic.getId());
        boolean result = topicManager.update(updateWrapper);
        pushTopic(topicManager.get(topic.getId()));
        return result;
    }

    @Override
    public Map<String, Boolean> checkUserVote(String topicId) {
        Assert.notNull(topicId, "参数不能为空!");
        BizMeetingTopic topic = topicManager.getById(topicId);
        Assert.notNull(topic, "没有找到topicId【" + topicId + "】对应的数据!");
        User currentUser = ContextUtil.getCurrentUser();
        List<BizMeetingVote> allVote = list();
        Map<String, Boolean> result = new HashMap<>(5);
        topic.getQuestionList().forEach(ql -> {
            long count = allVote.stream().filter(v -> v.getUserId().equals(currentUser.getId()) && v.getQuestionId().equals(ql.getId())).count();
            result.put(ql.getId(), count >= 1);
        });
        return result;
    }

    /**
     * 回滚事务标志检查 TODO 可优化成监听形式
     *
     * @param flag 成功标志，false 时回滚事务
     */
    private void roleBackCheck(boolean flag) {
        if (!flag) {
            throw new RuntimeException("操作失败，回滚事务......");
        }
    }

    /**
     * WebSocket 推送
     *
     * @param topic 推送信息
     */
    private void pushTopic(BizMeetingTopic topic) {
        ObjectNode obj = JsonUtil.getMapper().createObjectNode();
        obj.put(WebsocketConst.MSG_VOTE, WebsocketConst.VOTE_TOPIC);
        obj.put(WebsocketConst.MSG_ID, topic.getId());
        obj.put(WebsocketConst.MSG_TXT, JSON.toJSONString(topic));
        pushService.pushMsgToAll(obj.toString());
    }

    private void pushTopicId(BizMeetingTopic topic) {
        ObjectNode obj = JsonUtil.getMapper().createObjectNode();

        obj.put(WebsocketConst.TOPIC_ID, topic.getId());

        pushService.pushMsgToAll(obj.toString());
    }

    @Override
    public void exportDataToExcel(QueryFilter<BizMeetingTopic> queryFilter, HttpServletResponse response) throws IOException {

        // 参数限制
        Assert.notNull(queryFilter, "参数不能为空！必须在querys中传入meetingId！");
        Assert.notNull(queryFilter.getQuerys(), "querys不能为空！必须传入meetingId！");
        String meetingId = "";
        for (QueryField query : queryFilter.getQuerys()) {
            if ("meetingId".equalsIgnoreCase(query.getProperty()) && BeanUtils.isNotEmpty(query.getValue())) {
                meetingId = String.valueOf(query.getValue());
                break;
            }
        }
        Assert.notNull(meetingId, "meetingId不能为空！");

        // 数据处理
        Meeting meeting = meetingManager.get(meetingId);
        Assert.isTrue(BeanUtils.isNotEmpty(meeting), "传入的meetingId没有找到对应的数据！");
        List<BizMeetingTopic> originData = topicManager.query(queryFilter).getRows();
        JSONArray voteProportion = getVoteResult(meetingId);
        List<BizMeetingVote> allVote = list();
        List<BizMeetingVote> currentMeetingVoteRecord = Lists.newArrayList();
//        List<BizMeetingTopic> expertData = Lists.newArrayList();

        originData.forEach(od -> {
            od.setMettingTheme(meeting.getTheme());
            od.setAnonymous("0".equals(od.getAnonymous()) ? "是" : "否");
            od.getQuestionList().forEach(q -> {
                q.setQuestionType("1".equals(q.getQuestionType()) ? "单选" : "多选");

                q.getOptionList().forEach(o -> {
                    for (Object v : voteProportion) {
                        JSONObject vp = (JSONObject) v;
                        if (o.getId().equals(vp.getString("optionId"))) {
                            BigDecimal allCount = BigDecimal.valueOf(vp.getDoubleValue("allUserCount"));
                            BigDecimal votedCount = BigDecimal.valueOf(vp.getDoubleValue("votedUserCount"));
                            BigDecimal proportion = votedCount.divide(allCount, 4, BigDecimal.ROUND_HALF_DOWN);
                            o.setProportion(proportion.multiply(new BigDecimal(100)));
                            o.setPoll(votedCount.intValue());
                            break;
                        }
                    }

                });

                allVote.forEach(av -> {
                    if (q.getId().equals(av.getQuestionId())) {
                        List<String> optionIds = Arrays.asList(av.getOptionId().split(","));
                        q.getOptionList().forEach(o -> {
                            optionIds.forEach(oi -> {
                                if (oi.equals(o.getId())) {
                                    av.setOptionValue((StringUtil.isEmpty(av.getOptionValue()) ? "" : av.getOptionValue()) + o.getOptionValue() + "、");
                                }
                            });
                        });
                        av.setTopicId(od.getId());
                        av.setOptionValue(av.getOptionValue().substring(0, av.getOptionValue().length() - 1));
                        currentMeetingVoteRecord.add(av);
                    }
                });
            });
        });

        // 文件格式处理
        String fileName = "会议『" + meeting.getTheme() + "』投票记录导出结果.xlsx";
        XSSFWorkbook workbook = new XSSFWorkbook();
        originData.forEach(d -> {
            XSSFSheet sheet = workbook.createSheet(d.getTopicContent());
            createTitle(sheet, d);

            int questionCount = 1;
            Integer currentLine = 5;
            for (BizMeetingQuestion question : d.getQuestionList()) {
                XSSFRow qRow0 = sheet.createRow(currentLine);
                qRow0.createCell(0).setCellValue("问题" + questionCount);
                qRow0.createCell(1).setCellValue(question.getQuestionContent());
                boldValuesingle(workbook, qRow0.getCell(0));
                XSSFRow qRow1 = sheet.createRow(++currentLine);
                qRow1.createCell(0).setCellValue("选项类型");
                qRow1.createCell(1).setCellValue(question.getQuestionType());
                XSSFRow oRow0 = sheet.createRow(++currentLine);
                oRow0.createCell(0).setCellValue("选项");
                oRow0.createCell(1).setCellValue("票数");
                oRow0.createCell(2).setCellValue("占比");
                oRow0.createCell(3).setCellValue("用户");

                boldValue(sheet.getWorkbook(), oRow0);

                for (BizMeetingOption option : question.getOptionList()) {

                    XSSFRow oRow1 = sheet.createRow(++currentLine);
                    oRow1.createCell(0).setCellValue(option.getOptionValue());
                    oRow1.createCell(1).setCellValue(String.valueOf(BeanUtils.isEmpty(option.getPoll()) ? 0 : option.getPoll()));
                    oRow1.createCell(2).setCellValue((BeanUtils.isEmpty(option.getProportion()) ? 0 : option.getProportion().stripTrailingZeros().toPlainString()) + "%");

                    //是否匿名
                    Integer anonymous = question.getAnonymous();
                    //当前选项有哪些人选择
                    if(0 == anonymous ){
                        oRow1.createCell(3).setCellValue("匿名投票");
                    }else {
                        //
                        QueryWrapper<BizMeetingVote> queryWrapper = new QueryWrapper<>();
                        queryWrapper.eq("QUESTION_ID_", option.getQuestionId());
                        queryWrapper.like("OPTION_ID_", option.getId());
                        List<BizMeetingVote> list = bizMeetingVoteManager.list(queryWrapper);
                        if(null != list && list.size() != 0 ){
                            String names = list.stream().map(BizMeetingVote::getUserName).collect(Collectors.joining("、"));
                            oRow1.createCell(3).setCellValue(names);
                        }
                    }
                }

                currentLine += 2;
                questionCount++;
            }

//            XSSFRow detailRowTitle = sheet.createRow(++currentLine);
//            detailRowTitle.createCell(0).setCellValue("用户投票明细");
//            boldValuesingle(sheet.getWorkbook(), detailRowTitle.getCell(0));
//            for (int i = 0; i < d.getQuestionList().size(); i++) {
//                BizMeetingQuestion q = d.getQuestionList().get(i);
//                detailRowTitle.createCell(i + 1).setCellValue(q.getQuestionContent());
//            }
//
//            Map<String, List<BizMeetingVote>> currentMeetingVoteRecordGroup = currentMeetingVoteRecord.stream().filter(vote -> d.getId().equals(vote.getTopicId())).collect(Collectors.groupingBy(BizMeetingVote::getUserName));
//            for (Map.Entry<String, List<BizMeetingVote>> record : currentMeetingVoteRecordGroup.entrySet()) {
//                if (BeanUtils.isNotEmpty(record.getValue())) {
//                    XSSFRow detailRow = sheet.createRow(++currentLine);
//                    detailRow.createCell(0).setCellValue(record.getKey());
//
//                    for (int i = 0; i < record.getValue().size(); i++) {
//                        detailRow.createCell(i + 1).setCellValue(record.getValue().get(i).getOptionValue());
//                    }
//                }
//            }

            // 列宽自适应
            for (int i = 0; i < (d.getQuestionList().size() < 2 ? 2 : d.getQuestionList().size()); i++) {
                sheet.autoSizeColumn(i);
            }

        });

        ExcelUtil.downloadExcel(workbook, fileName, response);
    }

    private void createTitle(XSSFSheet sheet, BizMeetingTopic d) {
        XSSFRow row0 = sheet.createRow(0);
        row0.createCell(0).setCellValue("会议主题");
        row0.createCell(1).setCellValue("会议号");

        XSSFRow row1 = sheet.createRow(1);
        row1.createCell(0).setCellValue(d.getMettingTheme());
        row1.createCell(1).setCellValue(d.getMeetingId());

        XSSFRow row2 = sheet.createRow(2);
        row2.createCell(0).setCellValue("投票主题");
        boldValuesingle(sheet.getWorkbook(), row2.getCell(0));
        row2.createCell(1).setCellValue(d.getTopicContent());

        XSSFRow row3 = sheet.createRow(3);
        row3.createCell(0).setCellValue("是否匿名");
        row3.createCell(1).setCellValue(d.getAnonymous());

        boldValue(sheet.getWorkbook(), row0);
    }

    private void boldValue(XSSFWorkbook workbook, XSSFRow... row) {
        XSSFCellStyle style = workbook.createCellStyle();
        XSSFFont font = workbook.createFont();
        //font.setFontHeightInPoints((short) 24); // 字体高度
        //font.setFontName("宋体"); // 字体
        font.setBold(true); // 宽度
        style.setFont(font);
        for (int i = 0; i < row.length; i++) {
            for (int t = 0; t < row[i].getLastCellNum(); t++) {
                row[i].getCell(t).setCellStyle(style);
            }
        }
    }

    private void boldValuesingle(XSSFWorkbook workbook, XSSFCell cell) {
        XSSFCellStyle style = workbook.createCellStyle();
        XSSFFont font = workbook.createFont();
        //font.setFontHeightInPoints((short) 24); // 字体高度
        //font.setFontName("宋体"); // 字体
        font.setBold(true); // 宽度
        style.setFont(font);
        cell.setCellStyle(style);
    }


    @Override
    public JSONArray getVoteResult(String meetingId) {
        JSONArray result = new JSONArray();

        List<MeetingUser> allMU = meetingUserManager.getBaseMapper().selectList(
                new QueryWrapper<MeetingUser>().eq("MEETING_ID_", meetingId)
        );
        Integer attendance = allMU.size();

        List<BizMeetingOption> allOption = optionManager.list();
        List<BizMeetingQuestion> allQuestion = questionManager.list();
        List<BizMeetingVote> voteData = Lists.newArrayList();

        Map<String, String> params = new HashMap<>();
        params.put("meetingId", meetingId);
        baseMapper.getVotesByMeetingId(params).forEach(d -> {
            Arrays.asList(d.getOptionId().split(",")).forEach(o -> {
                BizMeetingVote voteTemp = null;
                try {
                    voteTemp = (BizMeetingVote) BeanUtils.cloneBean(d);
                } catch (Exception e) {
                    throw new RuntimeException(e);
                }
                voteTemp.setOptionId(o);
                voteData.add(voteTemp);
            });
        });


        Map<String, List<BizMeetingVote>> vots = voteData.stream().collect(Collectors.groupingBy(BizMeetingVote::getOptionId));
        for (Map.Entry<String, List<BizMeetingVote>> v : vots.entrySet()) {
            JSONObject data = new JSONObject();
            allOption.forEach(ao -> {
                if (ao.getId().equals(v.getKey())) {
                    allQuestion.forEach(aq -> {
                        if (aq.getId().equals(ao.getQuestionId())) {
                            data.put("topicId", aq.getTopicId());
                        }
                    });
                }
            });
            String votedUser = "", votedUserId = "";
            for (BizMeetingVote vv : v.getValue()) {
                votedUser += vv.getUserName() + ",";
                votedUserId += vv.getUserId() + ",";
            }
            votedUser = votedUser.substring(0, votedUser.length() - 1);
            votedUserId = votedUserId.substring(0, votedUserId.length() - 1);
            data.put("votedUser", votedUser);
            data.put("votedUserId", votedUserId);
            data.put("optionId", v.getKey());
            data.put("allUserCount", attendance);// 加上发起人
            data.put("votedUserCount", v.getValue().size());
            result.add(data);
        }

        return result;
    }

    @Override
    public Integer votedCount(String topicId) {
        Map<String, String> params = new HashMap<>();
        params.put("topicId", topicId);
        return baseMapper.getVotesByMeetingId(params).stream().collect(Collectors.groupingBy(BizMeetingVote::getUserId)).size();
    }
}
