import { defineComponent, getCurrentInstance, ref, computed, watch, onMounted, openBlock, createBlock, createVNode, watchEffect, Fragment, renderList, inject, reactive, nextTick, provide, resolveComponent, resolveDirective, withCtx, withDirectives, createCommentVNode, withKeys, createTextVNode, toDisplayString, vShow } from 'vue';
import { ClickOutside } from '../directives';
import isServer from '../utils/isServer';
import { on, off } from '../utils/dom';
import ElPopper from '../el-popper';
import ElButton from '../el-button';
import ElInput from '../el-input';
import { useLocaleInject } from '../hooks';
import { UPDATE_MODEL_EVENT } from '../utils/constants';
import { useGlobalConfig } from '../utils/util';
import { isValidComponentSize } from '../utils/validators';
import { elFormKey, elFormItemKey } from '../el-form';
import debounce from 'lodash/debounce';

/**
 * Make a map and return a function for checking if a key
 * is in that map.
 * IMPORTANT: all calls of this function must be prefixed with
 * \/\*#\_\_PURE\_\_\*\/
 * So that rollup can tree-shake them if necessary.
 */
const EMPTY_OBJ = (process.env.NODE_ENV !== 'production')
    ? Object.freeze({})
    : {};
const EMPTY_ARR = (process.env.NODE_ENV !== 'production') ? Object.freeze([]) : [];
const hasOwnProperty = Object.prototype.hasOwnProperty;
const hasOwn = (val, key) => hasOwnProperty.call(val, key);

const hsv2hsl = function(hue, sat, val) {
  return [
    hue,
    sat * val / ((hue = (2 - sat) * val) < 1 ? hue : 2 - hue) || 0,
    hue / 2
  ];
};
const isOnePointZero = function(n) {
  return typeof n === "string" && n.indexOf(".") !== -1 && parseFloat(n) === 1;
};
const isPercentage = function(n) {
  return typeof n === "string" && n.indexOf("%") !== -1;
};
const bound01 = function(value, max) {
  if (isOnePointZero(value))
    value = "100%";
  const processPercent = isPercentage(value);
  value = Math.min(max, Math.max(0, parseFloat(value + "")));
  if (processPercent) {
    value = parseInt(value * max + "", 10) / 100;
  }
  if (Math.abs(value - max) < 1e-6) {
    return 1;
  }
  return value % max / parseFloat(max);
};
const INT_HEX_MAP = { 10: "A", 11: "B", 12: "C", 13: "D", 14: "E", 15: "F" };
const toHex = function({ r, g, b }) {
  const hexOne = function(value) {
    value = Math.min(Math.round(value), 255);
    const high = Math.floor(value / 16);
    const low = value % 16;
    return "" + (INT_HEX_MAP[high] || high) + (INT_HEX_MAP[low] || low);
  };
  if (isNaN(r) || isNaN(g) || isNaN(b))
    return "";
  return "#" + hexOne(r) + hexOne(g) + hexOne(b);
};
const HEX_INT_MAP = { A: 10, B: 11, C: 12, D: 13, E: 14, F: 15 };
const parseHexChannel = function(hex) {
  if (hex.length === 2) {
    return (HEX_INT_MAP[hex[0].toUpperCase()] || +hex[0]) * 16 + (HEX_INT_MAP[hex[1].toUpperCase()] || +hex[1]);
  }
  return HEX_INT_MAP[hex[1].toUpperCase()] || +hex[1];
};
const hsl2hsv = function(hue, sat, light) {
  sat = sat / 100;
  light = light / 100;
  let smin = sat;
  const lmin = Math.max(light, 0.01);
  light *= 2;
  sat *= light <= 1 ? light : 2 - light;
  smin *= lmin <= 1 ? lmin : 2 - lmin;
  const v = (light + sat) / 2;
  const sv = light === 0 ? 2 * smin / (lmin + smin) : 2 * sat / (light + sat);
  return {
    h: hue,
    s: sv * 100,
    v: v * 100
  };
};
const rgb2hsv = function(r, g, b) {
  r = bound01(r, 255);
  g = bound01(g, 255);
  b = bound01(b, 255);
  const max = Math.max(r, g, b);
  const min = Math.min(r, g, b);
  let h;
  const v = max;
  const d = max - min;
  const s = max === 0 ? 0 : d / max;
  if (max === min) {
    h = 0;
  } else {
    switch (max) {
      case r: {
        h = (g - b) / d + (g < b ? 6 : 0);
        break;
      }
      case g: {
        h = (b - r) / d + 2;
        break;
      }
      case b: {
        h = (r - g) / d + 4;
        break;
      }
    }
    h /= 6;
  }
  return { h: h * 360, s: s * 100, v: v * 100 };
};
const hsv2rgb = function(h, s, v) {
  h = bound01(h, 360) * 6;
  s = bound01(s, 100);
  v = bound01(v, 100);
  const i = Math.floor(h);
  const f = h - i;
  const p = v * (1 - s);
  const q = v * (1 - f * s);
  const t = v * (1 - (1 - f) * s);
  const mod = i % 6;
  const r = [v, q, p, p, t, v][mod];
  const g = [t, v, v, q, p, p][mod];
  const b = [p, p, t, v, v, q][mod];
  return {
    r: Math.round(r * 255),
    g: Math.round(g * 255),
    b: Math.round(b * 255)
  };
};
class Color {
  constructor(options) {
    this._hue = 0;
    this._saturation = 100;
    this._value = 100;
    this._alpha = 100;
    this.enableAlpha = false;
    this.format = "hex";
    this.value = "";
    options = options || {};
    for (const option in options) {
      if (hasOwn(options, option)) {
        this[option] = options[option];
      }
    }
    this.doOnChange();
  }
  set(prop, value) {
    if (arguments.length === 1 && typeof prop === "object") {
      for (const p in prop) {
        if (hasOwn(prop, p)) {
          this.set(p, prop[p]);
        }
      }
      return;
    }
    this["_" + prop] = value;
    this.doOnChange();
  }
  get(prop) {
    return this["_" + prop];
  }
  toRgb() {
    return hsv2rgb(this._hue, this._saturation, this._value);
  }
  fromString(value) {
    if (!value) {
      this._hue = 0;
      this._saturation = 100;
      this._value = 100;
      this.doOnChange();
      return;
    }
    const fromHSV = (h, s, v) => {
      this._hue = Math.max(0, Math.min(360, h));
      this._saturation = Math.max(0, Math.min(100, s));
      this._value = Math.max(0, Math.min(100, v));
      this.doOnChange();
    };
    if (value.indexOf("hsl") !== -1) {
      const parts = value.replace(/hsla|hsl|\(|\)/gm, "").split(/\s|,/g).filter((val) => val !== "").map((val, index) => index > 2 ? parseFloat(val) : parseInt(val, 10));
      if (parts.length === 4) {
        this._alpha = Math.floor(parseFloat(parts[3]) * 100);
      } else if (parts.length === 3) {
        this._alpha = 100;
      }
      if (parts.length >= 3) {
        const { h, s, v } = hsl2hsv(parts[0], parts[1], parts[2]);
        fromHSV(h, s, v);
      }
    } else if (value.indexOf("hsv") !== -1) {
      const parts = value.replace(/hsva|hsv|\(|\)/gm, "").split(/\s|,/g).filter((val) => val !== "").map((val, index) => index > 2 ? parseFloat(val) : parseInt(val, 10));
      if (parts.length === 4) {
        this._alpha = Math.floor(parseFloat(parts[3]) * 100);
      } else if (parts.length === 3) {
        this._alpha = 100;
      }
      if (parts.length >= 3) {
        fromHSV(parts[0], parts[1], parts[2]);
      }
    } else if (value.indexOf("rgb") !== -1) {
      const parts = value.replace(/rgba|rgb|\(|\)/gm, "").split(/\s|,/g).filter((val) => val !== "").map((val, index) => index > 2 ? parseFloat(val) : parseInt(val, 10));
      if (parts.length === 4) {
        this._alpha = Math.floor(parseFloat(parts[3]) * 100);
      } else if (parts.length === 3) {
        this._alpha = 100;
      }
      if (parts.length >= 3) {
        const { h, s, v } = rgb2hsv(parts[0], parts[1], parts[2]);
        fromHSV(h, s, v);
      }
    } else if (value.indexOf("#") !== -1) {
      const hex = value.replace("#", "").trim();
      if (!/^[0-9a-fA-F]{3}$|^[0-9a-fA-F]{6}$|^[0-9a-fA-F]{8}$/.test(hex))
        return;
      let r, g, b;
      if (hex.length === 3) {
        r = parseHexChannel(hex[0] + hex[0]);
        g = parseHexChannel(hex[1] + hex[1]);
        b = parseHexChannel(hex[2] + hex[2]);
      } else if (hex.length === 6 || hex.length === 8) {
        r = parseHexChannel(hex.substring(0, 2));
        g = parseHexChannel(hex.substring(2, 4));
        b = parseHexChannel(hex.substring(4, 6));
      }
      if (hex.length === 8) {
        this._alpha = Math.floor(parseHexChannel(hex.substring(6)) / 255 * 100);
      } else if (hex.length === 3 || hex.length === 6) {
        this._alpha = 100;
      }
      const { h, s, v } = rgb2hsv(r, g, b);
      fromHSV(h, s, v);
    }
  }
  compare(color) {
    return Math.abs(color._hue - this._hue) < 2 && Math.abs(color._saturation - this._saturation) < 1 && Math.abs(color._value - this._value) < 1 && Math.abs(color._alpha - this._alpha) < 1;
  }
  doOnChange() {
    const { _hue, _saturation, _value, _alpha, format } = this;
    if (this.enableAlpha) {
      switch (format) {
        case "hsl": {
          const hsl = hsv2hsl(_hue, _saturation / 100, _value / 100);
          this.value = `hsla(${_hue}, ${Math.round(hsl[1] * 100)}%, ${Math.round(hsl[2] * 100)}%, ${_alpha / 100})`;
          break;
        }
        case "hsv": {
          this.value = `hsva(${_hue}, ${Math.round(_saturation)}%, ${Math.round(_value)}%, ${_alpha / 100})`;
          break;
        }
        default: {
          const { r, g, b } = hsv2rgb(_hue, _saturation, _value);
          this.value = `rgba(${r}, ${g}, ${b}, ${_alpha / 100})`;
        }
      }
    } else {
      switch (format) {
        case "hsl": {
          const hsl = hsv2hsl(_hue, _saturation / 100, _value / 100);
          this.value = `hsl(${_hue}, ${Math.round(hsl[1] * 100)}%, ${Math.round(hsl[2] * 100)}%)`;
          break;
        }
        case "hsv": {
          this.value = `hsv(${_hue}, ${Math.round(_saturation)}%, ${Math.round(_value)}%)`;
          break;
        }
        case "rgb": {
          const { r, g, b } = hsv2rgb(_hue, _saturation, _value);
          this.value = `rgb(${r}, ${g}, ${b})`;
          break;
        }
        default: {
          this.value = toHex(hsv2rgb(_hue, _saturation, _value));
        }
      }
    }
  }
}

let isDragging = false;
function draggable(element, options) {
  if (isServer)
    return;
  const moveFn = function(event) {
    var _a;
    (_a = options.drag) == null ? void 0 : _a.call(options, event);
  };
  const upFn = function(event) {
    var _a;
    off(document, "mousemove", moveFn);
    off(document, "mouseup", upFn);
    document.onselectstart = null;
    document.ondragstart = null;
    isDragging = false;
    (_a = options.end) == null ? void 0 : _a.call(options, event);
  };
  on(element, "mousedown", function(event) {
    var _a;
    if (isDragging)
      return;
    document.onselectstart = () => false;
    document.ondragstart = () => false;
    on(document, "mousemove", moveFn);
    on(document, "mouseup", upFn);
    isDragging = true;
    (_a = options.start) == null ? void 0 : _a.call(options, event);
  });
}

var script = defineComponent({
  name: "ElSlPanel",
  props: {
    color: {
      type: Object,
      required: true
    }
  },
  setup(props) {
    const instance = getCurrentInstance();
    const cursorTop = ref(0);
    const cursorLeft = ref(0);
    const background = ref("hsl(0, 100%, 50%)");
    const colorValue = computed(() => {
      const hue = props.color.get("hue");
      const value = props.color.get("value");
      return { hue, value };
    });
    function update() {
      const saturation = props.color.get("saturation");
      const value = props.color.get("value");
      const el = instance.vnode.el;
      let { clientWidth: width, clientHeight: height } = el;
      cursorLeft.value = saturation * width / 100;
      cursorTop.value = (100 - value) * height / 100;
      background.value = "hsl(" + props.color.get("hue") + ", 100%, 50%)";
    }
    function handleDrag(event) {
      const el = instance.vnode.el;
      const rect = el.getBoundingClientRect();
      let left = event.clientX - rect.left;
      let top = event.clientY - rect.top;
      left = Math.max(0, left);
      left = Math.min(left, rect.width);
      top = Math.max(0, top);
      top = Math.min(top, rect.height);
      cursorLeft.value = left;
      cursorTop.value = top;
      props.color.set({
        saturation: left / rect.width * 100,
        value: 100 - top / rect.height * 100
      });
    }
    watch(() => colorValue.value, () => {
      update();
    });
    onMounted(() => {
      draggable(instance.vnode.el, {
        drag: (event) => {
          handleDrag(event);
        },
        end: (event) => {
          handleDrag(event);
        }
      });
      update();
    });
    return {
      cursorTop,
      cursorLeft,
      background,
      colorValue,
      handleDrag,
      update
    };
  }
});

const _hoisted_1 = /* @__PURE__ */ createVNode("div", { class: "el-color-svpanel__white" }, null, -1);
const _hoisted_2 = /* @__PURE__ */ createVNode("div", { class: "el-color-svpanel__black" }, null, -1);
const _hoisted_3 = /* @__PURE__ */ createVNode("div", null, null, -1);
function render(_ctx, _cache, $props, $setup, $data, $options) {
  return openBlock(), createBlock("div", {
    class: "el-color-svpanel",
    style: {
      backgroundColor: _ctx.background
    }
  }, [
    _hoisted_1,
    _hoisted_2,
    createVNode("div", {
      class: "el-color-svpanel__cursor",
      style: {
        top: _ctx.cursorTop + "px",
        left: _ctx.cursorLeft + "px"
      }
    }, [
      _hoisted_3
    ], 4)
  ], 4);
}

script.render = render;
script.__file = "packages/color-picker/src/components/sv-panel.vue";

var script$1 = defineComponent({
  name: "ElColorHueSlider",
  props: {
    color: {
      type: Object,
      required: true
    },
    vertical: Boolean
  },
  setup(props) {
    const instance = getCurrentInstance();
    const thumb = ref(null);
    const bar = ref(null);
    const thumbLeft = ref(0);
    const thumbTop = ref(0);
    const hueValue = computed(() => {
      return props.color.get("hue");
    });
    watch(() => hueValue.value, () => {
      update();
    });
    function handleClick(event) {
      const target = event.target;
      if (target !== thumb.value) {
        handleDrag(event);
      }
    }
    function handleDrag(event) {
      const el = instance.vnode.el;
      const rect = el.getBoundingClientRect();
      let hue;
      if (!props.vertical) {
        let left = event.clientX - rect.left;
        left = Math.min(left, rect.width - thumb.value.offsetWidth / 2);
        left = Math.max(thumb.value.offsetWidth / 2, left);
        hue = Math.round((left - thumb.value.offsetWidth / 2) / (rect.width - thumb.value.offsetWidth) * 360);
      } else {
        let top = event.clientY - rect.top;
        top = Math.min(top, rect.height - thumb.value.offsetHeight / 2);
        top = Math.max(thumb.value.offsetHeight / 2, top);
        hue = Math.round((top - thumb.value.offsetHeight / 2) / (rect.height - thumb.value.offsetHeight) * 360);
      }
      props.color.set("hue", hue);
    }
    function getThumbLeft() {
      const el = instance.vnode.el;
      if (props.vertical)
        return 0;
      const hue = props.color.get("hue");
      if (!el)
        return 0;
      return Math.round(hue * (el.offsetWidth - thumb.value.offsetWidth / 2) / 360);
    }
    function getThumbTop() {
      const el = instance.vnode.el;
      if (!props.vertical)
        return 0;
      const hue = props.color.get("hue");
      if (!el)
        return 0;
      return Math.round(hue * (el.offsetHeight - thumb.value.offsetHeight / 2) / 360);
    }
    function update() {
      thumbLeft.value = getThumbLeft();
      thumbTop.value = getThumbTop();
    }
    onMounted(() => {
      const dragConfig = {
        drag: (event) => {
          handleDrag(event);
        },
        end: (event) => {
          handleDrag(event);
        }
      };
      draggable(bar.value, dragConfig);
      draggable(thumb.value, dragConfig);
      update();
    });
    return {
      bar,
      thumb,
      thumbLeft,
      thumbTop,
      hueValue,
      handleClick,
      update
    };
  }
});

function render$1(_ctx, _cache, $props, $setup, $data, $options) {
  return openBlock(), createBlock("div", {
    class: ["el-color-hue-slider", { "is-vertical": _ctx.vertical }]
  }, [
    createVNode("div", {
      ref: "bar",
      class: "el-color-hue-slider__bar",
      onClick: _cache[1] || (_cache[1] = (...args) => _ctx.handleClick && _ctx.handleClick(...args))
    }, null, 512),
    createVNode("div", {
      ref: "thumb",
      class: "el-color-hue-slider__thumb",
      style: {
        left: _ctx.thumbLeft + "px",
        top: _ctx.thumbTop + "px"
      }
    }, null, 4)
  ], 2);
}

script$1.render = render$1;
script$1.__file = "packages/color-picker/src/components/hue-slider.vue";

var script$2 = defineComponent({
  name: "ElColorAlphaSlider",
  props: {
    color: {
      type: Object,
      required: true
    },
    vertical: {
      type: Boolean,
      default: false
    }
  },
  setup(props) {
    const instance = getCurrentInstance();
    const thumb = ref(null);
    const bar = ref(null);
    const thumbLeft = ref(0);
    const thumbTop = ref(0);
    const background = ref(null);
    watch(() => props.color.get("alpha"), () => {
      update();
    });
    watch(() => props.color.value, () => {
      update();
    });
    function getThumbLeft() {
      if (props.vertical)
        return 0;
      const el = instance.vnode.el;
      const alpha = props.color.get("alpha");
      if (!el)
        return 0;
      return Math.round(alpha * (el.offsetWidth - thumb.value.offsetWidth / 2) / 100);
    }
    function getThumbTop() {
      const el = instance.vnode.el;
      if (!props.vertical)
        return 0;
      const alpha = props.color.get("alpha");
      if (!el)
        return 0;
      return Math.round(alpha * (el.offsetHeight - thumb.value.offsetHeight / 2) / 100);
    }
    function getBackground() {
      if (props.color && props.color.value) {
        const { r, g, b } = props.color.toRgb();
        return `linear-gradient(to right, rgba(${r}, ${g}, ${b}, 0) 0%, rgba(${r}, ${g}, ${b}, 1) 100%)`;
      }
      return null;
    }
    function handleClick(event) {
      const target = event.target;
      if (target !== thumb.value) {
        handleDrag(event);
      }
    }
    function handleDrag(event) {
      const el = instance.vnode.el;
      const rect = el.getBoundingClientRect();
      if (!props.vertical) {
        let left = event.clientX - rect.left;
        left = Math.max(thumb.value.offsetWidth / 2, left);
        left = Math.min(left, rect.width - thumb.value.offsetWidth / 2);
        props.color.set("alpha", Math.round((left - thumb.value.offsetWidth / 2) / (rect.width - thumb.value.offsetWidth) * 100));
      } else {
        let top = event.clientY - rect.top;
        top = Math.max(thumb.value.offsetHeight / 2, top);
        top = Math.min(top, rect.height - thumb.value.offsetHeight / 2);
        props.color.set("alpha", Math.round((top - thumb.value.offsetHeight / 2) / (rect.height - thumb.value.offsetHeight) * 100));
      }
    }
    function update() {
      thumbLeft.value = getThumbLeft();
      thumbTop.value = getThumbTop();
      background.value = getBackground();
    }
    onMounted(() => {
      const dragConfig = {
        drag: (event) => {
          handleDrag(event);
        },
        end: (event) => {
          handleDrag(event);
        }
      };
      draggable(bar.value, dragConfig);
      draggable(thumb.value, dragConfig);
      update();
    });
    return {
      thumb,
      bar,
      thumbLeft,
      thumbTop,
      background,
      handleClick,
      update
    };
  }
});

function render$2(_ctx, _cache, $props, $setup, $data, $options) {
  return openBlock(), createBlock("div", {
    class: ["el-color-alpha-slider", { "is-vertical": _ctx.vertical }]
  }, [
    createVNode("div", {
      ref: "bar",
      class: "el-color-alpha-slider__bar",
      style: {
        background: _ctx.background
      },
      onClick: _cache[1] || (_cache[1] = (...args) => _ctx.handleClick && _ctx.handleClick(...args))
    }, null, 4),
    createVNode("div", {
      ref: "thumb",
      class: "el-color-alpha-slider__thumb",
      style: {
        left: _ctx.thumbLeft + "px",
        top: _ctx.thumbTop + "px"
      }
    }, null, 4)
  ], 2);
}

script$2.render = render$2;
script$2.__file = "packages/color-picker/src/components/alpha-slider.vue";

var script$3 = defineComponent({
  props: {
    colors: { type: Array, required: true },
    color: {
      type: Object,
      required: true
    }
  },
  setup(props) {
    const { currentColor } = useOptions();
    const rgbaColors = ref(parseColors(props.colors, props.color));
    watch(() => currentColor.value, (val) => {
      const color = new Color();
      color.fromString(val);
      rgbaColors.value.forEach((item) => {
        item.selected = color.compare(item);
      });
    });
    watchEffect(() => {
      rgbaColors.value = parseColors(props.colors, props.color);
    });
    function handleSelect(index) {
      props.color.fromString(props.colors[index]);
    }
    function parseColors(colors, color) {
      return colors.map((value) => {
        const c = new Color();
        c.enableAlpha = true;
        c.format = "rgba";
        c.fromString(value);
        c.selected = c.value === color.value;
        return c;
      });
    }
    return {
      rgbaColors,
      handleSelect
    };
  }
});

const _hoisted_1$1 = { class: "el-color-predefine" };
const _hoisted_2$1 = { class: "el-color-predefine__colors" };
function render$3(_ctx, _cache, $props, $setup, $data, $options) {
  return openBlock(), createBlock("div", _hoisted_1$1, [
    createVNode("div", _hoisted_2$1, [
      (openBlock(true), createBlock(Fragment, null, renderList(_ctx.rgbaColors, (item, index) => {
        return openBlock(), createBlock("div", {
          key: _ctx.colors[index],
          class: ["el-color-predefine__color-selector", { selected: item.selected, "is-alpha": item._alpha < 100 }],
          onClick: ($event) => _ctx.handleSelect(index)
        }, [
          createVNode("div", {
            style: { "background-color": item.value }
          }, null, 4)
        ], 10, ["onClick"]);
      }), 128))
    ])
  ]);
}

script$3.render = render$3;
script$3.__file = "packages/color-picker/src/components/predefine.vue";

const OPTIONS_KEY = Symbol();
const useOptions = () => {
  return inject(OPTIONS_KEY);
};
var script$4 = defineComponent({
  name: "ElColorPicker",
  components: {
    ElPopper,
    ElInput,
    SvPanel: script,
    HueSlider: script$1,
    AlphaSlider: script$2,
    ElButton,
    Predefine: script$3
  },
  directives: {
    ClickOutside
  },
  props: {
    modelValue: String,
    showAlpha: Boolean,
    colorFormat: String,
    disabled: Boolean,
    size: {
      type: String,
      validator: isValidComponentSize
    },
    popperClass: String,
    predefine: Array
  },
  emits: ["change", "active-change", UPDATE_MODEL_EVENT],
  setup(props, { emit }) {
    const ELEMENT = useGlobalConfig();
    const { t } = useLocaleInject();
    const elForm = inject(elFormKey, {});
    const elFormItem = inject(elFormItemKey, {});
    const hue = ref(null);
    const svPanel = ref(null);
    const alpha = ref(null);
    const popper = ref(null);
    const color = reactive(new Color({
      enableAlpha: props.showAlpha,
      format: props.colorFormat
    }));
    const showPicker = ref(false);
    const showPanelColor = ref(false);
    const customInput = ref("");
    const displayedColor = computed(() => {
      if (!props.modelValue && !showPanelColor.value) {
        return "transparent";
      }
      return displayedRgb(color, props.showAlpha);
    });
    const colorSize = computed(() => {
      return props.size || elFormItem.size || ELEMENT.size;
    });
    const colorDisabled = computed(() => {
      return props.disabled || elForm.disabled;
    });
    const currentColor = computed(() => {
      return !props.modelValue && !showPanelColor.value ? "" : color.value;
    });
    watch(() => props.modelValue, (newVal) => {
      if (!newVal) {
        showPanelColor.value = false;
      } else if (newVal && newVal !== color.value) {
        color.fromString(newVal);
      }
    });
    watch(() => currentColor.value, (val) => {
      customInput.value = val;
      emit("active-change", val);
    });
    watch(() => color.value, () => {
      if (!props.modelValue && !showPanelColor.value) {
        showPanelColor.value = true;
      }
    });
    function displayedRgb(color2, showAlpha) {
      if (!(color2 instanceof Color)) {
        throw Error("color should be instance of _color Class");
      }
      const { r, g, b } = color2.toRgb();
      return showAlpha ? `rgba(${r}, ${g}, ${b}, ${color2.get("alpha") / 100})` : `rgb(${r}, ${g}, ${b})`;
    }
    function setShowPicker(value) {
      showPicker.value = value;
    }
    const debounceSetShowPicker = debounce(setShowPicker, 100);
    function hide() {
      debounceSetShowPicker(false);
      resetColor();
    }
    function resetColor() {
      nextTick(() => {
        if (props.modelValue) {
          color.fromString(props.modelValue);
        } else {
          showPanelColor.value = false;
        }
      });
    }
    function handleTrigger() {
      if (colorDisabled.value)
        return;
      debounceSetShowPicker(!showPicker.value);
    }
    function handleConfirm() {
      color.fromString(customInput.value);
    }
    function confirmValue() {
      var _a;
      const value = color.value;
      emit(UPDATE_MODEL_EVENT, value);
      emit("change", value);
      (_a = elFormItem.formItemMitt) == null ? void 0 : _a.emit("el.form.change", value);
      debounceSetShowPicker(false);
      nextTick(() => {
        const newColor = new Color({
          enableAlpha: props.showAlpha,
          format: props.colorFormat
        });
        newColor.fromString(props.modelValue);
        if (!color.compare(newColor)) {
          resetColor();
        }
      });
    }
    function clear() {
      var _a;
      debounceSetShowPicker(false);
      emit(UPDATE_MODEL_EVENT, null);
      emit("change", null);
      if (props.modelValue !== null) {
        (_a = elFormItem.formItemMitt) == null ? void 0 : _a.emit("el.form.change", null);
      }
      resetColor();
    }
    onMounted(() => {
      if (props.modelValue) {
        color.fromString(props.modelValue);
        customInput.value = currentColor.value;
      }
    });
    watch(() => showPicker.value, () => {
      nextTick(() => {
        var _a, _b, _c;
        (_a = hue.value) == null ? void 0 : _a.update();
        (_b = svPanel.value) == null ? void 0 : _b.update();
        (_c = alpha.value) == null ? void 0 : _c.update();
      });
    });
    provide(OPTIONS_KEY, {
      currentColor
    });
    return {
      color,
      colorDisabled,
      colorSize,
      displayedColor,
      showPanelColor,
      showPicker,
      customInput,
      handleConfirm,
      hide,
      handleTrigger,
      clear,
      confirmValue,
      t,
      hue,
      svPanel,
      alpha,
      popper
    };
  }
});

const _hoisted_1$2 = { class: "el-color-dropdown__main-wrapper" };
const _hoisted_2$2 = { class: "el-color-dropdown__btns" };
const _hoisted_3$1 = { class: "el-color-dropdown__value" };
const _hoisted_4 = {
  key: 0,
  class: "el-color-picker__mask"
};
const _hoisted_5 = {
  key: 0,
  class: "el-color-picker__empty el-icon-close"
};
const _hoisted_6 = { class: "el-color-picker__icon el-icon-arrow-down" };
function render$4(_ctx, _cache, $props, $setup, $data, $options) {
  const _component_hue_slider = resolveComponent("hue-slider");
  const _component_sv_panel = resolveComponent("sv-panel");
  const _component_alpha_slider = resolveComponent("alpha-slider");
  const _component_predefine = resolveComponent("predefine");
  const _component_el_input = resolveComponent("el-input");
  const _component_el_button = resolveComponent("el-button");
  const _component_el_popper = resolveComponent("el-popper");
  const _directive_click_outside = resolveDirective("click-outside");
  return openBlock(), createBlock(_component_el_popper, {
    ref: "popper",
    visible: _ctx.showPicker,
    "onUpdate:visible": _cache[3] || (_cache[3] = ($event) => _ctx.showPicker = $event),
    effect: "light",
    "manual-mode": "",
    trigger: "click",
    "show-arrow": false,
    "fallback-placements": ["bottom", "top", "right", "left"],
    offset: 0,
    transition: "el-zoom-in-top",
    "gpu-acceleration": false,
    "popper-class": `el-color-picker__panel el-color-dropdown ${_ctx.popperClass}`,
    "stop-popper-mouse-event": false
  }, {
    default: withCtx(() => [
      withDirectives(createVNode("div", null, [
        createVNode("div", _hoisted_1$2, [
          createVNode(_component_hue_slider, {
            ref: "hue",
            class: "hue-slider",
            color: _ctx.color,
            vertical: ""
          }, null, 8, ["color"]),
          createVNode(_component_sv_panel, {
            ref: "svPanel",
            color: _ctx.color
          }, null, 8, ["color"])
        ]),
        _ctx.showAlpha ? (openBlock(), createBlock(_component_alpha_slider, {
          key: 0,
          ref: "alpha",
          color: _ctx.color
        }, null, 8, ["color"])) : createCommentVNode("v-if", true),
        _ctx.predefine ? (openBlock(), createBlock(_component_predefine, {
          key: 1,
          ref: "predefine",
          color: _ctx.color,
          colors: _ctx.predefine
        }, null, 8, ["color", "colors"])) : createCommentVNode("v-if", true),
        createVNode("div", _hoisted_2$2, [
          createVNode("span", _hoisted_3$1, [
            createVNode(_component_el_input, {
              modelValue: _ctx.customInput,
              "onUpdate:modelValue": _cache[1] || (_cache[1] = ($event) => _ctx.customInput = $event),
              "validate-event": false,
              size: "mini",
              onKeyup: withKeys(_ctx.handleConfirm, ["enter"]),
              onBlur: _ctx.handleConfirm
            }, null, 8, ["modelValue", "onKeyup", "onBlur"])
          ]),
          createVNode(_component_el_button, {
            size: "mini",
            type: "text",
            class: "el-color-dropdown__link-btn",
            onClick: _ctx.clear
          }, {
            default: withCtx(() => [
              createTextVNode(toDisplayString(_ctx.t("el.colorpicker.clear")), 1)
            ]),
            _: 1
          }, 8, ["onClick"]),
          createVNode(_component_el_button, {
            plain: "",
            size: "mini",
            class: "el-color-dropdown__btn",
            onClick: _ctx.confirmValue
          }, {
            default: withCtx(() => [
              createTextVNode(toDisplayString(_ctx.t("el.colorpicker.confirm")), 1)
            ]),
            _: 1
          }, 8, ["onClick"])
        ])
      ], 512), [
        [_directive_click_outside, _ctx.hide]
      ])
    ]),
    trigger: withCtx(() => [
      createVNode("div", {
        class: [
          "el-color-picker",
          _ctx.colorDisabled ? "is-disabled" : "",
          _ctx.colorSize ? `el-color-picker--${_ctx.colorSize}` : ""
        ]
      }, [
        _ctx.colorDisabled ? (openBlock(), createBlock("div", _hoisted_4)) : createCommentVNode("v-if", true),
        createVNode("div", {
          class: "el-color-picker__trigger",
          onClick: _cache[2] || (_cache[2] = (...args) => _ctx.handleTrigger && _ctx.handleTrigger(...args))
        }, [
          createVNode("span", {
            class: ["el-color-picker__color", { "is-alpha": _ctx.showAlpha }]
          }, [
            createVNode("span", {
              class: "el-color-picker__color-inner",
              style: {
                backgroundColor: _ctx.displayedColor
              }
            }, null, 4),
            !_ctx.modelValue && !_ctx.showPanelColor ? (openBlock(), createBlock("span", _hoisted_5)) : createCommentVNode("v-if", true)
          ], 2),
          withDirectives(createVNode("span", _hoisted_6, null, 512), [
            [vShow, _ctx.modelValue || _ctx.showPanelColor]
          ])
        ])
      ], 2)
    ]),
    _: 1
  }, 8, ["visible", "popper-class"]);
}

script$4.render = render$4;
script$4.__file = "packages/color-picker/src/index.vue";

script$4.install = (app) => {
  app.component(script$4.name, script$4);
};
const _ColorPicker = script$4;

export default _ColorPicker;
