package com.artfess.reform.statistics.controller;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.util.BeanUtils;
import com.artfess.base.util.JsonUtil;
import com.artfess.poi.util.ExcelUtil;
import com.artfess.reform.fill.model.BizReformDemandAssignment;
import com.artfess.reform.majorProjects.manager.PilotMeasuresPushManager;
import com.artfess.reform.majorProjects.model.PilotBrandPush;
import com.artfess.reform.majorProjects.model.PilotExperiencePush;
import com.artfess.reform.majorProjects.model.PilotMeasuresPush;
import com.artfess.reform.majorProjects.model.PilotSpecificSchedule;
import com.artfess.reform.majorProjects.vo.PilotMeasuresPushVo;
import com.artfess.reform.majorProjects.vo.QuantitativeObjectivesVo;
import com.artfess.reform.statistics.manager.ProjectLargeScreeManager;
import com.artfess.reform.statistics.model.BizScoringProjectIteration;
import com.artfess.reform.statistics.vo.*;
import com.artfess.sysConfig.persistence.manager.SysDictionaryDetailManager;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.Assert;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 * 重大项目改革报表大屏接口
 *
 * @author 陈实
 * @since 2023/4/24 19:59
 */
@Slf4j
@RestController
@Api(tags = "重大项目改革报表大屏接口")
@RequestMapping("/project/largeScreen/v1/")
@ApiGroup(group = {ApiGroupConsts.GROUP_BIZ})
public class ProjectLargeScreeController {

    @Resource
    private ProjectLargeScreeManager baseManager;

    @Resource
    private PilotMeasuresPushManager measuresPushManager;

    @Autowired
    SysDictionaryDetailManager sysDictionaryDetailManager;

    @PostMapping("/indexRank")
    @ApiOperation(value = "指数排行榜接口", notes = "URI参数type: composite 综合排名、single 单项")
    public CommonResult<String> indexRank(@RequestParam String type,@RequestParam(required = false) Integer year, @RequestParam(required = false) Integer month) {
        JSONArray data = baseManager.indexRank(type,year,month);
        return CommonResult.success(data, "查询成功");
    }

    @PostMapping(value = "/monthPlanNum", produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "项目作战图统计", httpMethod = "POST", notes = "项目作战图统计")
    public CommonResult<String> monthPlanNum(@RequestParam(required = false) String projectId,@RequestParam(required = false) Integer year, @RequestParam(required = false) Integer month) {
        JSONObject data = baseManager.monthPlanNum(projectId,year,month);
        return CommonResult.success(data, "查询成功");
    }

    @PostMapping(value = "/monthPlanNumForProject", produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "各个项目作战图统计", httpMethod = "POST", notes = "项目作战图统计")
    public CommonResult<String> monthPlanNumForProject(@RequestParam(required = false) Integer year, @RequestParam(required = false) Integer month) {
        List<MonthPlanNumVO> list = baseManager.monthPlanNumForProject(year,month);
        return CommonResult.success(list, "查询成功");
    }

    @PostMapping(value = "/monthPlanNumDetails", produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "项目作战图详情统计", httpMethod = "POST", notes = "项目作战图详情统计")
    public CommonResult<String> monthPlanNumDetails(@RequestParam(required = false) Integer year, @RequestParam(required = false) Integer month, @RequestParam(required = false) String projectId) {
        JSONObject data = baseManager.monthPlanNumDetails(year,month,projectId);
        return CommonResult.success(data, "查询成功");
    }


    @PostMapping(value = "/monthPlanPageList", produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "分页查询项目月度计划及推进情况", httpMethod = "POST", notes = "分页查询项目月度计划及推进情况")
    public PageList<PilotMeasuresPushVo> monthPlanPageList(@ApiParam(name = "queryFilter", value = "查询对象") @RequestBody QueryFilter<PilotMeasuresPush> queryFilter) {
        PageList<PilotMeasuresPushVo> pageList = measuresPushManager.queryList(queryFilter);
        return pageList;
    }

    @RequestMapping(value = "exportMonthPlanList", method = RequestMethod.POST, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "导出月度计划推进列表", httpMethod = "POST", notes = "导出月度计划推进列表")
    public void exportList(HttpServletResponse response, @ApiParam(name = "queryFilter", value = "通用查询对象") @RequestBody QueryFilter<PilotMeasuresPush> queryFilter) throws Exception {
        try {
            queryFilter.getPageBean().setPageSize(-1);
            PageList<PilotMeasuresPushVo> pageList = measuresPushManager.queryList(queryFilter);
            List<Map<String, Object>> list = new ArrayList<>();
            if(BeanUtils.isNotEmpty(pageList.getRows())){


                for (PilotMeasuresPushVo effect : pageList.getRows()) {
                    String jsonStr = JsonUtil.toJson(effect);
                    Map<String, Object> map = JsonUtil.toMap(jsonStr);

                    //项目类型
                    if(map.get("projectType")!=null){
                        String projectType = map.get("projectType").toString();
                        String projectTypeText = sysDictionaryDetailManager.getTextByValue("xmlx",projectType);
                        map.put("projectTypeText",projectTypeText);
                    }else{
                        map.put("projectTypeText","");
                    }

                    //所属领域
                    if(map.get("projectField")!=null){
                        String projectField = map.get("projectField").toString();
                        String projectFieldText = sysDictionaryDetailManager.getTextByValue("ggbb_ssly",projectField);
                        map.put("projectFieldText",projectFieldText);
                    }else{
                        map.put("projectFieldText","");
                    }

                    //所属跑道
                    if(map.get("projectRunway")!=null){
                        String projectRunway = map.get("projectRunway").toString();
                        String projectRunwayText = sysDictionaryDetailManager.getTextByValue("ggbb_sspd",projectRunway);
                        map.put("projectRunwayText",projectRunwayText);
                    }else{
                        map.put("projectRunwayText","");
                    }

                    //计划月度
                    if(map.get("planYear")!=null&&map.get("planMonth")!=null){
                        String planYear = map.get("planYear").toString();
                        String planMonth = map.get("planMonth").toString();
                        String planMonthText =planYear+"-"+planMonth;
                        map.put("planMonthText",planMonthText);
                    }else{
                        map.put("planMonthText","");
                    }

                    //推进状态
                    if(map.get("isOver")!=null){
                        String isOver = map.get("isOver").toString();
                        String isOverText = sysDictionaryDetailManager.getTextByValue("tjzt",isOver);
                        map.put("isOverText",isOverText);
                    }else{
                        map.put("isOverText","");
                    }

                    //计划类型
                    if(map.get("planType")!=null){
                        String planType = map.get("planType").toString();
                        String planTypeText = sysDictionaryDetailManager.getTextByValue("yd_jhlx",planType);
                        map.put("planTypeText",planTypeText);
                    }else{
                        map.put("planTypeText","");
                    }

                    list.add(map);
                }
            }
            String tempName = "重大项目月度计划推进表";
            Map<String, String> exportMaps = new LinkedHashMap<String, String>();
            exportMaps.put("projectName", "项目名称");
            exportMaps.put("leader", "牵头市领导");
            exportMaps.put("projectFieldText", "所属领域");
            exportMaps.put("projectRunwayText", "所属跑道");
            exportMaps.put("unitName", "责任单位");
            exportMaps.put("projectTypeText", "项目类型");
            exportMaps.put("planMonthText", "计划月度");
            exportMaps.put("planTypeText", "计划类型");
            exportMaps.put("isOverText", "推进状态");
            exportMaps.put("planContent", "计划内容");
            exportMaps.put("completedWork", "已完成工作");
            exportMaps.put("problems", "存在问题");
            exportMaps.put("workPlan", "下一步工作计划");
            HSSFWorkbook book = ExcelUtil.exportTemplateExcel(tempName, exportMaps, list,1,"excel/monthPlanPush.xls");
            ExcelUtil.downloadExcel(book, tempName, response);
        } catch (Exception e) {
            throw new RuntimeException("导出失败：" + e.getMessage());
        }
    }

    @PostMapping("/targetNum")
    @ApiOperation(value = "量化目标")
    public CommonResult<String> targetNum(@RequestParam(required = false) Integer year, @RequestParam(required = false) Integer month) {
        JSONObject data = baseManager.targetNum(year,month);
        return CommonResult.success(data, "查询成功");
    }


    @PostMapping("/demandHeatList")
    @ApiOperation(value = "需求热力榜", notes = "month: 月度")
    public JSONObject demandHeatList(@RequestParam(required = false) Integer year,@RequestParam Integer month) {
        Assert.isTrue(month >= 1 && month <= 12, "月度传递错误，允许值【1~12】");
        return baseManager.demandHeatList(year,month);
    }

    @PostMapping("/demandDetailList")
    @ApiOperation(value = "改革需求清单")
    public JSONObject demandDetailList(@ApiParam(name = "queryFilter", value = "查询对象") @RequestBody QueryFilter<BizReformDemandAssignment> queryFilter) {
        return baseManager.demandDetailList(queryFilter);
    }

    @PostMapping(value="/brandLineChart")
    @ApiOperation("按月统计品牌榜数据（折线图）")
    public List<BrandLineChartVO> brandLineChart(@RequestParam(required = false) String projectId,@RequestParam(required = false) Integer year, @RequestParam(required = false) Integer month) {
        List<BrandLineChartVO> list = baseManager.brandLineChart(projectId,year,month);
        return list;
    }

    @PostMapping(value="/queryBrandForProject")
    @ApiOperation("按照项目统计品牌榜数据")
    public List<BrandLineChartVO> queryBrandForProject(@RequestParam(required = false) String projectId,@RequestParam(required = false) Integer year, @RequestParam(required = false) Integer month) {
        List<BrandLineChartVO> list = baseManager.queryBrandForProject(projectId,year,month);
        return list;
    }

    @PostMapping(value="/queryBrandForProjectInfo")
    @ApiOperation("项目品牌榜详情数据")
    public List<PilotBrandPush> queryBrandForProjectInfo(@RequestParam(required = false) String projectId, @RequestParam(required = false) Integer year, @RequestParam(required = false) Integer month) {
        return baseManager.queryBrandForProjectInfo(projectId,year,month);
    }

    @PostMapping(value="/queryIterationForProject")
    @ApiOperation("按照项目统计迭代次数（迭代榜）")
    public List<BizScoringProjectIteration> queryIterationForProject(@RequestParam(value = "year",required = false) Integer year, @RequestParam(value = "month",required = false) Integer month) {
        return baseManager.queryIterationForProject(year,month);
    }

    @PostMapping(value="/queryIterationForProjectInfo")
    @ApiOperation("按照项目查询迭代数据")
    public IterationVo queryIterationForProjectInfo(@RequestParam("projectId") String projectId,@RequestParam(value = "year",required = false) Integer year,@RequestParam(value = "month",required = false) Integer month) {
        return baseManager.queryIterationForProjectInfo(projectId,year,month);
    }

    @PostMapping(value="/achievementCount")
    @ApiOperation("成果榜数量和环比统计")
    public ScheduleVo achievementCount(@RequestParam(required = false) String projectId,@RequestParam(required = false) Integer year, @RequestParam(required = false) Integer month) {
        return baseManager.achievementCount(projectId,year,month);
    }

    @PostMapping(value="/achievementInvCount")
    @ApiOperation("查询各个重大项目成果数量")
    public List<ScheduleVo> achievementInvCount(@RequestParam(required = false) Integer year, @RequestParam(required = false) Integer month) {
        return baseManager.achievementInvCount(year,month);
    }

    @PostMapping(value="/achievementInfo")
    @ApiOperation("成果榜查询")
    public PageList<PilotMeasuresPush> achievementInfo(@RequestBody QueryFilter<PilotMeasuresPush> queryFilter) {
        return baseManager.achievementInfo(queryFilter);
    }


    @PostMapping(value="/achievementFindById/{id}")
    @ApiOperation("成果榜查询详情")
    public PilotMeasuresPush achievementFindById(@PathVariable(value = "id")String id) {
        return measuresPushManager.achievementFindById(id);
    }

    @PostMapping(value="/reachRateCount", produces={"application/json; charset=utf-8" })
    @ApiOperation("指标库展示页面查询")
    public IndexVo reachRateCount(@RequestParam(required = false) String projectId, @RequestParam(required = false) Integer year, @RequestParam(required = false) Integer month) {
        return baseManager.reachRateCount(projectId,year,month);
    }

    @PostMapping(value="/reachProjectRateCount")
    @ApiOperation("指标库环形图查询")
    public IndexProjectVo reachProjectRateCount(@RequestParam(required = false) Integer year, @RequestParam(required = false) Integer month) {
        return baseManager.reachProjectRateCount(year,month);
    }

    @PostMapping(value="/indexDetails")
    @ApiOperation("指标库详情页面查询")
    public IndexDetailsVo indexDetails(@RequestParam(required = false) String projectId, @RequestParam(required = false) Integer year, @RequestParam(required = false) Integer month) {
        return baseManager.indexDetails(projectId,year,month);
    }

    @PostMapping(value="/schedulePlanInfo")
    @ApiOperation("市级大屏进入各项目分驾驶舱月度计划清单查询")
    public List<PilotSpecificSchedule> schedulePlanInfo(@RequestParam(required = false) String projectId, @RequestParam(required = false) Integer year, @RequestParam(required = false) Integer month) {
        return baseManager.schedulePlanInfo(projectId,year,month);
    }

    @PostMapping(value="/pilotExperiencePushInfo")
    @ApiOperation("市级大屏进入各项目分驾驶舱经验推广查询")
    public List<PilotExperiencePush> pilotExperiencePushInfo(@RequestParam(required = false) String projectId, @RequestParam(required = false) Integer year, @RequestParam(required = false) Integer month) {
        return baseManager.pilotExperiencePushInfo(projectId,year,month);
    }

    @PostMapping(value="/quantitativeObjectivesInfo")
    @ApiOperation("市级大屏进入各项目分驾驶舱量化目标查询")
    public  List<QuantitativeObjectivesVo> quantitativeObjectivesInfo(@RequestParam(required = false) String projectId, @RequestParam(required = false) Integer year, @RequestParam(required = false) Integer month) {
        return baseManager.quantitativeObjectivesInfo(projectId,year,month);
    }

    @PostMapping(value="/acceptanceRateInfo")
    @ApiOperation("区县大屏重点关注-验收率详情")
    public AcceptanceVo quantitativeObjectivesInfo(@RequestParam(required = false) LocalDate countDate, @RequestParam(required = false) String unitId) {
        return baseManager.acceptanceRateInfo(countDate,unitId);
    }
}
