package com.artfess.reform.fill.manager.impl;

import com.artfess.base.exception.ApplicationException;
import com.artfess.base.manager.impl.BaseManagerImpl;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.util.BeanUtils;
import com.artfess.reform.approvalLog.manager.AchieveStatusLogManager;
import com.artfess.reform.approvalLog.vo.AchieveStatusLogAndAuditResultVo;
import com.artfess.reform.fill.dao.BizReformDemandAnalysisLibraryDao;
import com.artfess.reform.fill.manager.BizReformDemandAnalysisLibraryManager;
import com.artfess.reform.fill.model.BizReformDemandAnalysisLibrary;
import com.artfess.reform.fill.model.BizReformDemandAssignmentRecord;
import com.artfess.reform.majorProjects.model.MunicipalMajorProjectManagement;
import com.artfess.reform.statistics.vo.CountyChartVo;
import com.artfess.reform.utils.BizUtils;
import com.artfess.sysConfig.persistence.param.DictModel;
import com.artfess.uc.api.impl.util.ContextUtil;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.Date;
import java.util.List;

/**
 * 改革需求分析-需求库表 服务实现类
 *
 * @company 阿特菲斯信息技术有限公司
 * @author 管理员
 * @since 2023-03-07
 */
@Service
public class BizReformDemandAnalysisLibraryManagerImpl extends BaseManagerImpl<BizReformDemandAnalysisLibraryDao, BizReformDemandAnalysisLibrary> implements BizReformDemandAnalysisLibraryManager {

    @Resource
    AchieveStatusLogManager achieveStatusLogManager;

    @Transactional
    @Override
    public String insertDemand(BizReformDemandAnalysisLibrary demand) {
        LocalDateTime localDateTime =LocalDateTime.now();
        if(demand.getStatus()==null){
            demand.setStatus(0);
        }
        demand.setDemandDate(new Date());
        demand.setDemandYear(localDateTime.getYear());
        demand.setDemandMonth(localDateTime.getMonthValue());
        demand.setDemandQuarter(Integer.valueOf(localDateTime.getMonthValue()+2)/3);
        demand.setReportUnitId(ContextUtil.getCurrentOrgId());
        demand.setReportUnitName(ContextUtil.getCurrentOrgName());
        demand.setResponsibleUnitCode(ContextUtil.getCurrentOrgCode());
        demand.setResponsibleUnitGrade(ContextUtil.getCurrentOrgGrade());
        this.baseMapper.insert(demand);
        //记录操作日志
        setApprovalLog(demand.getId(),demand.getStatus(),"",demand.getApprovalResults());
        return demand.getId();
    }

    @Override
    @Transactional
    public String updateDemand(BizReformDemandAnalysisLibrary demand) {
        BizReformDemandAnalysisLibrary old = this.baseMapper.selectById(demand.getId());
        //判断是否满足记录操作日志条件,避免重复记录
        boolean check=old==null||old.getStatus().equals(demand.getStatus());
        if(old.getStatus()==0 || old.getStatus()==2|| old.getStatus()==4){
            this.baseMapper.updateById(demand);
            //记录操作日志
            if (check)setApprovalLog(demand.getId(),demand.getStatus(),"",demand.getApprovalResults());
        }else{
            throw new ApplicationException("已开始审核不能修改！");
        }
        return demand.getId();
    }

    @Override
    @Transactional
    public String updateAuditStatus(BizReformDemandAnalysisLibrary demand) {
        if(demand.getStatus()==3){
            demand.setAuditDate(new Date());
            demand.setAuditYear(LocalDateTime.now().getYear());
            demand.setAuditMonth(LocalDateTime.now().getMonthValue());
            demand.setAuditQuarter(Integer.valueOf(LocalDateTime.now().getMonthValue()+2)/3);
        }
        //判断是否满足操作日志记录条件,避免重复记录
        BizReformDemandAnalysisLibrary entity = this.getOne(
                new QueryWrapper<BizReformDemandAnalysisLibrary>()
                        .eq("ID_", demand.getId())
                        .eq("IS_DELE_", "0"), false);
        boolean check=entity==null||entity.getStatus()!=demand.getStatus();
        this.baseMapper.updateById(demand);
        //记录操作日志
        if(check)setApprovalLog(demand.getId(), demand.getStatus(),demand.getApprovalComments(),demand.getApprovalResults());
        return demand.getId();
    }

    @Override
    @Transactional
    public String updateAdoptStatus(BizReformDemandAnalysisLibrary demand) {
        if(demand.getStatus()!=5){
            throw new ApplicationException("改革办尚未确认！");
        }
        this.baseMapper.updateById(demand);
        return demand.getId();
    }

    @Override
    @Transactional
    public boolean deleteBatchByIds(List<String> ids) {
        Assert.notEmpty(ids, "id不能为空！");
        QueryWrapper<BizReformDemandAnalysisLibrary> queryWrapper = new QueryWrapper<>();
        queryWrapper.in("ID_",ids).and(qr -> qr.eq("STATUS_", 0).or().eq("STATUS_", 2).or().eq("STATUS_", 4));
        List<BizReformDemandAnalysisLibrary> list = this.baseMapper.selectList(queryWrapper);
        if(list.size()!=ids.size()){
            throw new ApplicationException("非草稿状态的数据不能删除！");
        }
        this.baseMapper.deleteBatchIds(ids);
        return true;
    }

    /**
     * 设置审批日志实体类录入
     *
     * @param id
     * @param status
     */
    @Transactional
    public void setApprovalLog(String id, Integer status,String approvalComments,Integer approvalResults) {
        // 审批日志
        BizReformDemandAnalysisLibrary demand =  this.baseMapper.selectById(id);
        String title = demand.getDemandTopic();
        AchieveStatusLogAndAuditResultVo logVo = new AchieveStatusLogAndAuditResultVo();
        logVo.setBizId(id);
        logVo.setModeType(1);
        logVo.setBizType("1");
        logVo.setModuleTableName("biz_reform_demand_analysis_library");
        logVo.setApprovalComments(approvalComments);
        if (approvalResults!=null &&approvalResults==2){
            if(status==0){
                logVo.setItemTitle("【"+title+"】撤回为草稿！");
                logVo.setOperateInfo("撤回为草稿！");
            }else if(status==1){
                logVo.setItemTitle("【"+title+"】撤回为提交审核！");
                logVo.setOperateInfo("撤回为提交审核！");
            }else if(status==3){
                logVo.setItemTitle("【"+title+"】撤回为审核通过！");
                logVo.setOperateInfo("撤回为审核通过！");
            }
            logVo.setApprovalResults(2);
        }else {
            //0：草稿，1：已上报待审核，2：审核驳回，3：审核通过改革办待审核 ， 4：改革办审核确认驳回 ， 5：改革办审核确认完成
            if(status==0){
                logVo.setItemTitle("【"+title+"】录入！");
                logVo.setOperateInfo("改革需求信息录入！");
                logVo.setApprovalResults(0);
            }else if(status==1){
                logVo.setItemTitle("【"+title+"】提交审核！");
                logVo.setOperateInfo("提交审核");
                logVo.setApprovalResults(0);
            }else if(status==2){
                logVo.setItemTitle("【"+title+"】审核驳回！");
                logVo.setOperateInfo("审核驳回");
                logVo.setApprovalResults(1);
            }else if(status==3){
                logVo.setItemTitle("【"+title+"】审核通过！");
                logVo.setOperateInfo("审核通过");
                logVo.setApprovalResults(0);
            }else if(status==4){
                logVo.setItemTitle("【"+title+"】改革办驳回！");
                logVo.setOperateInfo("改革办驳回");
                logVo.setApprovalResults(1);
            }else if(status==5){
                logVo.setItemTitle("【"+title+"】改革办通过！");
                logVo.setOperateInfo("改革办通过");
                logVo.setApprovalResults(0);
            }
        }
        logVo.setStatus(status.toString());
        logVo.setModeType(0);
        logVo.setPriority(0);
        logVo.setCreateOrgId(ContextUtil.getCurrentOrgId());
        logVo.setOperateOrgId(ContextUtil.getCurrentOrgId());
        logVo.setOperateOrgName(ContextUtil.getCurrentOrgName());
        logVo.setOperateDeptId(ContextUtil.getCurrentDeptId());
        logVo.setOperateDeptName(ContextUtil.getCurrentDeptName());
        logVo.setOperateUserId(ContextUtil.getCurrentUserId());
        logVo.setOperateUserName(ContextUtil.getCurrentUserName());
        logVo.setOperateUserAccount(ContextUtil.getCurrentUserAccount());
        logVo.setOperateDate(LocalDateTime.now());
        // 保存审批日志
        achieveStatusLogManager.createAchieveStatusLogAndAuditResult(logVo);
    }

    @Override
    public List<CountyChartVo> chartDataForLargeScreen(Integer year,Integer month) {
        return baseMapper.chartDataForLargeScreen(year, month);
    }

    @Override
    public PageList<BizReformDemandAnalysisLibrary> queryByPage(QueryFilter<BizReformDemandAnalysisLibrary> queryFilter) {
        IPage<BizReformDemandAnalysisLibrary> iPage = baseMapper.queryByPage(convert2IPage(queryFilter.getPageBean()), convert2Wrapper(queryFilter, currentModelClass()));
        return new PageList<>(iPage);
    }
}
