package com.artfess.gyl.supplierEva.manager.impl;

import com.artfess.base.exception.ApplicationException;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.util.BeanUtils;
import com.artfess.file.model.DefaultFile;
import com.artfess.file.persistence.manager.FileManager;
import com.artfess.gyl.supplierEva.dto.ProcessDTO;
import com.artfess.gyl.supplierEva.dto.ResultEvaDTO;
import com.artfess.gyl.supplierEva.dto.SupplierInfoDTO;
import com.artfess.gyl.supplierEva.manager.BizSupplierPerformanceEvaluationAcknowledgementManager;
import com.artfess.gyl.supplierEva.manager.BizSupplierPerformanceEvaluationCorrectionReportManager;
import com.artfess.gyl.supplierEva.model.BizSupplierPerformanceEvaluationAcknowledgement;
import com.artfess.gyl.supplierEva.model.BizSupplierPerformanceEvaluationCorrectionReport;
import com.artfess.gyl.supplierEva.model.BizSupplierPerformanceEvaluationNotification;
import com.artfess.gyl.supplierEva.dao.BizSupplierPerformanceEvaluationNotificationDao;
import com.artfess.gyl.supplierEva.manager.BizSupplierPerformanceEvaluationNotificationManager;
import com.artfess.base.manager.impl.BaseManagerImpl;
import com.artfess.gyl.supplierEva.vo.EvaluationVo;
import com.artfess.uc.api.impl.util.ContextUtil;
import com.artfess.uc.api.model.IUser;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import org.apache.commons.lang.StringUtils;
import org.apache.http.util.Asserts;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.awt.geom.RectangularShape;
import java.io.Serializable;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 供应商绩效考核通知信息表 服务实现类
 *
 * @author huqi
 * @since 2025-11-12
 */
@Service
public class BizSupplierPerformanceEvaluationNotificationManagerImpl extends BaseManagerImpl<BizSupplierPerformanceEvaluationNotificationDao, BizSupplierPerformanceEvaluationNotification> implements BizSupplierPerformanceEvaluationNotificationManager {

    @Resource
    private BizSupplierPerformanceEvaluationAcknowledgementManager acknowledgementManager;

    @Resource
    private BizSupplierPerformanceEvaluationCorrectionReportManager reportManager;

    @Resource
    private FileManager fileManager;

    @Override
    public void saveEva(ResultEvaDTO dto) {
        Asserts.notNull(dto.getSupplierId(), "需通知的供应商不能为空");
        String processType = dto.getProcessType();
        final String PROCESS_TYPE_NORMAL = "0";
        final String PROCESS_TYPE_DEADLINE = "2";
        if (PROCESS_TYPE_NORMAL.equals(processType)) {
            return;
        }
        BizSupplierPerformanceEvaluationNotification notification = new BizSupplierPerformanceEvaluationNotification();
        if (StringUtils.isNotBlank(dto.getId())) {
            notification.setId(dto.getId());
        }
        boolean isDeadlineProcess = PROCESS_TYPE_DEADLINE.equals(processType);
        Integer distributeType = dto.getDistributeType();
        // 设置企业信息及发起人信息
        IUser user = ContextUtil.getCurrentUser();
        String currentOrgId = ContextUtil.getCurrentOrgId();
        String currentOrgName = ContextUtil.getCurrentOrgName();
        String currentOrgCode = ContextUtil.getCurrentOrgCode();
        notification.setInitiatorId(user.getUserId());
        notification.setInitiatorAccount(user.getAccount());
        notification.setInitiatorName(user.getFullname());
        notification.setEnterpriseId(BeanUtils.isNotEmpty(dto.getEnterpriseId()) ? dto.getEnterpriseId() : currentOrgId);
        notification.setEnterpriseName(BeanUtils.isNotEmpty(dto.getEnterpriseName()) ? dto.getEnterpriseName() : currentOrgName);
        notification.setEnterpriseCode(BeanUtils.isNotEmpty(dto.getEnterpriseCode()) ? dto.getEnterpriseCode() : currentOrgCode);
        //设置基本类型
        notification.setAssessmentType(dto.getAssessmentType());
        notification.setAssessmentMonth(dto.getAssessmentMonth());
        notification.setAssessmentYear(dto.getAssessmentYear());
        notification.setAssessmentQuater(dto.getAssessmentQuarter());
        notification.setValuationConfigurationName(dto.getValuationConfigurationName());
        notification.setLevel(dto.getLevel());
        // 设置供应商信息
        notification.setSupplierId(dto.getSupplierId());
        notification.setSupplierName(dto.getSupplierName());
        notification.setSupplierCode(dto.getSupplierCode());
        // 设置分发类型（默认值为1）
        notification.setDistributeType(distributeType != null ? distributeType : 1);
        // 设置基本通知信息
        notification.setPerformanceEvaluationId(dto.getPerformanceEvaluationId());
        notification.setProcessType(processType);
        notification.setNotificationTitle(dto.getNotificationTitle());
        notification.setNotificationContent(dto.getNotificationContent());
        notification.setNotifyDate(dto.getNotifyDate());
        notification.setNotificationStatus("1");
        // 根据流程类型设置额外信息
        if (isDeadlineProcess) {
            notification.setReplyDeadline(dto.getReplyDeadline());
            notification.setFinishDeadline(dto.getFinishDeadline());
            notification.setEnterpriseAddress(dto.getEnterpriseAddress());
            notification.setEnterpriseContact(dto.getEnterpriseContact());
            notification.setEnterpriseTel(dto.getEnterpriseTel());
        }
        super.saveOrUpdate(notification);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean confirmNotification(ProcessDTO dto) {
        // 参数校验
        validateParams(dto);
        // 获取用户信息
        IUser user = getUser();
        // 查询并验证通知
        BizSupplierPerformanceEvaluationNotification existingNotification = getAndValidateNotification(dto.getNotificationId());
        // 更新通知状态
        updateNotificationStatus(dto.getNotificationId());
        // 创建确认记录
        createAcknowledgementRecord(dto.getNotificationId(), user);
        // 处理纠正报告
        processCorrectionReport(dto, existingNotification);
        return true;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean updateReportStatus(ProcessDTO dto) {
        // 查询是否存在报告
        BizSupplierPerformanceEvaluationCorrectionReport report = reportManager.getById(dto.getReportId());
        if (report == null) {
          throw new ApplicationException("整改报告不存在");
        }

        try {
            // 更新报告状态
            report.setCorrectionStatus(dto.getStatus());
            // 根据状态处理不同业务逻辑
            if ("2".equals(dto.getStatus())) {
                // 状态2：更新结果描述、完成日期和附件
                report.setResultDescription(dto.getResultDescription());
                report.setActualCompleteDate(dto.getActualCompleteDate());

                // 更新附件
                if (dto.getFiles() != null && !dto.getFiles().isEmpty()) {
                    List<String> fileIds = dto.getFiles().stream()
                            .map(DefaultFile::getId)
                            .collect(Collectors.toList());
                    fileManager.updateFileBizIdByIds(fileIds, dto.getReportId());
                }


            } else if ("3".equals(dto.getStatus()) || "5".equals(dto.getStatus())) {
                // 状态3和5：更新通知状态为4
                updateNotificationStatus(report.getNotificationId(), "4");
            }
            // 更新报告并保存结果
            updateReportAndSaveResult(report,dto.getApprovalOpinion());
            return true;
        } catch (Exception e) {
            return false;
        }
    }

    /**
     * 更新通知状态
     */
    private void updateNotificationStatus(String notificationId, String status) {
        LambdaUpdateWrapper<BizSupplierPerformanceEvaluationNotification> wrapper = new LambdaUpdateWrapper<>();
        wrapper.eq(BizSupplierPerformanceEvaluationNotification::getId, notificationId)
                .set(BizSupplierPerformanceEvaluationNotification::getNotificationStatus, status);
        super.update(wrapper);
    }

    /**
     * 更新报告并保存结果
     */
    private void updateReportAndSaveResult(BizSupplierPerformanceEvaluationCorrectionReport report,String approvalOpinion) {
        reportManager.updateById(report);
        reportManager.saveReportResult(report,approvalOpinion);
    }

    @Override
    @Transactional(readOnly = true)
    public BizSupplierPerformanceEvaluationCorrectionReport getReport(String notificationId) {
        LambdaQueryWrapper<BizSupplierPerformanceEvaluationCorrectionReport> reportWrapper = new LambdaQueryWrapper<>();
        reportWrapper.eq(BizSupplierPerformanceEvaluationCorrectionReport::getNotificationId, notificationId);
        return reportManager.getOne(reportWrapper);
    }

    @Override
    @Transactional(readOnly = true)
    public BizSupplierPerformanceEvaluationNotification getById(Serializable id) {
        BizSupplierPerformanceEvaluationNotification entity = super.getById(id);
        if (entity == null) return null;
        BizSupplierPerformanceEvaluationCorrectionReport report = this.getReport(entity.getId());
        if (report != null) {
            entity.setReport(report);
            List<DefaultFile> files = fileManager.queryFileByBizId(report.getId());
            report.setFiles(files);
        }
        return entity;
    }

    @Override
    public PageList<BizSupplierPerformanceEvaluationNotification> queryEarlyEvaluationList(QueryFilter<BizSupplierPerformanceEvaluationNotification> queryFilter) {
        IPage<BizSupplierPerformanceEvaluationNotification> iPage = baseMapper.queryByEarly(convert2IPage(queryFilter.getPageBean()),
                convert2Wrapper(queryFilter, currentModelClass())
        );
        return new PageList<>(iPage);
    }

    /**
     * 参数校验
     */
    private void validateParams(ProcessDTO dto) {
        if (dto == null || StringUtils.isBlank(dto.getNotificationId())) {
            throw new ApplicationException("通知ID不能为空");
        }
    }

    /**
     * 获取当前用户
     */
    private IUser getUser() {
        return ContextUtil.getCurrentUser();
    }

    /**
     * 查询并验证通知
     */
    private BizSupplierPerformanceEvaluationNotification getAndValidateNotification(String notificationId) {
        BizSupplierPerformanceEvaluationNotification existingNotification = super.getById(notificationId);
        if (existingNotification == null) {
            throw new ApplicationException("未找到对应的考核通知");
        }

        // 检查通知状态，避免重复确认
        if ("2".equals(existingNotification.getNotificationStatus())) {
            throw new ApplicationException("该通知已被确认，请勿重复操作");
        }

        return existingNotification;
    }

    /**
     * 更新通知状态
     */
    private void updateNotificationStatus(String notificationId) {
        LambdaUpdateWrapper<BizSupplierPerformanceEvaluationNotification> notificationWrapper = new LambdaUpdateWrapper<>();
        notificationWrapper.eq(BizSupplierPerformanceEvaluationNotification::getId, notificationId)
                .set(BizSupplierPerformanceEvaluationNotification::getNotificationStatus, "2");

        boolean update = super.update(notificationWrapper);
        if (!update) {
            throw new ApplicationException("更新通知状态失败");
        }
    }

    /**
     * 创建确认记录
     */
    private void createAcknowledgementRecord(String notificationId, IUser user) {
        BizSupplierPerformanceEvaluationAcknowledgement acknowledgement = new BizSupplierPerformanceEvaluationAcknowledgement();
        acknowledgement.setNotificationId(notificationId);
        acknowledgement.setAckStatus("1");
        acknowledgement.setAckTime(LocalDateTime.now());
        acknowledgement.setAcknowledgedById(user.getUserId());
        acknowledgement.setAcknowledgedByAccount(user.getAccount());
        acknowledgement.setAcknowledgedByName(user.getFullname());
        acknowledgement.setAckMethod("1");

        acknowledgementManager.save(acknowledgement);
    }

    /**
     * 处理纠正报告
     */
    private void processCorrectionReport(ProcessDTO dto, BizSupplierPerformanceEvaluationNotification existingNotification) {
        if (!"2".equals(existingNotification.getProcessType())) {
            return;
        }

        BizSupplierPerformanceEvaluationCorrectionReport report = new BizSupplierPerformanceEvaluationCorrectionReport();
        report.setPerformanceEvaluationId(dto.getNotificationId());
        report.setNotificationId(dto.getNotificationId());
        report.setRootCauseAnalysis(dto.getRootCauseAnalysis());
        report.setTemporaryMeasure(dto.getTemporaryMeasure());
        report.setLongTermCorrection(dto.getLongTermCorrection());
        report.setPlanCompleteDate(dto.getPlanCompleteDate());
        report.setCorrectionStatus("1");

        reportManager.save(report);
    }
}
