package com.artfess.portal.persistence.manager.impl;

import com.artfess.base.conf.SaaSConfig;
import com.artfess.base.exception.BaseException;
import com.artfess.base.manager.impl.BaseManagerImpl;
import com.artfess.base.query.FieldRelation;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.query.QueryOP;
import com.artfess.base.util.AppUtil;
import com.artfess.base.util.BeanUtils;
import com.artfess.base.util.FileUtil;
import com.artfess.base.util.JsonUtil;
import com.artfess.base.util.UniqueIdUtil;
import com.artfess.portal.model.PortalNewsNotice;
import com.artfess.portal.model.PortalNewsTree;
import com.artfess.portal.persistence.dao.PortalNewsNoticeDao;
import com.artfess.portal.persistence.manager.PortalNewsNoticeManager;
import com.artfess.portal.persistence.manager.PortalNewsTreeManager;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.io.File;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 *  服务实现类
 *
 * @company 阿特菲斯信息技术有限公司
 * @author 赵祥云
 * @since 2020-07-03
 */
@Service
public class PortalNewsNoticeManagerImpl extends BaseManagerImpl<PortalNewsNoticeDao, PortalNewsNotice> implements PortalNewsNoticeManager {

	@Resource
	PortalNewsNoticeDao portalNewsNoticeDao;

	@Resource
	PortalNewsTreeManager portalNewsTreeManager;

	@Override
	public List<PortalNewsNotice> getNews(String classifyId) {
		QueryWrapper<PortalNewsNotice> queryWrapper = new QueryWrapper<>();
		LocalDateTime now = LocalDateTime.now();
		if (!classifyId.equals("all")){
			queryWrapper.eq("CLASSIFY_ID_", classifyId);
		}
//		queryWrapper.gt("END_TIME_", now);
		List<PortalNewsNotice> selectList = baseMapper.selectList(queryWrapper);
		List<PortalNewsNotice> respNotice = new ArrayList<>();
		selectList.forEach(notice -> {
			if (notice.getEndTime()!=null && notice.getEndTime().isBefore(now)){
				// 已过期，不予展示
			}else{
				respNotice.add(notice);
			}
		});
		return respNotice;
	}

	@Override
	@Transactional
	public Boolean insertNews(PortalNewsNotice news) {
		if (BeanUtils.isNotEmpty(news)){
			Map<String,Object> paramsMap = new HashMap<>();
			paramsMap.put("CLASSIFY_ID_",news.getClassifyId());
			news.setSn(this.getNextSequence(paramsMap));
			Integer result = this.baseMapper.insert(news);
			return null != result && result >= 1;
		}else{
			return false;
		}
	}

	@Override
	@Transactional
	public void deleteByClassifyId(String classifyId) {
		portalNewsNoticeDao.deleteByClassifyId(classifyId);
	}

	@Override
	@Transactional
	public void deleteNewsByParentId(String parentId) {
		portalNewsNoticeDao.deleteNewsByParentId(parentId);
	}

	@Override
	public String export(List<String> ids) throws Exception{
		if (BeanUtils.isEmpty(ids)) {
			throw new BaseException("传入的id集合为空");
		}
		List<PortalNewsNotice> portalNewsNotices = baseMapper.selectBatchIds(ids);
		if (BeanUtils.isEmpty(portalNewsNotices)) {
			throw new BaseException("根据id集合查找，未查到任何记录");
		}
		return JsonUtil.toJson(portalNewsNotices);
	}

	@Override
	@Transactional
	public void importFile(String uzPath, String classifyId) {
		try {
			String fileStr = FileUtil.readFile(uzPath + File.separator + "newsNotice.json");
			SaaSConfig saaSConfig = AppUtil.getBean(SaaSConfig.class);
			boolean enable = saaSConfig.isEnable();
			List list = JsonUtil.toBean(fileStr, List.class);
			PortalNewsTree portalNewsTree = portalNewsTreeManager.get(classifyId);
			if (BeanUtils.isEmpty(portalNewsTree)) {
				throw new BaseException("请先选择要导入的分类");
			}
			String treeType;
			if(portalNewsTree.getPath().startsWith("100")){
				treeType = "1";
			}else if(portalNewsTree.getPath().startsWith("200")){
				treeType = "2";
			}else{
				throw new BaseException("所属分类未知");
			}
			for (Object o : list) {
				PortalNewsNotice portalNewsNotice = JsonUtil.toBean(JsonUtil.toJson(o), PortalNewsNotice.class);
				if(treeType.equals(portalNewsNotice.getType().toString())){
					throw new BaseException("所属类型不匹配，无法导入");
				}
				portalNewsNotice.setClassifyPath(portalNewsTree.getPath());
				portalNewsNotice.setClassifyId(classifyId);
				portalNewsNotice.setClassifyName(portalNewsTree.getName());
				Map<String,Object> paramsMap = new HashMap<>();
				paramsMap.put("CLASSIFY_ID_",portalNewsNotice.getClassifyId());
				portalNewsNotice.setSn(this.getNextSequence(paramsMap));
				QueryFilter<PortalNewsNotice> filter = QueryFilter.build();
				filter.addFilter("CLASSIFY_ID_", portalNewsTree.getId(), QueryOP.EQUAL, FieldRelation.AND);
				filter.addFilter("ID_", portalNewsNotice.getId(), QueryOP.EQUAL, FieldRelation.AND);
				//如果当前分类下已存在，则更新
				PageList<PortalNewsNotice> old = this.query(filter);
				if (BeanUtils.isEmpty(old.getRows()) || old.getRows().isEmpty()) {
					portalNewsNotice.setId(UniqueIdUtil.getSuid());
					if(enable){
						//启用租户模式下
						portalNewsNotice.setTenantId(null);
					}
					this.create(portalNewsNotice);
				}else{
					this.updateById(portalNewsNotice);
				}
			}

		}catch (Exception e){
			throw new BaseException(e.getMessage());
		}
	}
}
