package com.artfess.gyl.utils;

import javax.crypto.Cipher;
import java.nio.charset.StandardCharsets;
import java.security.*;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.Base64;

/**
 * @Description:RSA加密工具类
 * @Author: Rong Tao
 * @Date: 2025/8/6 10:41
 */
public class RSAUtils {
    // 密钥长度，非对称加密一般使用2048位
    private static final int KEY_SIZE = 2048;
    // 算法名称
    private static final String ALGORITHM = "RSA";
    // 加密模式和填充方式
    private static final String TRANSFORMATION = "RSA/ECB/PKCS1Padding";

    /**
     * 生成RSA密钥对（公钥和私钥）
     * @return 包含公钥和私钥的对象数组，[0]为公钥，[1]为私钥
     * @throws NoSuchAlgorithmException 算法不存在异常
     */
    public static KeyPair generateKeyPair() throws NoSuchAlgorithmException {
        KeyPairGenerator keyPairGenerator = KeyPairGenerator.getInstance(ALGORITHM);
        keyPairGenerator.initialize(KEY_SIZE);
        return keyPairGenerator.generateKeyPair();
    }

    /**
     * 用公钥加密数据
     * @param publicKey 公钥（Base64编码字符串）
     * @param data 要加密的数据
     * @return 加密后的Base64编码字符串
     * @throws Exception 加密过程中的异常
     */
    public static String encrypt(String publicKey, String data) throws Exception {
        // 将Base64编码的公钥转换为PublicKey对象
        byte[] publicKeyBytes = Base64.getDecoder().decode(publicKey);
        X509EncodedKeySpec keySpec = new X509EncodedKeySpec(publicKeyBytes);
        KeyFactory keyFactory = KeyFactory.getInstance(ALGORITHM);
        PublicKey pubKey = keyFactory.generatePublic(keySpec);

        // 加密
        Cipher cipher = Cipher.getInstance(TRANSFORMATION);
        cipher.init(Cipher.ENCRYPT_MODE, pubKey);
        byte[] encryptedData = cipher.doFinal(data.getBytes(StandardCharsets.UTF_8));

        // 返回Base64编码的加密结果
        return Base64.getEncoder().encodeToString(encryptedData);
    }

    /**
     * 用私钥解密数据
     * @param privateKey 私钥（Base64编码字符串）
     * @param encryptedData 加密后的Base64编码字符串
     * @return 解密后的原始数据
     * @throws Exception 解密过程中的异常
     */
    public static String decrypt(String privateKey, String encryptedData) throws Exception {
        // 将Base64编码的私钥转换为PrivateKey对象
        byte[] privateKeyBytes = Base64.getDecoder().decode(privateKey);
        PKCS8EncodedKeySpec keySpec = new PKCS8EncodedKeySpec(privateKeyBytes);
        KeyFactory keyFactory = KeyFactory.getInstance(ALGORITHM);
        PrivateKey priKey = keyFactory.generatePrivate(keySpec);

        // 解密
        Cipher cipher = Cipher.getInstance(TRANSFORMATION);
        cipher.init(Cipher.DECRYPT_MODE, priKey);
        byte[] decryptedData = cipher.doFinal(Base64.getDecoder().decode(encryptedData));

        // 返回解密后的字符串
        return new String(decryptedData, StandardCharsets.UTF_8);
    }
}
