package com.artfess.gyl.supplierEva.manager.impl;

import com.artfess.base.exception.ApplicationException;
import com.artfess.file.model.DefaultFile;
import com.artfess.file.persistence.manager.FileManager;
import com.artfess.gyl.supplierEva.dto.ProcessDTO;
import com.artfess.gyl.supplierEva.dto.ResultEvaDTO;
import com.artfess.gyl.supplierEva.manager.BizSupplierPerformanceEvaluationAcknowledgementManager;
import com.artfess.gyl.supplierEva.manager.BizSupplierPerformanceEvaluationCorrectionReportManager;
import com.artfess.gyl.supplierEva.model.BizSupplierPerformanceEvaluationAcknowledgement;
import com.artfess.gyl.supplierEva.model.BizSupplierPerformanceEvaluationCorrectionReport;
import com.artfess.gyl.supplierEva.model.BizSupplierPerformanceEvaluationNotification;
import com.artfess.gyl.supplierEva.dao.BizSupplierPerformanceEvaluationNotificationDao;
import com.artfess.gyl.supplierEva.manager.BizSupplierPerformanceEvaluationNotificationManager;
import com.artfess.base.manager.impl.BaseManagerImpl;
import com.artfess.uc.api.impl.util.ContextUtil;
import com.artfess.uc.api.model.IUser;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.awt.geom.RectangularShape;
import java.time.LocalDateTime;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 供应商绩效考核通知信息表 服务实现类
 *
 * @author huqi
 * @since 2025-11-12
 */
@Service
public class BizSupplierPerformanceEvaluationNotificationManagerImpl extends BaseManagerImpl<BizSupplierPerformanceEvaluationNotificationDao, BizSupplierPerformanceEvaluationNotification> implements BizSupplierPerformanceEvaluationNotificationManager {

    @Resource
    private BizSupplierPerformanceEvaluationAcknowledgementManager acknowledgementManager;

    @Resource
    private BizSupplierPerformanceEvaluationCorrectionReportManager reportManager;

    @Resource
    private FileManager fileManager;

    @Override
    public void saveEva(ResultEvaDTO dto) {
        String processType = dto.getProcessType();
        if (!processType.equals("0")){
            BizSupplierPerformanceEvaluationNotification notification = new BizSupplierPerformanceEvaluationNotification();
            String id = dto.getId();
            if (StringUtils.isNotBlank(id)){
                notification.setId(id);
            }
            boolean isTrue = processType.equals("2");
            notification.setPerformanceEvaluationId(dto.getPerformanceEvaluationId());
            notification.setProcessType(processType);
            notification.setNotificationTitle(dto.getNotificationTitle());
            notification.setNotificationContent(dto.getNotificationContent());
            notification.setNotifyDate(dto.getNotifyDate());
            notification.setReplyDeadline(isTrue?dto.getReplyDeadline():null);
            notification.setFinishDeadline(isTrue?dto.getFinishDeadline():null);
            notification.setEnterpriseAddress(isTrue?dto.getEnterpriseAddress():null);
            notification.setEnterpriseContact(isTrue?dto.getEnterpriseContact():null);
            notification.setEnterpriseTel(isTrue?dto.getEnterpriseTel():null);
            notification.setNotificationStatus("1");
            super.saveOrUpdate(notification);
        }

//        //当处理类型为限期整改时
//        if (processType.equals("2")){
//            BizSupplierPerformanceEvaluationCorrectionReport report = new BizSupplierPerformanceEvaluationCorrectionReport();
//            report.setPerformanceEvaluationId(dto.getId());
//            report.setNotificationId(notification.getId());
//        }
//        BizSupplierPerformanceEvaluationAcknowledgement acknowledgement = new BizSupplierPerformanceEvaluationAcknowledgement();
//        acknowledgement.set
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean confirmNotification(ProcessDTO dto) {
        // 参数校验
        validateParams(dto);
        // 获取用户信息
        IUser user = getUser();
        // 查询并验证通知
        BizSupplierPerformanceEvaluationNotification existingNotification = getAndValidateNotification(dto.getNotificationId());
        // 更新通知状态
        updateNotificationStatus(dto.getNotificationId());
        // 创建确认记录
        createAcknowledgementRecord(dto.getNotificationId(), user);
        // 处理纠正报告
        processCorrectionReport(dto, existingNotification);
        return true;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean updateReportStatus(ProcessDTO dto) {
        // 查询是否存在报告
        BizSupplierPerformanceEvaluationCorrectionReport report = reportManager.getById(dto.getReportId());
        if (report == null) {
          throw new ApplicationException("整改报告不存在");
        }

        try {
            // 更新报告状态
            report.setCorrectionStatus(dto.getStatus());
            // 根据状态处理不同业务逻辑
            if ("2".equals(dto.getStatus())) {
                // 状态2：更新结果描述、完成日期和附件
                report.setResultDescription(dto.getResultDescription());
                report.setActualCompleteDate(dto.getActualCompleteDate());

                // 更新附件
                List<String> fileIds = dto.getFiles().stream()
                        .map(DefaultFile::getId)
                        .collect(Collectors.toList());
                fileManager.updateFileBizIdByIds(fileIds, dto.getReportId());

            } else if ("3".equals(dto.getStatus()) || "5".equals(dto.getStatus())) {
                // 状态3和5：更新通知状态为4
                updateNotificationStatus(report.getNotificationId(), "4");
            }
            // 更新报告并保存结果
            updateReportAndSaveResult(report,dto.getApprovalOpinion());
            return true;
        } catch (Exception e) {
            return false;
        }
    }

    /**
     * 更新通知状态
     */
    private void updateNotificationStatus(String notificationId, String status) {
        LambdaUpdateWrapper<BizSupplierPerformanceEvaluationNotification> wrapper = new LambdaUpdateWrapper<>();
        wrapper.eq(BizSupplierPerformanceEvaluationNotification::getId, notificationId)
                .set(BizSupplierPerformanceEvaluationNotification::getNotificationStatus, status);
        super.update(wrapper);
    }

    /**
     * 更新报告并保存结果
     */
    private void updateReportAndSaveResult(BizSupplierPerformanceEvaluationCorrectionReport report,String approvalOpinion) {
        reportManager.updateById(report);
        reportManager.saveReportResult(report,approvalOpinion);
    }

    @Override
    public BizSupplierPerformanceEvaluationCorrectionReport getReport(String notificationId) {
        LambdaQueryWrapper<BizSupplierPerformanceEvaluationCorrectionReport> reportWrapper = new LambdaQueryWrapper<>();
        reportWrapper.eq(BizSupplierPerformanceEvaluationCorrectionReport::getNotificationId, notificationId);
        return reportManager.getOne(reportWrapper);
    }

    /**
     * 参数校验
     */
    private void validateParams(ProcessDTO dto) {
        if (dto == null || StringUtils.isBlank(dto.getNotificationId())) {
            throw new ApplicationException("通知ID不能为空");
        }
    }

    /**
     * 获取当前用户
     */
    private IUser getUser() {
        return ContextUtil.getCurrentUser();
    }

    /**
     * 查询并验证通知
     */
    private BizSupplierPerformanceEvaluationNotification getAndValidateNotification(String notificationId) {
        BizSupplierPerformanceEvaluationNotification existingNotification = super.getById(notificationId);
        if (existingNotification == null) {
            throw new ApplicationException("未找到对应的考核通知");
        }

        // 检查通知状态，避免重复确认
        if ("2".equals(existingNotification.getNotificationStatus())) {
            throw new ApplicationException("该通知已被确认，请勿重复操作");
        }

        return existingNotification;
    }

    /**
     * 更新通知状态
     */
    private void updateNotificationStatus(String notificationId) {
        LambdaUpdateWrapper<BizSupplierPerformanceEvaluationNotification> notificationWrapper = new LambdaUpdateWrapper<>();
        notificationWrapper.eq(BizSupplierPerformanceEvaluationNotification::getId, notificationId)
                .set(BizSupplierPerformanceEvaluationNotification::getNotificationStatus, "2");

        boolean update = super.update(notificationWrapper);
        if (!update) {
            throw new ApplicationException("更新通知状态失败");
        }
    }

    /**
     * 创建确认记录
     */
    private void createAcknowledgementRecord(String notificationId, IUser user) {
        BizSupplierPerformanceEvaluationAcknowledgement acknowledgement = new BizSupplierPerformanceEvaluationAcknowledgement();
        acknowledgement.setNotificationId(notificationId);
        acknowledgement.setAckStatus("1");
        acknowledgement.setAckTime(LocalDateTime.now());
        acknowledgement.setAcknowledgedById(user.getUserId());
        acknowledgement.setAcknowledgedByAccount(user.getAccount());
        acknowledgement.setAcknowledgedByName(user.getFullname());
        acknowledgement.setAckMethod("1");

        acknowledgementManager.save(acknowledgement);
    }

    /**
     * 处理纠正报告
     */
    private void processCorrectionReport(ProcessDTO dto, BizSupplierPerformanceEvaluationNotification existingNotification) {
        if (!"2".equals(existingNotification.getProcessType())) {
            return;
        }

        BizSupplierPerformanceEvaluationCorrectionReport report = new BizSupplierPerformanceEvaluationCorrectionReport();
        report.setPerformanceEvaluationId(dto.getNotificationId());
        report.setNotificationId(dto.getNotificationId());
        report.setRootCauseAnalysis(dto.getRootCauseAnalysis());
        report.setTemporaryMeasure(dto.getTemporaryMeasure());
        report.setLongTermCorrection(dto.getLongTermCorrection());
        report.setPlanCompleteDate(dto.getPlanCompleteDate());
        report.setCorrectionStatus("1");

        reportManager.save(report);
    }
}
