package com.artfess.gyl.supplierEva.manager.impl;

import com.artfess.base.exception.ApplicationException;
import com.artfess.base.manager.impl.BaseManagerImpl;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.file.model.DefaultFile;
import com.artfess.file.persistence.manager.FileManager;
import com.artfess.gyl.supplierEva.dao.BizSupplierPerformanceEvaluationDao;
import com.artfess.gyl.supplierEva.dto.EvaluationDTO;
import com.artfess.gyl.supplierEva.dto.ResultEvaDTO;
import com.artfess.gyl.supplierEva.manager.*;
import com.artfess.gyl.supplierEva.model.*;
import com.artfess.gyl.supplierEva.vo.*;
import com.artfess.uc.api.impl.util.ContextUtil;
import com.artfess.uc.api.model.IUser;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.http.util.Asserts;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 供应商绩效考核评价表 服务实现类
 *
 * @author huqi
 * @since 2025-11-12
 */
@Service
public class BizSupplierPerformanceEvaluationManagerImpl extends BaseManagerImpl<BizSupplierPerformanceEvaluationDao, BizSupplierPerformanceEvaluation> implements BizSupplierPerformanceEvaluationManager {

    // 依赖注入
    @Resource
    private BizSupplierPerformanceEvaluationRatingManager ratingManager;
    @Resource
    private BizSupplierPerformanceEvaluationResultManager resultManager;
    @Resource
    private BizSupplierPerformanceEvaluationRatingUserManager ratingUserManager;
    @Resource
    private BizSupplierPerformanceEvaluationIntervalManager intervalManager;
    @Resource
    private BizSupplierPerformanceEvaluationNotificationManager notificationManager;
    @Resource
    private BizSupplierPerformanceEvaluationApprovalRecordManager recordManager;
    @Resource
    private BizSupplierPerformanceEvaluationConfigurationUserManager supplierManager;
    @Resource
    private FileManager fileManager;

    // 定义常量
    private static final List<String> COMPLETED_LEVELS = Arrays.asList("1", "2", "3");
    private static final String STATUS_INIT = "0";
    private static final String STATUS_COMPLETED = "1";
    private static final String STATUS_SUBMITTED = "2";
    private static final String STATUS_PARTIAL_COMPLETED = "3";
    private static final String STATUS_ALL_COMPLETED = "4";

    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean batchCreateEvaluationFromTemplate(EvaluationDTO dto) {
        IUser user = getCurrentUser();
        validateEvaluationData(dto);

        List<BizSupplierPerformanceEvaluationConfigurationUser> supplierEnterprise = getSupplierEnterprise(dto);
        if (CollectionUtils.isEmpty(supplierEnterprise)) {
            throw new ApplicationException("无合格供应商进行分发");
        }

        initEvaluationData(dto, user);
        boolean saveResult = super.save(dto);
        if (!saveResult) {
            return false;
        }

        saveEvaluationResults(dto, supplierEnterprise);
        return true;
    }

    /**
     * 获取供应商企业列表
     */
    private List<BizSupplierPerformanceEvaluationConfigurationUser> getSupplierEnterprise(EvaluationDTO dto) {
        LambdaQueryWrapper<BizSupplierPerformanceEvaluationConfigurationUser> configurationWrapper =
                new LambdaQueryWrapper<>();
        configurationWrapper.eq(BizSupplierPerformanceEvaluationConfigurationUser::getValuationConfigurationId,
                dto.getValuationConfigurationId());
        return supplierManager.list(configurationWrapper);
    }

    /**
     * 保存评估结果
     */
    private void saveEvaluationResults(EvaluationDTO dto,
                                       List<BizSupplierPerformanceEvaluationConfigurationUser> supplierEnterprise) {
        int totalScore = calculateTotalScore(dto.getDetails());
        IUser user = ContextUtil.getCurrentUser();
        List<BizSupplierPerformanceEvaluationResult> results = createEvaluationResults(dto, supplierEnterprise, totalScore);

        resultManager.saveBatch(results);

        // 为每个供应商保存评分明细
        results.forEach(result ->{
            saveRatingDetailsAndUsers(dto.getId(), result.getId(), dto.getDetails());
//            createApprovalRecord(result.getId(), user, STATUS_SUBMITTED, "1");
         }
        );
    }

    /**
     * 计算总分
     */
    private int calculateTotalScore(List<BizSupplierPerformanceEvaluationRating> details) {
        return details.stream()
                .mapToInt(BizSupplierPerformanceEvaluationRating::getScore)
                .sum();
    }

    /**
     * 创建评估结果列表
     */
    private List<BizSupplierPerformanceEvaluationResult> createEvaluationResults(
            EvaluationDTO dto,
            List<BizSupplierPerformanceEvaluationConfigurationUser> supplierEnterprise,
            int totalScore) {

        return supplierEnterprise.stream()
                .map(item -> createEvaluationResult(dto, item, totalScore))
                .collect(Collectors.toList());
    }

    /**
     * 创建单个评估结果
     */
    private BizSupplierPerformanceEvaluationResult createEvaluationResult(
            EvaluationDTO dto,
            BizSupplierPerformanceEvaluationConfigurationUser item,
            int totalScore) {

        BizSupplierPerformanceEvaluationResult result = new BizSupplierPerformanceEvaluationResult();
        result.setValuationConfigurationId(dto.getValuationConfigurationId());
        result.setEvaluationId(dto.getId());
        result.setSupplierId(item.getSupplierId());
        result.setSupplierName(item.getSupplierName());
        result.setSupplierCode(item.getSupplierCode());
        result.setTotalScore(totalScore);
        result.setStatus(STATUS_SUBMITTED);
        return result;
    }

    /**
     * 获取当前用户
     */
    private IUser getCurrentUser() {
        IUser user = ContextUtil.getCurrentUser();
        if (user == null) {
            throw new ApplicationException("未找到用户信息");
        }
        return user;
    }

    /**
     * 验证评估数据
     */
    private void validateEvaluationData(EvaluationDTO dto) {
        Asserts.notNull(dto, "考核数据不能为空");
        Asserts.notNull(dto.getDetails(), "考核详情不能为空");
    }

    /**
     * 初始化评估数据
     */
    private void initEvaluationData(EvaluationDTO dto, IUser user) {
        dto.setInitiatorId(user.getUserId());
        dto.setInitiatorAccount(user.getAccount());
        dto.setInitiatorName(user.getFullname());
        dto.setStatus(STATUS_SUBMITTED);
        dto.setAssessmentStartDate(LocalDate.now());
    }

    /**
     * 保存评分明细和用户
     */
    private void saveRatingDetailsAndUsers(String evaluationId, String resultId,
                                           List<BizSupplierPerformanceEvaluationRating> details) {
        List<BizSupplierPerformanceEvaluationRating> savedDetails = saveRatingDetails(evaluationId, resultId, details);
        saveRatingUsers(evaluationId, savedDetails);
    }

    /**
     * 保存评分明细
     */
    private List<BizSupplierPerformanceEvaluationRating> saveRatingDetails(String evaluationId, String resultId,
                                                                           List<BizSupplierPerformanceEvaluationRating> details) {
        details.forEach(item -> {
            item.setId(null);
            item.setPerformanceEvaluationId(evaluationId);
            item.setPerformanceEvaluationResultId(resultId);
            item.setStatus(STATUS_INIT);
        });

        if (!ratingManager.saveBatch(details)) {
            throw new ApplicationException("评分明细保存失败");
        }

        return details;
    }

    /**
     * 保存评分用户
     */
    private void saveRatingUsers(String evaluationId, List<BizSupplierPerformanceEvaluationRating> savedDetails) {
        List<BizSupplierPerformanceEvaluationRatingUser> userList = savedDetails.stream()
                .filter(item -> CollectionUtils.isNotEmpty(item.getUsers()))
                .flatMap(item -> createRatingUsers(evaluationId, item))
                .collect(Collectors.toList());

        if (CollectionUtils.isNotEmpty(userList) && !ratingUserManager.saveBatch(userList)) {
            throw new ApplicationException("评分用户保存失败");
        }
    }

    /**
     * 创建评分用户列表
     */
    private java.util.stream.Stream<BizSupplierPerformanceEvaluationRatingUser> createRatingUsers(
            String evaluationId, BizSupplierPerformanceEvaluationRating item) {

        return item.getUsers().stream()
                .map(sourceUser -> createRatingUser(evaluationId, item, sourceUser));
    }

    /**
     * 创建单个评分用户
     */
    private BizSupplierPerformanceEvaluationRatingUser createRatingUser(
            String evaluationId,
            BizSupplierPerformanceEvaluationRating item,
            BizSupplierPerformanceEvaluationRatingUser sourceUser) {

        BizSupplierPerformanceEvaluationRatingUser user = new BizSupplierPerformanceEvaluationRatingUser();
        user.setEvaluationRatingId(item.getId());
        user.setPerformanceEvaluationId(evaluationId);
        user.setRatingId(sourceUser.getRatingId());
        user.setRatingAccount(sourceUser.getRatingAccount());
        user.setRatingName(sourceUser.getRatingName());
        user.setScore(item.getScore());
        user.setStatus(STATUS_INIT);
        return user;
    }

    @Override
    @Transactional(readOnly = true)
    public EvaluationResultVo getModelById(String id) {
        EvaluationResultVo vo = resultManager.getModel(id);
        if (vo == null) {
            return null;
        }

        loadRelatedData(id, vo);
        return vo;
    }

    /**
     * 加载相关数据
     */
    private void loadRelatedData(String resultId, EvaluationResultVo vo) {
        CompletableFuture<List<BizSupplierPerformanceEvaluationRating>> ratingsFuture =
                CompletableFuture.supplyAsync(() -> loadRatingDetails(resultId));

        CompletableFuture<List<BizSupplierPerformanceEvaluationNotification>> notificationsFuture =
                CompletableFuture.supplyAsync(() -> loadNotifications(resultId));

        CompletableFuture<List<BizSupplierPerformanceEvaluationApprovalRecord>> recordsFuture =
                CompletableFuture.supplyAsync(() -> getRecords(resultId));

        try {
            setRatingDetails(vo, ratingsFuture.get());
            setNotificationInfo(vo, notificationsFuture.get());
            setApprovalRecords(vo, recordsFuture.get());
        } catch (Exception e) {
            throw new ApplicationException("加载评估数据失败", e);
        }
    }

    /**
     * 设置评分明细
     */
    private void setRatingDetails(EvaluationResultVo vo, List<BizSupplierPerformanceEvaluationRating> ratings) {
        if (CollectionUtils.isNotEmpty(ratings)) {
            loadRatingUsers(ratings);
            vo.setDetails(ratings);
        }
    }

    /**
     * 设置通知信息
     */
    private void setNotificationInfo(EvaluationResultVo vo,
                                     List<BizSupplierPerformanceEvaluationNotification> notifications) {
        if (CollectionUtils.isNotEmpty(notifications)) {
            vo.setNotification(notifications.get(0));
            loadCorrectionReportIfNeeded(vo, notifications.get(0));
        }
    }

    /**
     * 设置审批记录
     */
    private void setApprovalRecords(EvaluationResultVo vo,
                                    List<BizSupplierPerformanceEvaluationApprovalRecord> records) {
        if (CollectionUtils.isNotEmpty(records)) {
            vo.setRecords(records);
        }
    }

    /**
     * 加载评分明细
     */
    private List<BizSupplierPerformanceEvaluationRating> loadRatingDetails(String resultId) {
        return ratingManager.list(new LambdaQueryWrapper<BizSupplierPerformanceEvaluationRating>()
                .eq(BizSupplierPerformanceEvaluationRating::getPerformanceEvaluationResultId, resultId));
    }

    /**
     * 加载评分用户
     */
    private void loadRatingUsers(List<BizSupplierPerformanceEvaluationRating> ratings) {
        List<String> ratingIds = ratings.stream()
                .map(BizSupplierPerformanceEvaluationRating::getId)
                .collect(Collectors.toList());

        if (CollectionUtils.isEmpty(ratingIds)) {
            return;
        }

        List<BizSupplierPerformanceEvaluationRatingUser> ratingUsers = ratingUserManager.list(
                new LambdaQueryWrapper<BizSupplierPerformanceEvaluationRatingUser>()
                        .in(BizSupplierPerformanceEvaluationRatingUser::getEvaluationRatingId, ratingIds));

        if (CollectionUtils.isNotEmpty(ratingUsers)) {
            Map<String, List<BizSupplierPerformanceEvaluationRatingUser>> userMap = ratingUsers.stream()
                    .collect(Collectors.groupingBy(BizSupplierPerformanceEvaluationRatingUser::getEvaluationRatingId));

            ratings.forEach(rating -> rating.setUsers(userMap.get(rating.getId())));
        }
    }

    /**
     * 加载通知信息
     */
    private List<BizSupplierPerformanceEvaluationNotification> loadNotifications(String performanceEvaluationId) {
        return notificationManager.list(new LambdaQueryWrapper<BizSupplierPerformanceEvaluationNotification>()
                .eq(BizSupplierPerformanceEvaluationNotification::getPerformanceEvaluationId, performanceEvaluationId));
    }

    /**
     * 加载整改报告（如果需要）
     */
    private void loadCorrectionReportIfNeeded(EvaluationResultVo vo,
                                              BizSupplierPerformanceEvaluationNotification notification) {
        if (vo.getProcessType() != null && "2".equals(vo.getProcessType())) {
            BizSupplierPerformanceEvaluationCorrectionReport report = notificationManager.getReport(notification.getId());
            if (report != null) {
                vo.setReport(report);
                loadReportFiles(report);
            }
        }
    }

    /**
     * 加载报告文件
     */
    private void loadReportFiles(BizSupplierPerformanceEvaluationCorrectionReport report) {
        List<DefaultFile> files = fileManager.queryFileByBizId(report.getId());
        report.setFiles(files);
    }

    @Override
    @Transactional(readOnly = true)
    public EvaluationOptVo getModelByUser(String id) {
        BizSupplierPerformanceEvaluation evaluation = this.get(id);
        if (evaluation == null) {
            return null;
        }

        EvaluationOptVo vo = new EvaluationOptVo(evaluation);
        IUser user = getCurrentUser();
        List<BizSupplierPerformanceEvaluationRating> ratings = ratingManager.getByUser(user.getAccount(), id);
        vo.setDetails(ratings);
        return vo;
    }

    @Override
    @Transactional(readOnly = true)
    public List<SupplierScore> getUserRatings(String evaluationId, String detailsId) {
        IUser user = ContextUtil.getCurrentUser();
        return baseMapper.getSupplierScores(user.getAccount(), evaluationId, detailsId);
    }

    @Override
    public List<SupplierScore> getUserEvaluations(String ratingId) {
        return baseMapper.getUserEvaluations(ratingId);
    }

    @Override
    public PageList<EvaluationVo> queryEvaluationList(QueryFilter<BizSupplierPerformanceEvaluation> queryFilter) {
        IPage<EvaluationVo> iPage = baseMapper.queryByEva(
                convert2IPage(queryFilter.getPageBean()),
                convert2Wrapper(queryFilter, currentModelClass()));
        return new PageList<>(iPage);
    }

    @Override
    public PageList<EvaluationVo> queryEvaluationListByUser(QueryFilter<BizSupplierPerformanceEvaluation> queryFilter, String account) {
        IPage<EvaluationVo> iPage = baseMapper.queryByUser(
                convert2IPage(queryFilter.getPageBean()),
                convert2Wrapper(queryFilter, currentModelClass()),
                account
        );
        return new PageList<>(iPage);
    }

    @Override
    public PageList<EvaluationVo> queryEarlyEvaluationList(QueryFilter<BizSupplierPerformanceEvaluation> queryFilter) {
        IPage<EvaluationVo> iPage = baseMapper.queryByEarly(
                convert2IPage(queryFilter.getPageBean()),
                convert2Wrapper(queryFilter, currentModelClass())
        );
        return new PageList<>(iPage);
    }

    @Override
    public PageList<CorePerformanceVo> queryBySupplierLevel(QueryFilter<BizSupplierPerformanceEvaluation> queryFilter) {
        IPage<CorePerformanceVo> iPage = baseMapper.countSupplierLevel(
                convert2IPage(queryFilter.getPageBean()),
                convert2Wrapper(queryFilter, currentModelClass())
        );
        return new PageList<>(iPage);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean submitUserRating(List<BizSupplierPerformanceEvaluationRatingUser> ratingList) {
        if (CollectionUtils.isEmpty(ratingList)) {
            return false;
        }

        LocalDateTime now = LocalDateTime.now();
        IUser user = getCurrentUser();
        updateRatingUsers(ratingList, now);
        processCompletedRatings(ratingList, now, user);
        return true;
    }

    /**
     * 更新评分用户记录
     */
    private void updateRatingUsers(List<BizSupplierPerformanceEvaluationRatingUser> ratingList, LocalDateTime now) {
        ratingList.forEach(item -> {
            item.setStatus(STATUS_COMPLETED);
            item.setRatingTime(now);
        });

        if (!ratingUserManager.updateBatchById(ratingList)) {
            throw new ApplicationException("评分用户更新失败");
        }
    }

    /**
     * 处理完成的评分项
     */
    private void processCompletedRatings(List<BizSupplierPerformanceEvaluationRatingUser> ratingList,
                                         LocalDateTime now, IUser user) {
        List<String> evaluationRatingIds = ratingList.stream()
                .map(BizSupplierPerformanceEvaluationRatingUser::getEvaluationRatingId)
                .distinct()
                .collect(Collectors.toList());

        Map<String, List<BizSupplierPerformanceEvaluationRatingUser>> allRatingsMap =
                getRatingUsersMap(evaluationRatingIds);
        Map<String, BizSupplierPerformanceEvaluationRating> ratingItemMap =
                getRatingItemsMap(evaluationRatingIds);

        List<BizSupplierPerformanceEvaluationRating> ratingsToUpdate = processCompletedRatingGroups(
                evaluationRatingIds, allRatingsMap, ratingItemMap, now);

        batchSaveUpdates(ratingsToUpdate);
    }

    /**
     * 处理完成的评分组
     */
    private List<BizSupplierPerformanceEvaluationRating> processCompletedRatingGroups(
            List<String> evaluationRatingIds,
            Map<String, List<BizSupplierPerformanceEvaluationRatingUser>> allRatingsMap,
            Map<String, BizSupplierPerformanceEvaluationRating> ratingItemMap,
            LocalDateTime now) {

        return evaluationRatingIds.stream()
                .map(evaluationRatingId -> processRatingGroup(evaluationRatingId, allRatingsMap, ratingItemMap, now))
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
    }

    /**
     * 处理单个评分组
     */
    private BizSupplierPerformanceEvaluationRating processRatingGroup(
            String evaluationRatingId,
            Map<String, List<BizSupplierPerformanceEvaluationRatingUser>> allRatingsMap,
            Map<String, BizSupplierPerformanceEvaluationRating> ratingItemMap,
            LocalDateTime now) {

        List<BizSupplierPerformanceEvaluationRatingUser> ratings = allRatingsMap.get(evaluationRatingId);
        BizSupplierPerformanceEvaluationRating ratingItem = ratingItemMap.get(evaluationRatingId);

        if (ratings == null || ratingItem == null || !isRatingGroupCompleted(ratings)) {
            return null;
        }

        updateRatingItem(ratingItem, ratings, now);
        return ratingItem;
    }

    /**
     * 获取评分用户映射
     */
    private Map<String, List<BizSupplierPerformanceEvaluationRatingUser>> getRatingUsersMap(List<String> evaluationRatingIds) {
        return ratingUserManager.list(
                new LambdaQueryWrapper<BizSupplierPerformanceEvaluationRatingUser>()
                        .in(BizSupplierPerformanceEvaluationRatingUser::getEvaluationRatingId, evaluationRatingIds)
        ).stream().collect(Collectors.groupingBy(BizSupplierPerformanceEvaluationRatingUser::getEvaluationRatingId));
    }

    /**
     * 获取评分项映射
     */
    private Map<String, BizSupplierPerformanceEvaluationRating> getRatingItemsMap(List<String> evaluationRatingIds) {
        return ratingManager.listByIds(evaluationRatingIds).stream()
                .collect(Collectors.toMap(BizSupplierPerformanceEvaluationRating::getId, Function.identity()));
    }

    /**
     * 检查评分组是否完成
     */
    private boolean isRatingGroupCompleted(List<BizSupplierPerformanceEvaluationRatingUser> ratings) {
        return ratings.stream().allMatch(rating -> STATUS_COMPLETED.equals(rating.getStatus()));
    }

    /**
     * 更新评分项
     */
    private void updateRatingItem(BizSupplierPerformanceEvaluationRating ratingItem,
                                  List<BizSupplierPerformanceEvaluationRatingUser> ratings,
                                  LocalDateTime now) {
        BigDecimal averageScore = calculateAverageScore(ratings);
        ratingItem.setStatus(STATUS_COMPLETED);
        ratingItem.setRatingTime(now);
        ratingItem.setOperationScore(averageScore);
    }

    /**
     * 批量保存更新
     */
    private void batchSaveUpdates(List<BizSupplierPerformanceEvaluationRating> ratingsToUpdate) {
        if (CollectionUtils.isNotEmpty(ratingsToUpdate)) {
            ratingManager.updateBatchById(ratingsToUpdate);
            updatePerformanceEvaluationsForItems(ratingsToUpdate);
        }
    }

    /**
     * 为评分项更新绩效评估
     */
    private void updatePerformanceEvaluationsForItems(List<BizSupplierPerformanceEvaluationRating> ratingsToUpdate) {
        ratingsToUpdate.forEach(item ->
                updatePerformanceEvaluations(item.getPerformanceEvaluationResultId())
        );
    }

    /**
     * 批量更新绩效评估
     */
    private void updatePerformanceEvaluations(String resultId) {
        BizSupplierPerformanceEvaluationResult result = resultManager.getById(resultId);
        if (!Objects.equals(result.getStatus(), STATUS_SUBMITTED) &&
                !Objects.equals(result.getStatus(), STATUS_PARTIAL_COMPLETED)) {
            throw new ApplicationException("已完成评分操作，请刷新页面后进行查看");
        }

        String finalStatus = determineFinalStatus(resultId);
        result.setStatus(STATUS_PARTIAL_COMPLETED);

        if (STATUS_ALL_COMPLETED.equals(finalStatus)) {
            updateCompletedEvaluation(result);
        }

        resultManager.updateById(result);
    }

    /**
     * 确定最终状态
     */
    private String determineFinalStatus(String resultId) {
        List<BizSupplierPerformanceEvaluationRating> unfinishedRatings = getUnfinishedRatings(resultId);
        return unfinishedRatings.isEmpty() ? STATUS_ALL_COMPLETED : STATUS_PARTIAL_COMPLETED;
    }

    /**
     * 更新已完成的绩效评估
     */
    private void updateCompletedEvaluation(BizSupplierPerformanceEvaluationResult evaluation) {
        List<BizSupplierPerformanceEvaluationRating> allRatings = ratingManager.list(
                new LambdaQueryWrapper<BizSupplierPerformanceEvaluationRating>()
                        .eq(BizSupplierPerformanceEvaluationRating::getPerformanceEvaluationResultId, evaluation.getId()));

        BigDecimal totalScore = calculateTotalOperationScore(allRatings);
        String level = calculateLevel(evaluation.getValuationConfigurationId(), totalScore);
        evaluation.setStatus(COMPLETED_LEVELS.contains(level)?STATUS_COMPLETED:STATUS_ALL_COMPLETED);
        evaluation.setOperationScore(totalScore);
        evaluation.setLevel(level);
    }

    /**
     * 计算总操作分数
     */
    private BigDecimal calculateTotalOperationScore(List<BizSupplierPerformanceEvaluationRating> allRatings) {
        return allRatings.stream()
                .map(BizSupplierPerformanceEvaluationRating::getOperationScore)
                .filter(Objects::nonNull)
                .reduce(BigDecimal.ZERO, BigDecimal::add);
    }

    /**
     * 查询未完成的评分项
     */
    private List<BizSupplierPerformanceEvaluationRating> getUnfinishedRatings(String resultId) {
        return ratingManager.list(new LambdaQueryWrapper<BizSupplierPerformanceEvaluationRating>()
                .eq(BizSupplierPerformanceEvaluationRating::getPerformanceEvaluationResultId, resultId)
                .ne(BizSupplierPerformanceEvaluationRating::getStatus, STATUS_COMPLETED));
    }

    /**
     * 计算平均分
     */
    private BigDecimal calculateAverageScore(List<BizSupplierPerformanceEvaluationRatingUser> ratings) {
        if (CollectionUtils.isEmpty(ratings)) {
            return BigDecimal.ZERO;
        }

        List<BigDecimal> validScores = ratings.stream()
                .map(BizSupplierPerformanceEvaluationRatingUser::getOperationScore)
                .filter(Objects::nonNull)
                .collect(Collectors.toList());

        if (validScores.isEmpty()) {
            return BigDecimal.ZERO;
        }

        BigDecimal sum = validScores.stream()
                .reduce(BigDecimal.ZERO, BigDecimal::add);

        return sum.divide(BigDecimal.valueOf(validScores.size()), 2, RoundingMode.HALF_UP);
    }

    /**
     * 根据总分计算等级
     */
    private String calculateLevel(String configurationId, BigDecimal totalScore) {
        if (StringUtils.isBlank(configurationId) || totalScore == null) {
            return "1";
        }

        List<BizSupplierPerformanceEvaluationInterval> intervals = intervalManager.list(
                new LambdaQueryWrapper<BizSupplierPerformanceEvaluationInterval>()
                        .eq(BizSupplierPerformanceEvaluationInterval::getPerformanceEvaluationConfigurationId, configurationId)
                        .le(BizSupplierPerformanceEvaluationInterval::getMinScore, totalScore.doubleValue())
                        .ge(BizSupplierPerformanceEvaluationInterval::getMaxScore, totalScore.doubleValue()));

        return intervals.stream()
                .findFirst()
                .map(BizSupplierPerformanceEvaluationInterval::getLevel)
                .orElse("1");
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean submitUserEvaluation(ResultEvaDTO dto) {
        notificationManager.saveEva(dto);

        LambdaUpdateWrapper<BizSupplierPerformanceEvaluationResult> wrapper = new LambdaUpdateWrapper<>();
        wrapper.eq(BizSupplierPerformanceEvaluationResult::getId, dto.getPerformanceEvaluationId())
                .set(BizSupplierPerformanceEvaluationResult::getAssessmentResult, dto.getAssessmentResult())
                .set(BizSupplierPerformanceEvaluationResult::getStatus, STATUS_COMPLETED)
                .set(BizSupplierPerformanceEvaluationResult::getProcessType, dto.getProcessType());
//        createApprovalRecord(dto.getPerformanceEvaluationId(), user, STATUS_COMPLETED, "1");
        return resultManager.update(wrapper);
    }

    /**
     * 创建审批记录
     */
//    private BizSupplierPerformanceEvaluationApprovalRecord createApprovalRecord(
//            String performanceEvaluationId, IUser user, String status, String approvalResult) {
//        BizSupplierPerformanceEvaluationApprovalRecord record = new BizSupplierPerformanceEvaluationApprovalRecord();
//        record.setStatus(status);
//        record.setPerformanceEvaluationId(performanceEvaluationId);
//        record.setApprovalId(user.getUserId());
//        record.setApprovalAccount(user.getAccount());
//        record.setApprovalName(user.getFullname());
//        record.setApprovalOrgId(ContextUtil.getCurrentOrgId());
//        record.setApprovalOrgName(ContextUtil.getCurrentOrgName());
//        record.setApprovalOrgCode(ContextUtil.getCurrentOrgCode());
//        record.setApprovalResult(approvalResult);
//        recordManager.fillModel(record);
//        return record;
//    }

    /**
     * 获取审批记录
     */
    private List<BizSupplierPerformanceEvaluationApprovalRecord> getRecords(String performanceEvaluationId) {
        LambdaQueryWrapper<BizSupplierPerformanceEvaluationApprovalRecord> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(BizSupplierPerformanceEvaluationApprovalRecord::getPerformanceEvaluationId, performanceEvaluationId)
                .groupBy(BizSupplierPerformanceEvaluationApprovalRecord::getApprovalAccount,
                        BizSupplierPerformanceEvaluationApprovalRecord::getStatus)
                .orderByAsc(BizSupplierPerformanceEvaluationApprovalRecord::getCreateTime);
        return recordManager.list(wrapper);
    }

    @Override
    public boolean remove(List<String> idList) {
        List<BizSupplierPerformanceEvaluationResult> results = resultManager.listByIds(idList);
        if (CollectionUtils.isEmpty(results)) {
            return false;
        }
        boolean anyMatch = results.stream().anyMatch(result -> !result.getStatus().equals(STATUS_SUBMITTED));
        if (anyMatch) {
            throw new ApplicationException("存在已评分的数据,无法删除。");
        }
        return resultManager.removeByIds(idList);
    }
}