package com.artfess.gyl.supplierApply3.manager.impl;

import com.artfess.base.exception.ApplicationException;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.query.QueryOP;
import com.artfess.base.util.BeanUtils;
import com.artfess.file.model.DefaultFile;
import com.artfess.file.persistence.manager.FileManager;
import com.artfess.gyl.supplierApply3.dto.SupplierRegisterDTO;
import com.artfess.gyl.supplierApply3.manager.*;
import com.artfess.gyl.supplierApply3.model.*;
import com.artfess.gyl.supplierApply3.dao.BizSupplierEnterpriseRegisterInformationDao;
import com.artfess.base.manager.impl.BaseManagerImpl;
import com.artfess.gyl.supplierApply3.vo.SupplierRegisterVo;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.stream.Collectors;

/**
 * 合格供应商注册信息，在准入申请时默认查询出此数据并填充，如果没有此数据准入申请通过后默认添加一条此数据 服务实现类
 */
@Service
public class BizSupplierEnterpriseRegisterInformationManagerImpl extends BaseManagerImpl<BizSupplierEnterpriseRegisterInformationDao, BizSupplierEnterpriseRegisterInformation> implements BizSupplierEnterpriseRegisterInformationManager {

    @Resource
    private BizSupplierAccessBasicInformation3Manager information3Manager;
    @Resource
    private BizSupplierEnterpriseCategory3Manager category3Manager;
    @Resource
    private BizSupplierEnterpriseGrading3Manager grading3Manager;
    @Resource
    private BizSupplierEnterpriseQualification3Manager qualification3Manager;
    @Resource
    private BizSupplierMainProductsPerformance3Manager productsPerformance3Manager;
    @Resource
    private BizSupplierMainProductsSalesStatus3Manager mainProductsSalesStatus3Manager;
    @Resource
    private BizSupplierMainRawMaterialsSuppliers3Manager suppliers3Manager;
    @Resource
    private BizSupplierMainSupplyMaterialCategory3Manager materialCategory3Manager;
    @Resource
    private BizSupplierOtherSpecialNotes3Manager notes3Manager;
    @Resource
    private BizSupplierProductionInspectionEquipment3Manager equipment3Manager;
    @Resource
    private BizSupplierProductionProcessPerformance3Manager processPerformance3Manager;
    @Resource
    private FileManager fileManager;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean saveOrUpdateModel(SupplierRegisterDTO dto) {
        if (dto.getEnterpriseId() == null) {
            throw new ApplicationException("请先保存供应商基本信息");
        }

        boolean save = super.saveOrUpdate(dto);
        if (!save) {
            return false;
        }

        String id = dto.getId();
        Long version = dto.getVersion();
        String code = dto.getCode();

        if (code != null) {
            saveAllRelatedData(dto, id, version, code);
        }
        return true;
    }

    @Override
    @Transactional(readOnly = true)
    public SupplierRegisterVo getById(String id) {
        BizSupplierEnterpriseRegisterInformation information = super.get(id);
        if (information == null) {
            return null;
        }

        SupplierRegisterVo vo = new SupplierRegisterVo(information);
        getAllRelatedData(vo, id, information.getVersion());
        return vo;
    }

    @Override
    @Transactional(readOnly = true)
    public SupplierRegisterVo getBySupplierId(String supplierId) {
        BizSupplierEnterpriseRegisterInformation information = super.getOne(new LambdaQueryWrapper<BizSupplierEnterpriseRegisterInformation>().eq(BizSupplierEnterpriseRegisterInformation::getEnterpriseId, supplierId)
                .orderByDesc(BizSupplierEnterpriseRegisterInformation::getVersion)
                .last("limit 1"));
        if (information == null) {
            return null;
        }

        SupplierRegisterVo vo = new SupplierRegisterVo(information);
        getAllRelatedData(vo, information.getId(), information.getVersion());
        return vo;
    }

    @Override
    public PageList<BizSupplierEnterpriseRegisterInformation> queryByVersion(QueryFilter<BizSupplierEnterpriseRegisterInformation> queryFilter) {
        queryFilter.addFilter("i.IS_DELE_","0", QueryOP.EQUAL);
        IPage<BizSupplierEnterpriseRegisterInformation> iPage = baseMapper.queryByVersion(convert2IPage(queryFilter.getPageBean()), convert2Wrapper(queryFilter, currentModelClass()));
        return new PageList<>(iPage);
    }



    /**
     * 保存所有关联数据
     */
    private void saveAllRelatedData(SupplierRegisterDTO dto, String id, Long version, String code) {

        // 先删除对应模块的旧数据
        if (!StringUtils.isEmpty(id)){
            deleteRelatedDataByCode(id, code, version);
        }

        if ("-1".equals(code)) {
            // 处理所有模块
//            processAllModules(dto, id, version);
        } else {
            // 处理指定模块
            processSingleModule(dto, id, version, code);
        }
    }

    /**
     * 处理所有模块
     */
//    private void processAllModules(SupplierRegisterDTO dto, String id, Long version) {
//        // 基础拓展信息
//        processBasicInformation(dto.getBasicInformation(), id, version);
//        // 分类
//        processCategoryData(dto.getEnterpriseCategory(), id, version);
//        // 分级
//        processGradingData(dto.getEnterpriseGrading(), id, version);
//        // 主要原材料及供应商
//        processSuppliersData(dto.getSuppliers(), id, version);
//        // 供应商主供物资分类
//        processMaterialCategoryData(dto.getCategory(), id, version);
//        // 其他特别说明
//        processOtherSpecialNotesData(dto.getOtherSpecialNotes(), id, version);
//        // 生产及检验设备
//        processProdEquipmentData(dto, id, version);
//        // 生产及检验设备
//        processMonitEquipmentData(dto, id, version);
//        // 资质
//        processQualificationData(dto.getQualification(), version);
//        // 主营产品情况及质量业绩
//        processProductsPerformanceData(dto.getPerformance(), id, version);
//        // 生产制造过程绩效
//        processProcessPerformanceData(dto.getProcessPerformance(), id, version);
//        //
//        processMainProductsSalesData(dto.getSalesStatus(), id, version);
//    }

    /**
     * 处理单个模块
     */
    private void processSingleModule(SupplierRegisterDTO dto, String id, Long version, String code) {
        switch (code) {
            case "1":
                processBasicInformation(dto.getBasicInformation(), id, version);
                break;
//            case "2":
//                processCategoryData(dto.getEnterpriseCategory(), id, version);
//                break;
//            case "3":
//                processGradingData(dto.getEnterpriseGrading(), id, version);
//                break;
            case "4":
                processSuppliersData(dto.getSuppliers(), id, version);
                break;
            case "5":
                processMaterialCategoryData(dto.getCategory(), id, version);
                break;
            case "6":
                processOtherSpecialNotesData(dto.getOtherSpecialNotes(), id, version);
                break;
            case "7":
                processProdEquipmentData(dto, id, version);
                break;
//            case "8":
//                processQualificationData(dto.getQualification(), version);
//                break;
            case "9":
                processProductsPerformanceData(dto.getPerformance(), id, version);
                break;
            case "10":
                processProcessPerformanceData(dto.getProcessPerformance(), id, version);
                break;
            case "11":
                processMainProductsSalesData(dto.getSalesStatus(), id, version);
                break;
            case "12":
                processMonitEquipmentData(dto, id, version);
                break;
        }
    }

    /**
     * 处理基础信息数据
     */
    private void processBasicInformation(BizSupplierAccessBasicInformation3 entity, String id, Long version) {
        if (entity != null) {
            entity.setId(null);
            entity.setSupplierEnterpriseId(id);
            entity.setVersion(version);
            information3Manager.save(entity);
        }
    }

    /**
     * 处理分类数据
     */
    private void processCategoryData(List<BizSupplierEnterpriseCategory3> list, String id, Long version) {
        if (CollectionUtils.isNotEmpty(list)) {
            list.forEach(entity -> {
                entity.setId(null);
                entity.setSupplierEnterpriseId(id);
                entity.setVersion(version);
            });
            category3Manager.saveBatch(list);
        }
    }

    /**
     * 处理分级数据
     */
    private void processGradingData(List<BizSupplierEnterpriseGrading3> list, String id, Long version) {
        if (CollectionUtils.isNotEmpty(list)) {
            list.forEach(entity -> {
                entity.setId(null);
                entity.setSupplierEnterpriseId(id);
                entity.setVersion(version);
            });
            grading3Manager.saveBatch(list);
        }
    }

    /**
     * 处理供应商数据
     */
    private void processSuppliersData(List<BizSupplierMainRawMaterialsSuppliers3> list, String id, Long version) {
        if (CollectionUtils.isNotEmpty(list)) {
            list.forEach(entity -> {
                entity.setSupplierEnterpriseId(id);
                entity.setVersion(version);
                entity.setId(null);
            });
            boolean saveBatch = suppliers3Manager.saveBatch(list);
            if(saveBatch){
                list.forEach(entity -> {
                    List<DefaultFile> files = entity.getFiles();
                    if (files != null && !files.isEmpty()) {
                        List<String> fileIds = files.stream().map(DefaultFile::getId).collect(Collectors.toList());
                        fileManager.updateFileBizIdByIds(fileIds,entity.getId());
                    }
                });
            }
        }
    }

    /**
     * 处理物资分类数据
     */
    private void processMaterialCategoryData(List<BizSupplierMainSupplyMaterialCategory3> list, String id, Long version) {
        if (CollectionUtils.isNotEmpty(list)) {
            list.forEach(entity -> {
                entity.setSupplierEnterpriseId(id);
                entity.setVersion(version);
                entity.setId(null);
            });
            boolean saveBatch = materialCategory3Manager.saveBatch(list);
            if(saveBatch){
                list.forEach(entity -> {
                    List<DefaultFile> files = entity.getFiles();
                    if (files != null && !files.isEmpty()) {
                        List<String> fileIds = files.stream().map(DefaultFile::getId).collect(Collectors.toList());
                        fileManager.updateFileBizIdByIds(fileIds,entity.getId());
                    }
                });
            }
        }
    }

    /**
     * 处理其他特别说明数据
     */
    private void processOtherSpecialNotesData(BizSupplierOtherSpecialNotes3 entity, String id, Long version) {
        if (BeanUtils.isNotEmpty(entity)) {
                entity.setSupplierEnterpriseId(id);
                entity.setVersion(version);
                entity.setId(null);
            notes3Manager.save(entity);
            //处理附件
            List<DefaultFile> files = entity.getFiles();
            if ( files!= null && files.size() > 0) {
                List<String> fileIds = files.stream().map(DefaultFile::getId).collect(Collectors.toList());
                //删除未在本次更新内的附件
                fileManager.removeFileByBizId(entity.getId(),fileIds);
                //更新附件
                fileManager.updateFileBizIdByIds(fileIds, entity.getId());
            }else {
                fileManager.removeFileByBizId(entity.getId(),null);
            }
        }
    }

    /**
     * 处理设备数据
     */
    private void processProdEquipmentData(SupplierRegisterDTO dto, String id, Long version) {
        List<BizSupplierProductionInspectionEquipment3> equipmentList = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(dto.getProdEquipment())) {
            equipmentList.addAll(dto.getProdEquipment());
        }
        if (CollectionUtils.isNotEmpty(equipmentList)) {
            equipmentList.forEach(entity -> {
                entity.setSupplierEnterpriseId(id);
                entity.setVersion(version);
                entity.setId(null);
            });
            boolean saveBatch = equipment3Manager.saveBatch(equipmentList);
            if(saveBatch){
                equipmentList.forEach(entity -> {
                    List<DefaultFile> files = entity.getFiles();
                    if (files != null && !files.isEmpty()) {
                        List<String> fileIds = files.stream().map(DefaultFile::getId).collect(Collectors.toList());
                        fileManager.updateFileBizIdByIds(fileIds,entity.getId());
                    }
                });
            }
        }
    }

    /**
     * 处理设备数据
     */
    private void processMonitEquipmentData(SupplierRegisterDTO dto, String id, Long version) {
        List<BizSupplierProductionInspectionEquipment3> equipmentList = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(dto.getMonitEquipment())) {
            equipmentList.addAll(dto.getMonitEquipment());
        }

        if (CollectionUtils.isNotEmpty(equipmentList)) {
            equipmentList.forEach(entity -> {
                entity.setSupplierEnterpriseId(id);
                entity.setVersion(version);
                entity.setId(null);
            });
            boolean saveBatch = equipment3Manager.saveBatch(equipmentList);
            if(saveBatch){
                for (BizSupplierProductionInspectionEquipment3 equipment3 : equipmentList) {
                    //处理附件
                    List<DefaultFile> files = equipment3.getFiles();
                    if ( files!= null && files.size() > 0) {
                        List<String> fileIds = files.stream().map(DefaultFile::getId).collect(Collectors.toList());
                        //删除未在本次更新内的附件
                        fileManager.removeFileByBizId(equipment3.getId(),fileIds);
                        //更新附件
                        fileManager.updateFileBizIdByIds(fileIds, equipment3.getId());
                    }else {
                        fileManager.removeFileByBizId(equipment3.getId(),null);
                    }
                }
            }

        }
    }


    /**
     * 处理产品性能数据
     */
    private void processProductsPerformanceData(BizSupplierMainProductsPerformance3 entity, String id, Long version) {
        if (entity != null) {
            productsPerformance3Manager.fillModel(entity, id, version);
        }
    }

    /**
     * 处理主营产品生产销售情况
     */
    private void processMainProductsSalesData(List<BizSupplierMainProductsSalesStatus3> list, String id, Long version) {
        if (CollectionUtils.isNotEmpty(list)) {
            list.forEach(entity -> {
                entity.setVersion(version);
                entity.setId(null);
                entity.setSupplierEnterpriseId(id);
            });
            mainProductsSalesStatus3Manager.saveBatch(list);
            for (BizSupplierMainProductsSalesStatus3 salesStatus3 : list) {
                //处理附件
                List<DefaultFile> files = salesStatus3.getFiles();
                if ( files!= null && files.size() > 0) {
                    List<String> fileIds = files.stream().map(DefaultFile::getId).collect(Collectors.toList());
                    //删除未在本次更新内的附件
                    fileManager.removeFileByBizId(salesStatus3.getId(),fileIds);
                    //更新附件
                    fileManager.updateFileBizIdByIds(fileIds, salesStatus3.getId());
                }else {
                    fileManager.removeFileByBizId(salesStatus3.getId(),null);
                }
            }

        }
    }

    /**
     * 处理过程性能数据
     */
    private void processProcessPerformanceData(BizSupplierProductionProcessPerformance3 entity, String id, Long version) {
        if (entity != null) {
            processPerformance3Manager.fillModel(entity, id, version);
        }
    }

    /**
     * 获取所有关联数据
     */
    private void getAllRelatedData(SupplierRegisterVo vo, String id, Long version) {
        // 创建线程池（根据实际情况调整）
        ExecutorService executor = Executors.newFixedThreadPool(8);

        try {
            // 并行执行所有查询
            CompletableFuture<BizSupplierAccessBasicInformation3> basicInfoFuture = CompletableFuture
                    .supplyAsync(() -> getBasicInformation(id), executor);

            CompletableFuture<List<BizSupplierMainRawMaterialsSuppliers3>> suppliersFuture = CompletableFuture
                    .supplyAsync(() -> getSuppliersData(id), executor);

            CompletableFuture<List<BizSupplierMainSupplyMaterialCategory3>> categoryFuture = CompletableFuture
                    .supplyAsync(() -> getMaterialCategoryData(id), executor);

            CompletableFuture<BizSupplierOtherSpecialNotes3> notesFuture = CompletableFuture
                    .supplyAsync(() -> getOtherSpecialNotesData(id), executor);

            CompletableFuture<Map<String, List<BizSupplierProductionInspectionEquipment3>>> equipmentFuture =
                    CompletableFuture.supplyAsync(() -> getEquipmentData(id), executor);

            CompletableFuture<BizSupplierMainProductsPerformance3> performanceFuture = CompletableFuture
                    .supplyAsync(() -> getProductsPerformanceData(id), executor);

            CompletableFuture<BizSupplierProductionProcessPerformance3> processPerfFuture = CompletableFuture
                    .supplyAsync(() -> getProcessPerformanceData(id), executor);

            CompletableFuture<List<BizSupplierMainProductsSalesStatus3>> salesStatusFuture = CompletableFuture
                    .supplyAsync(() -> getProcessMainProductsSalesData(id), executor);

            // 等待所有查询完成
            CompletableFuture.allOf(
                    basicInfoFuture, suppliersFuture, categoryFuture, notesFuture,
                    equipmentFuture, performanceFuture, processPerfFuture, salesStatusFuture
            ).join();

            // 设置结果
            vo.setBasicInformation(basicInfoFuture.join());
            vo.setSuppliers(suppliersFuture.join());
            vo.setCategory(categoryFuture.join());
            vo.setOtherSpecialNotes(notesFuture.join());
            Map<String, List<BizSupplierProductionInspectionEquipment3>> equipmentMap = equipmentFuture.join();
            vo.setProdEquipment(equipmentMap.getOrDefault("1", Collections.emptyList()));
            vo.setMonitEquipment(equipmentMap.getOrDefault("2", Collections.emptyList()));
            vo.setPerformance(performanceFuture.join());
            vo.setProcessPerformance(processPerfFuture.join());
            vo.setSalesStatus(salesStatusFuture.join());
        } finally {
            executor.shutdown();
        }
    }

    /**
     * 查询基础信息
     */
    private BizSupplierAccessBasicInformation3 getBasicInformation(String id) {
        LambdaQueryWrapper<BizSupplierAccessBasicInformation3> query = new LambdaQueryWrapper<>();
        query.eq(BizSupplierAccessBasicInformation3::getSupplierEnterpriseId, id);
        return information3Manager.getOne(query);
    }

    /**
     * 查询供应商数据
     */
    private List<BizSupplierMainRawMaterialsSuppliers3> getSuppliersData(String id) {
        LambdaQueryWrapper<BizSupplierMainRawMaterialsSuppliers3> query = new LambdaQueryWrapper<>();
        query.eq(BizSupplierMainRawMaterialsSuppliers3::getSupplierEnterpriseId, id);
        List<BizSupplierMainRawMaterialsSuppliers3> list = suppliers3Manager.list(query);
        // 2. 如果没有数据，直接返回
        if (CollectionUtils.isEmpty(list)) {
            return Collections.emptyList();
        }
        // 3. 提取所有物料分类ID
        List<String> categoryIds = list.stream()
                .map(BizSupplierMainRawMaterialsSuppliers3::getId)
                .filter(Objects::nonNull)
                .distinct()
                .collect(Collectors.toList());
        // 4. 批量查询文件
        List<DefaultFile> files = fileManager.queryFileByBizIds(categoryIds);
        // 5. 按业务ID分组文件
        Map<String, List<DefaultFile>> fileMap = files.stream()
                .filter(Objects::nonNull)
                .collect(Collectors.groupingBy(DefaultFile::getBizId));
        // 6. 为每个物料分类设置文件
        list.forEach(category ->
                category.setFiles(fileMap.getOrDefault(category.getId(), Collections.emptyList()))
        );
        return list;
    }

    /**
     * 查询物资分类数据
     */
    private List<BizSupplierMainSupplyMaterialCategory3> getMaterialCategoryData(String id) {
        // 1. 查询物料分类
        List<BizSupplierMainSupplyMaterialCategory3> categories = materialCategory3Manager.lambdaQuery()
                .eq(BizSupplierMainSupplyMaterialCategory3::getSupplierEnterpriseId, id)
                .list();
        // 2. 如果没有数据，直接返回
        if (CollectionUtils.isEmpty(categories)) {
            return Collections.emptyList();
        }
        // 3. 提取所有物料分类ID
        List<String> categoryIds = categories.stream()
                .map(BizSupplierMainSupplyMaterialCategory3::getId)
                .filter(Objects::nonNull)
                .distinct()
                .collect(Collectors.toList());
        // 4. 批量查询文件
        List<DefaultFile> files = fileManager.queryFileByBizIds(categoryIds);
        // 5. 按业务ID分组文件
        Map<String, List<DefaultFile>> fileMap = files.stream()
                .filter(Objects::nonNull)
                .collect(Collectors.groupingBy(DefaultFile::getBizId));
        // 6. 为每个物料分类设置文件
        categories.forEach(category ->
                category.setFiles(fileMap.getOrDefault(category.getId(), Collections.emptyList()))
        );
        return categories;
    }

    /**
     * 查询其他特别说明数据
     */
    private BizSupplierOtherSpecialNotes3 getOtherSpecialNotesData(String id) {
        LambdaQueryWrapper<BizSupplierOtherSpecialNotes3> query = new LambdaQueryWrapper<>();
        query.eq(BizSupplierOtherSpecialNotes3::getSupplierEnterpriseId, id);
        BizSupplierOtherSpecialNotes3 entity = notes3Manager.getOne(query);
        if (entity==null){
            return null;
        }
        List<DefaultFile> files = fileManager.queryFileByBizId(entity.getId());
        if (!CollectionUtils.isEmpty(files)) {
            entity.setFiles(files);
        }
        return entity;
    }

    /**
     * 获取设备数据并按类型分组
     */
    private Map<String, List<BizSupplierProductionInspectionEquipment3>> getEquipmentData(String id) {
        LambdaQueryWrapper<BizSupplierProductionInspectionEquipment3> query = new LambdaQueryWrapper<>();
        query.eq(BizSupplierProductionInspectionEquipment3::getSupplierEnterpriseId, id);
        List<BizSupplierProductionInspectionEquipment3> list = equipment3Manager.list(query);
        if (CollectionUtils.isEmpty(list)) {
            return Collections.emptyMap();
        }
        // 3. 提取所有物料分类ID
        List<String> bizIds = list.stream()
                .map(BizSupplierProductionInspectionEquipment3::getId)
                .filter(Objects::nonNull)
                .distinct()
                .collect(Collectors.toList());
        // 4. 批量查询文件
        List<DefaultFile> files = fileManager.queryFileByBizIds(bizIds);
        // 5. 按业务ID分组文件
        Map<String, List<DefaultFile>> fileMap = files.stream()
                .filter(Objects::nonNull)
                .collect(Collectors.groupingBy(DefaultFile::getBizId));
        // 6. 为每个物料分类设置文件
        list.forEach(category ->
                category.setFiles(fileMap.getOrDefault(category.getId(), Collections.emptyList()))
        );
        return list.stream().collect(Collectors.groupingBy(BizSupplierProductionInspectionEquipment3::getDeviceType));
    }


    /**
     * 查询设备数据
     */
    private List<BizSupplierMainProductsSalesStatus3> getProcessMainProductsSalesData(String id) {
        LambdaQueryWrapper<BizSupplierMainProductsSalesStatus3> query = new LambdaQueryWrapper<>();
        query.eq(BizSupplierMainProductsSalesStatus3::getSupplierEnterpriseId, id);
        List<BizSupplierMainProductsSalesStatus3> list = mainProductsSalesStatus3Manager.list(query);
        // 2. 如果没有数据，直接返回
        if (CollectionUtils.isEmpty(list)) {
            return Collections.emptyList();
        }
        // 3. 提取所有物料分类ID
        List<String> categoryIds = list.stream()
                .map(BizSupplierMainProductsSalesStatus3::getId)
                .filter(Objects::nonNull)
                .distinct()
                .collect(Collectors.toList());
        // 4. 批量查询文件
        List<DefaultFile> files = fileManager.queryFileByBizIds(categoryIds);
        // 5. 按业务ID分组文件
        Map<String, List<DefaultFile>> fileMap = files.stream()
                .filter(Objects::nonNull)
                .collect(Collectors.groupingBy(DefaultFile::getBizId));
        // 6. 为每个物料分类设置文件
        list.forEach(entity ->
                entity.setFiles(fileMap.getOrDefault(entity.getId(), Collections.emptyList()))
        );
        return list;
    }

    /**
     * 查询产品性能数据
     */
    private BizSupplierMainProductsPerformance3 getProductsPerformanceData(String id) {
        return productsPerformance3Manager.getModel(id);
    }

    /**
     * 查询过程性能数据
     */
    private BizSupplierProductionProcessPerformance3 getProcessPerformanceData(String id) {
        return processPerformance3Manager.getModel(id);
    }



    /**
     * 根据模块代码删除关联数据
     */
    private void deleteRelatedDataByCode(String id, String code, Long version) {
        if ("-1".equals(code)) {
            // 删除所有模块数据
            deleteAllModules(id, version);
        } else {
            deleteSingleModule(id, code, version);
        }
    }

    /**
     * 删除所有模块数据
     */
    private void deleteAllModules(String id, Long version) {
        deleteBasicInformation(id);
        deleteCategoryData(id);
        deleteGradingData(id);
        deleteSuppliersData(id);
        deleteMaterialCategoryData(id);
        deleteOtherSpecialNotesData(id);
        deleteEquipmentData(id,null);
        deleteQualificationData(version);
        deleteProductsPerformanceData(id);
        deleteProcessPerformanceData(id);
        deleteProcessMainProductsSalesData(id);
    }

    /**
     * 删除单个模块数据
     */
    private void deleteSingleModule(String id, String code, Long version) {
        switch (code) {
            case "1":
                deleteBasicInformation(id);
                break;
            case "2":
                deleteCategoryData(id);
                break;
            case "3":
                deleteGradingData(id);
                break;
            case "4":
                deleteSuppliersData(id);
                break;
            case "5":
                deleteMaterialCategoryData(id);
                break;
            case "6":
                deleteOtherSpecialNotesData(id);
                break;
            case "7":
                deleteEquipmentData(id,"1");
                break;
//            case "8":
//                deleteQualificationData(version);
//                break;
            case "9":
                deleteProductsPerformanceData(id);
                break;
            case "10":
                deleteProcessPerformanceData(id);
                break;
            case "11":
                deleteProcessMainProductsSalesData(id);
                break;
            case "12":
                deleteEquipmentData(id,"2");
                break;
        }
    }

    /**
     * 删除基础信息
     */
    private void deleteBasicInformation(String id) {
        LambdaQueryWrapper<BizSupplierAccessBasicInformation3> query = new LambdaQueryWrapper<>();
        query.eq(BizSupplierAccessBasicInformation3::getSupplierEnterpriseId, id);
        information3Manager.remove(query);
    }

    /**
     * 删除分类数据
     */
    private void deleteCategoryData(String id) {
        LambdaQueryWrapper<BizSupplierEnterpriseCategory3> query = new LambdaQueryWrapper<>();
        query.eq(BizSupplierEnterpriseCategory3::getSupplierEnterpriseId, id);
        category3Manager.remove(query);
    }

    /**
     * 删除分级数据
     */
    private void deleteGradingData(String id) {
        LambdaQueryWrapper<BizSupplierEnterpriseGrading3> query = new LambdaQueryWrapper<>();
        query.eq(BizSupplierEnterpriseGrading3::getSupplierEnterpriseId, id);
        grading3Manager.remove(query);
    }

    /**
     * 删除供应商数据
     */
    private void deleteSuppliersData(String id) {
        LambdaQueryWrapper<BizSupplierMainRawMaterialsSuppliers3> query = new LambdaQueryWrapper<>();
        query.eq(BizSupplierMainRawMaterialsSuppliers3::getSupplierEnterpriseId, id);
        suppliers3Manager.remove(query);
    }

    /**
     * 删除物资分类数据
     */
    private void deleteMaterialCategoryData(String id) {
        LambdaQueryWrapper<BizSupplierMainSupplyMaterialCategory3> query = new LambdaQueryWrapper<>();
        query.eq(BizSupplierMainSupplyMaterialCategory3::getSupplierEnterpriseId, id);
        materialCategory3Manager.remove(query);
    }

    /**
     * 删除其他特别说明数据
     */
    private void deleteOtherSpecialNotesData(String id) {
        LambdaQueryWrapper<BizSupplierOtherSpecialNotes3> query = new LambdaQueryWrapper<>();
        query.eq(BizSupplierOtherSpecialNotes3::getSupplierEnterpriseId, id);
        notes3Manager.remove(query);
    }

    /**
     * 删除其他特别说明数据
     */
    private void deleteData(String id) {
        LambdaQueryWrapper<BizSupplierOtherSpecialNotes3> query = new LambdaQueryWrapper<>();
        query.eq(BizSupplierOtherSpecialNotes3::getSupplierEnterpriseId, id);
        notes3Manager.remove(query);
    }

    /**
     * 删除设备数据
     */
    private void deleteEquipmentData(String id,String deviceType) {
        LambdaQueryWrapper<BizSupplierProductionInspectionEquipment3> query = new LambdaQueryWrapper<>();
        query.eq(BizSupplierProductionInspectionEquipment3::getSupplierEnterpriseId, id);
        if (deviceType!=null) {
            query.eq(BizSupplierProductionInspectionEquipment3::getDeviceType, deviceType);
        }
        equipment3Manager.remove(query);
    }

    /**
     * 删除资质数据
     */
    private void deleteQualificationData(Long version) {
        LambdaQueryWrapper<BizSupplierEnterpriseQualification3> query = new LambdaQueryWrapper<>();
        query.eq(BizSupplierEnterpriseQualification3::getVersion, version);
        qualification3Manager.remove(query);
    }

    /**
     * 删除产品性能数据
     */
    private void deleteProductsPerformanceData(String id) {
        productsPerformance3Manager.deleteModelById(id);
    }

    /**
     * 删除过程性能数据
     */
    private void deleteProcessPerformanceData(String id) {
        processPerformance3Manager.deleteModelById(id);
    }

    /**
     * 删除设备数据
     */
    private void deleteProcessMainProductsSalesData(String id) {
        LambdaQueryWrapper<BizSupplierMainProductsSalesStatus3> query = new LambdaQueryWrapper<>();
        query.eq(BizSupplierMainProductsSalesStatus3::getSupplierEnterpriseId, id);
        mainProductsSalesStatus3Manager.remove(query);
    }


}