package com.artfess.gyl.base.manager.impl;

import com.artfess.base.context.BaseContext;
import com.artfess.base.enums.FileTypeEnum;
import com.artfess.base.exception.BaseException;
import com.artfess.base.query.*;
import com.artfess.base.util.BeanUtils;
import com.artfess.base.util.FileUtil;
import com.artfess.base.util.HttpUtil;
import com.artfess.file.util.MinioUtil;
import com.artfess.file.util.OfficeToPdf;
import com.artfess.gyl.base.dao.BizAccessoryDao;
import com.artfess.gyl.base.model.BizAccessory;
import com.artfess.gyl.base.manager.BizAccessoryManager;
import com.artfess.base.manager.impl.BaseManagerImpl;
import com.artfess.gyl.base.vo.Chunk;
import com.artfess.gyl.utils.MinioUtils;
import com.artfess.uc.api.impl.util.ContextUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import io.jsonwebtoken.lang.Assert;
import net.lingala.zip4j.core.ZipFile;
import net.lingala.zip4j.model.ZipParameters;
import net.lingala.zip4j.util.Zip4jConstants;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;
import java.io.*;
import java.nio.channels.FileChannel;
import java.text.DecimalFormat;
import java.time.LocalDateTime;
import java.util.*;

/**
 * 通用附件表 服务实现类
 *
 * @author 管理员
 * @since 2025-12-10
 */
@Service
public class BizAccessoryManagerImpl extends BaseManagerImpl<BizAccessoryDao, BizAccessory> implements BizAccessoryManager {

    @Resource
    private MinioUtils utils;
    @Resource
    private OfficeToPdf officeToPdf;

    @Override
    public List<BizAccessory> getAccessoryBySourceId(String sourceId) {
        QueryFilter<BizAccessory> queryFilter = QueryFilter.build();
//        queryFilter.addFilter("IS_DELE_", "0", QueryOP.EQUAL);
        queryFilter.addFilter("SOURCE_ID_", sourceId, QueryOP.EQUAL);
        if(BeanUtils.isEmpty(queryFilter.getPageBean())){
            queryFilter.setPageBean(new PageBean(1,-1));
        }
        IPage<BizAccessory> accessoryIPage = baseMapper.selectPageBySourceId(
                convert2IPage(queryFilter.getPageBean()),
                convert2Wrapper(queryFilter, currentModelClass()));
        return accessoryIPage.getRecords();

    }

    @Override
    public List<BizAccessory> getAccessoryBySourceIdList(List<String> sourceIds) {
        QueryFilter<BizAccessory> queryFilter = QueryFilter.build();
//        queryFilter.addFilter("IS_DELE_", "0", QueryOP.EQUAL);
        queryFilter.addFilter("SOURCE_ID_", sourceIds, QueryOP.IN);
        if(BeanUtils.isEmpty(queryFilter.getPageBean())){
            queryFilter.setPageBean(new PageBean(1,-1));
        }
        IPage<BizAccessory> accessoryIPage = baseMapper.selectPageBySourceId(
                convert2IPage(queryFilter.getPageBean()),
                convert2Wrapper(queryFilter, currentModelClass()));
        return accessoryIPage.getRecords();

    }

    @Override
    public boolean saveAccessory(List<BizAccessory> list) {
        // 同步设置更新信息为创建信息，WPS预览时需要相关数据，不能为空
        for(BizAccessory ele:list){
            String currentUserId = ContextUtil.getCurrentUserId();
            String currentUserName = ContextUtil.getCurrentUserName();
            if (StringUtils.isBlank(ele.getId())) {
                ele.setCreateBy(currentUserId);
                ele.setCreateName(currentUserName);
                ele.setCreateTime(LocalDateTime.now());
            } else {
                ele.setUpdateName(currentUserName);
                ele.setUpdateBy(currentUserId);
                ele.setUpdateTime(LocalDateTime.now());
            }
        }
        boolean result=saveBatch(list);
        return result;
    }

    @Override
    public String fileUploadPost(Chunk chunk, HttpServletResponse response) throws IOException {
        /**
         * 每一个上传块都会包含如下分块信息：
         * chunkNumber: 当前块的次序，第一个块是 1，注意不是从 0 开始的。
         * totalChunks: 文件被分成块的总数。
         * chunkSize: 分块大小，根据 totalSize 和这个值你就可以计算出总共的块数。注意最后一块的大小可能会比这个要大。
         * currentChunkSize: 当前块的大小，实际大小。
         * totalSize: 文件总大小。
         * identifier: 这个就是每个文件的唯一标示。
         * filename: 文件名。
         * relativePath: 文件夹上传的时候文件的相对路径属性。
         * 一个分块可以被上传多次，当然这肯定不是标准行为，但是在实际上传过程中是可能发生这种事情的，这种重传也是本库的特性之一。
         *
         * 根据响应码认为成功或失败的：
         * 200 文件上传完成
         * 201 文加快上传成功
         * 500 第一块上传失败，取消整个文件上传
         * 507 服务器出错自动重试该文件块上传
         */
        // 以文件名作为目录保存各片文件
        String path = utils.getFileDir(chunk.getFilename());
        // 各片文件加上后缀，固定格式
//        File file= new File(path, "temp_"+chunk.getChunkNumber());
        File file = new File(path, "" + chunk.getChunkNumber());
        // 创建文件
        if (!file.exists()) {
            // 父级文件夹，用于管理所有分片文件
            new File(path).mkdirs();
            // 分片块文件
            file.createNewFile();
        }
        //进行写文件操作
        try (
                //将块文件写入文件中
                InputStream fos = chunk.getFile().getInputStream();
                RandomAccessFile raf = new RandomAccessFile(file, "rw")
        ) {
            int len = -1;
            byte[] buffer = new byte[1024];
//            raf.seek((chunk.getChunkNumber()-1)*1024*1024*5);
            while ((len = fos.read(buffer)) != -1) {
                raf.write(buffer, 0, len);
            }
        } catch (IOException e) {
            e.printStackTrace();
            if (chunk.getChunkNumber() == 1) {
                file.delete();
            }
            response.setStatus(507);
            return "exception:文件写入异常";
        }
        if (chunk.getChunkNumber().equals(chunk.getTotalChunks())) {
            response.setStatus(200);
            return "over";
        } else {
            response.setStatus(201);
            return "ok";
        }
    }

    @Override
    public List<BizAccessory> minioBatchUpload(List<MultipartFile> fileList, String bizPath, HttpServletResponse response) throws Exception{
        List<BizAccessory> accessoryList = new ArrayList<>(fileList.size());
        for (MultipartFile file : fileList){
            accessoryList.add(this.minioUpload(file, bizPath, response));
        }
        return accessoryList;
    }

    @Override
    public PageList<BizAccessory> getAccessoryPageBySourceId(QueryFilter<BizAccessory> queryFilter) {
        // 限制SourceID不能为空
        String sourceID = null;
        List<QueryField> test = queryFilter.getQuerys();
        for (QueryField ele : test) {
            if ("SOURCE_ID_".equals(ele.getProperty())) {
                sourceID = String.valueOf(ele.getValue());
                break;
            }
        }
        Assert.notNull(sourceID, "SOURCE_ID_不能为空");
        queryFilter.addFilter("IS_DELE_", "0", QueryOP.EQUAL);
        IPage<BizAccessory> accessoryIPage = baseMapper.selectPageBySourceId(
                convert2IPage(queryFilter.getPageBean()),
                convert2Wrapper(queryFilter, currentModelClass())
        );
        List<BizAccessory> records = accessoryIPage.getRecords();
        return new PageList<BizAccessory>(accessoryIPage);

    }

    @Override
    public boolean removeBySourceId(String sourceId) {
        LambdaQueryWrapper<BizAccessory> wrapper=new LambdaQueryWrapper<>();
        wrapper.eq(BizAccessory::getSourceId,sourceId);
        boolean remove = remove(wrapper);
        return remove;
    }

    @Override
    public boolean removeBySourceIds(List<String> sourceIds) {
        LambdaQueryWrapper<BizAccessory> wrapper=new LambdaQueryWrapper<>();
        wrapper.in(BizAccessory::getSourceId,sourceIds);
        boolean remove = remove(wrapper);
        return remove;
    }

    /**
     * kkutIO上传
     *
     * @param fileName 上传文件名称
     * @param bizPath  文件夹路径
     * @return 附件对象
     * @throws Exception 异常
     */
    @Override
    public BizAccessory minioUpload(String bizPath, String fileName) throws Exception {
        Assert.notNull(fileName, "文件名不能为空！");
        File file = mergeFile(fileName);
        BizAccessory accessory = new BizAccessory();
        if (null == file || file.length() == 0 || !file.exists()) {
            throw new BaseException("文件为空");
        }
        accessory.setName(fileName);
        String name = DigestUtils.md5Hex(UUID.randomUUID().toString()) + fileName.substring(fileName.lastIndexOf("."));
        //如果为空则上传到MinIO中的默认文件夹
        if (StringUtils.isEmpty(bizPath)) {
            bizPath = "default";
        }
        String suffix = fileName.substring(fileName.lastIndexOf(".") + 1);
        accessory.setSuffix(suffix);

        List<String> suffixList = Arrays.asList("png,jpg,jpeg,xlsx,xls,pdf,docx,doc,zip,rar,excel,pptx".split(","));
        if(!suffixList.contains(suffix.toLowerCase())){
            throw new BaseException("该文件类型不能上传");
        }

        String url = MinioUtil.getMinIoUrl(bizPath, name);
        // URL MinIO 异步上传
        accessory.setUrl(url);
        String userId = ContextUtil.getCurrentUserId();
        String userName = ContextUtil.getCurrentUserName();
        accessory.setCreateBy(userId);
        accessory.setUpdateBy(userId);
        accessory.setCreateName(userName);
        accessory.setUpdateName(userName);
        accessory.setCreateTime(LocalDateTime.now());
        accessory.setUpdateTime(LocalDateTime.now());
        accessory.setType(handleFileType(accessory.getSuffix()));
        accessory.setFileSize(new DecimalFormat("#,##0.##").format(file.length() / 1024.00 / 1024.00) + " MB");
        utils.startMinIOUpload(file, bizPath, name);
        int a = 0;
        return accessory;
    }


    /**
     * kkutIO上传
     *
     * @param fileName 上传文件名称
     * @param bizPath  文件夹路径
     * @return 附件对象
     * @throws Exception 异常
     */
    @Override
    public BizAccessory minioUpload(String bizPath, String fileName,String bucketName) throws Exception {
        Assert.notNull(fileName, "文件名不能为空！");
        File file = mergeFile(fileName);
        BizAccessory accessory = new BizAccessory();
        if (null == file || file.length() == 0 || !file.exists()) {
            throw new BaseException("文件为空");
        }
        accessory.setName(fileName);
        String name = DigestUtils.md5Hex(UUID.randomUUID().toString()) + fileName.substring(fileName.lastIndexOf("."));
        //如果为空则上传到MinIO中的默认文件夹
        if (StringUtils.isEmpty(bizPath)) {
            bizPath = "default";
        }
        String suffix = fileName.substring(fileName.lastIndexOf(".") + 1);
        accessory.setSuffix(suffix);

        List<String> suffixList = Arrays.asList("png,jpg,jpeg,xlsx,xls,pdf,docx,doc,zip,rar,excel,pptx,stp".split(","));
        if(!suffixList.contains(suffix.toLowerCase())){
            throw new BaseException("该文件类型不能上传");
        }

        String url = MinioUtil.getMinIoUrl(bizPath, name);
        // URL MinIO 异步上传
        accessory.setUrl(url);
        String userId = ContextUtil.getCurrentUserId();
        String userName = ContextUtil.getCurrentUserName();
        accessory.setCreateBy(userId);
        accessory.setUpdateBy(userId);
        accessory.setCreateName(userName);
        accessory.setUpdateName(userName);
        accessory.setCreateTime(LocalDateTime.now());
        accessory.setUpdateTime(LocalDateTime.now());
//        accessory.setIsDele("0");
        accessory.setType(handleFileType(accessory.getSuffix()));
        accessory.setFileSize(new DecimalFormat("#,##0.##").format(file.length() / 1024.00 / 1024.00) + " MB");
        utils.startMinIOUpload(file, bizPath, name,bucketName);
        accessory.setBidSignStatus(1);
        return accessory;
    }

    /***
     *
     * @param accessory 附件对象
     * @param bizPath
     * @return 返回
     * @throws Exception
     */
    @Override
    public String minIoDownFile(BizAccessory accessory, String bizPath) throws Exception {
        if (StringUtils.isEmpty(bizPath)) {
            bizPath = "default";
        }
        String customBucket = "cgpt", filePath = "download/" + bizPath + "/" + accessory.getId();
        File file = new File(filePath);
        if (!file.exists()) {
            file.mkdirs();
        }
//        Accessory accessory = this.getById(id);
        String url = accessory.getUrl();
        String dir = url.substring(url.lastIndexOf("/"));
        String suffix = accessory.getSuffix();
        File fileFullPath = new File(file + "/" + accessory.getName());
        if(fileFullPath.exists()){
            return filePath + "/";
        }
        FileOutputStream out = new FileOutputStream(file + "/" + accessory.getName());
        MinioUtil.downFile(customBucket, bizPath + dir, out, false);
        String downPath = filePath + "/";
//        FileUtil.deleteFile(downPath);
        return downPath;
    }


    /**********************下载文件************************/
    @Override
    public void minIoDownFile(String ids, String bizPath, HttpServletResponse response) throws Exception {
        String customBucket = "cgpt", filePath = "download/" + bizPath;
        if (StringUtils.isNotBlank(ids)) {
            File path = new File(filePath);
            if (!path.exists()) {
                path.mkdirs();
            }
            List<String> idList = Arrays.asList(ids.split(","));
            List<BizAccessory> accessories = this.listByIds(idList);
            if (accessories.size() == 1) {
                BizAccessory accessory = accessories.get(0);
                String url = accessory.getUrl();

                String newUrl = url.substring(0, url.lastIndexOf("&"));
                String substring = newUrl.substring(newUrl.lastIndexOf("=")+1);

                FileOutputStream out = new FileOutputStream(path + "/" + accessory.getName());
                MinioUtil.downFile(customBucket, bizPath + "/" + substring, out, false);
                HttpUtil.downLoadFile(response, filePath + "/" + accessory.getName(), accessory.getName());
                FileUtil.deleteFile(filePath + "/" + accessory.getName());
                return;
            }

            accessories.forEach(s -> {
                String url = s.getUrl();
                String newUrl = url.substring(0, url.lastIndexOf("&"));
                String substring = newUrl.substring(newUrl.lastIndexOf("=")+1);
                FileOutputStream out = null;
                try {
                    out = new FileOutputStream(path + "/" + s.getName());
                } catch (FileNotFoundException e) {
                    e.printStackTrace();
                }
                MinioUtil.downFile(customBucket, bizPath + "/" + substring, out, false);

            });
            this.zip(filePath, true);
            String zipPath = filePath + ".zip";
            HttpUtil.downLoadFile(response, zipPath, zipPath.substring(zipPath.lastIndexOf("/") + 1));
            // 删除导出的文件
            FileUtil.deleteFile(zipPath);
        }

    }

    @Override
    public void minIoDownFileBySourceId(String sourceIds, String bizPath, HttpServletResponse response) throws Exception {
        String customBucket = "cgpt", filePath = "download/" + bizPath;
        if (StringUtils.isNotBlank(sourceIds)) {
            File path = new File(filePath);
            if (!path.exists()) {
                path.mkdirs();
            }
            List<String> idList = Arrays.asList(sourceIds.split(","));
            LambdaQueryWrapper<BizAccessory> wrapper = new LambdaQueryWrapper<BizAccessory>()
                    .in(BizAccessory::getSourceId, idList);
            List<BizAccessory> accessories = this.list(wrapper);
            if (accessories.size() == 1) {
                BizAccessory accessory = accessories.get(0);
                String url = accessory.getUrl();

                String newUrl = url.substring(0, url.lastIndexOf("&"));
                String substring = newUrl.substring(newUrl.lastIndexOf("=")+1);

                FileOutputStream out = new FileOutputStream(path + "/" + accessory.getName());
                MinioUtil.downFile(customBucket, bizPath + "/" + substring, out, false);
                HttpUtil.downLoadFile(response, filePath + "/" + accessory.getName(), accessory.getName());
                FileUtil.deleteFile(filePath + "/" + accessory.getName());
                return;
            }

            accessories.forEach(s -> {
                String url = s.getUrl();
                String newUrl = url.substring(0, url.lastIndexOf("&"));
                String substring = newUrl.substring(newUrl.lastIndexOf("=")+1);
                FileOutputStream out = null;
                try {
                    out = new FileOutputStream(path + "/" + s.getName());
                } catch (FileNotFoundException e) {
                    e.printStackTrace();
                }
                MinioUtil.downFile(customBucket, bizPath + "/" + substring, out, false);

            });
            this.zip(filePath, true);
            String zipPath = filePath + ".zip";
            HttpUtil.downLoadFile(response, zipPath, zipPath.substring(zipPath.lastIndexOf("/") + 1));
            // 删除导出的文件
            FileUtil.deleteFile(zipPath);
        }
    }

    /**********************下载文件************************/
    @Override
    public void minIoDown(String name, String bizPath, HttpServletResponse response) throws Exception {
        String customBucket = "cgpt", filePath = "download/" + bizPath;
        if (StringUtils.isNotBlank(name)) {
            File path = new File(filePath);
            if (!path.exists()) {
                path.mkdirs();
            }
            LambdaQueryWrapper<BizAccessory> accessoryQW = new LambdaQueryWrapper<>();
            accessoryQW.like(BizAccessory::getUrl,name)
                    .orderByDesc(BizAccessory::getCreateTime)
                    .last("limit 1");
            BizAccessory accessory = baseMapper.selectOne(accessoryQW);
            if(BeanUtils.isEmpty(accessory)){
                throw new BaseException("文件未找到");
            }

            FileOutputStream out = new FileOutputStream(path + "/" + accessory.getName());
            MinioUtil.downFile(customBucket, bizPath + "/" + name, out, false);
            HttpUtil.downLoadFile(response, filePath + "/" + accessory.getName(), accessory.getName());
            FileUtil.deleteFile(filePath + "/" + accessory.getName());
            return;
        }

    }

    /**********************下载文件************************/
    @Override
    public void minIoDown(String name, String bizPath,String bucketName, HttpServletResponse response) throws Exception {
        String customBucket = "cgpt";
        if(BeanUtils.isNotEmpty(bucketName)){
            customBucket = bucketName;
        }

        String fileName = name;

        String filePath = "download/" + bizPath;
        if (StringUtils.isNotBlank(name)) {
            File path = new File(filePath);
            if (!path.exists()) {
                path.mkdirs();
            }

            LambdaQueryWrapper<BizAccessory> accessoryQW = new LambdaQueryWrapper<>();
            accessoryQW.like(BizAccessory::getUrl,name)
                    .orderByDesc(BizAccessory::getCreateTime)
                    .last("limit 1");
            BizAccessory accessory = baseMapper.selectOne(accessoryQW);

            if(BeanUtils.isNotEmpty(accessory)){
                fileName = accessory.getName();
            }

            FileOutputStream out = new FileOutputStream(path + "/" + fileName);
            MinioUtil.downFile(customBucket, bizPath+ "/" + name, out, false);
            HttpUtil.downLoadFile(response, filePath + "/" + fileName, fileName);
            FileUtil.deleteFile(filePath + "/" + fileName);
            return;
        }

    }


    /**********************下载文件************************/
    @Override
    public FileOutputStream minIoDownFile(String id, String bizPath) throws Exception {
        String customBucket = "cgpt", filePath = "download/" + bizPath;
        File path = new File(filePath);
        if (!path.exists()) {
            path.mkdirs();
        }
        BizAccessory accessory = this.getById(id);
        String url = accessory.getUrl();
        String newUrl = url.substring(0, url.lastIndexOf("&"));
        String substring = newUrl.substring(newUrl.lastIndexOf("=")+1);
        FileOutputStream out = new FileOutputStream(path + "/" + accessory.getName());
        MinioUtil.downFile(customBucket, bizPath + "/" + substring, out, false);
        return out;
    }

    @Override
    public InputStream minIoGetFile(String id, String bizPath) throws Exception {
        String customBucket = "cgpt", filePath = "download/" + bizPath;
        File path = new File(filePath);
        if (!path.exists()) {
            path.mkdirs();
        }
        BizAccessory accessory = this.getById(id);
        String url = accessory.getUrl();
        String newUrl = url.substring(0, url.lastIndexOf("&"));
        String substring = newUrl.substring(newUrl.lastIndexOf("=")+1);
        InputStream inputSteam = MinioUtil.getInputSteam(customBucket, bizPath + "/" + substring, false);
        return inputSteam;
    }

    @Override
    public InputStream minIoGetFileByUrl(String url, String bizPath) throws Exception {
        String customBucket = "cgpt", filePath = "download/" + bizPath;
        File path = new File(filePath);
        if (!path.exists()) {
            path.mkdirs();
        }
        String newUrl = url.lastIndexOf("&") != -1 ? url.substring(0, url.lastIndexOf("&")) : url;
        String substring = newUrl.substring(newUrl.lastIndexOf("=")+1);
        return MinioUtil.getInputSteam(customBucket, bizPath + "/" + substring, false);
    }


    private void zip(String path, Boolean isDelete) {
        ZipFile zipFile = null;
        try {
            ZipParameters parameters = new ZipParameters();
            parameters.setCompressionMethod(Zip4jConstants.COMP_DEFLATE);
            parameters.setCompressionLevel(Zip4jConstants.DEFLATE_LEVEL_NORMAL);
            File file = new File(path);
            Assert.isTrue(file.exists(), "文件不存在");
            ArrayList<String> list = new ArrayList<>();
            if (file.isDirectory()) {
                zipFile = new ZipFile(new File(path + ".zip"));
                zipFile.setFileNameCharset("utf-8");
                zipFile.addFolder(path, parameters);
            } else {
                zipFile = new ZipFile(new File(path.split(".")[0] + ".zip"));
                zipFile.setFileNameCharset("utf-8");
                zipFile.addFile(file, parameters);
            }
            if (isDelete) {
                FileUtil.deleteDir(file);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
    /**
     * 合并文件
     *
     * @param fileName 文件名
     * @return 合并后的文件
     * @throws FileNotFoundException 找不到文件
     */
    private File mergeFile(String fileName) throws FileNotFoundException {
        String dirPath = utils.getFileDir(fileName);
        // 创建合并后的文件对象，如果这个文件存在，将其删除（后续会在RandomAccessFile中重新构建），确保fs数组读取到的只有分片碎片文件，
        File result = new File(dirPath + "/" + fileName);
        if (result.exists()) {
            result.delete();
        }
        // 获取路径下的所有文件，将其转变为List<RandomAccessFile>
        File[] fs = new File(dirPath).listFiles();
        List<File> fileList = Arrays.asList(fs);
        Collections.sort(fileList, (o1, o2) -> {
            return Integer.parseInt(o1.getName()) - Integer.parseInt(o2.getName());
        });
        List<RandomAccessFile> rafList = new ArrayList<>();
        for (File ele : fileList) {
            RandomAccessFile tempFile = new RandomAccessFile(ele, "rw");
            rafList.add(tempFile);
        }
        try (
                // 开始合并文件，这里的资源会自动释放
                // 输出管道
                FileChannel outputChannel = new RandomAccessFile(result, "rw").getChannel();
        ) {
            // 遍历分片文件集合
            for (RandomAccessFile ele : rafList) {
                // 获取单个文件的输出管道
                FileChannel inputChannel = ele.getChannel();
                // 从输入管道写入数据到输出管道——输入管道、偏移量、写入长度
                outputChannel.transferFrom(inputChannel, outputChannel.size(), inputChannel.size());
                //释放资源
                inputChannel.close();
            }
            return result;
        } catch (IOException e) {
            e.printStackTrace();
        }
        return null;
    }

//    @Override
    ////    public void updateProjectIdByProiectId(String oldProjectIds, String newProjectId) {
//
//    }

    @Override
    public String fileToPdf(BizAccessory accessory) {
        String suffix = accessory.getSuffix();
        if("pdf".equalsIgnoreCase(suffix)){
            return null;
        }
        String url = accessory.getUrl();
        String[] urlArr = url.split("/");
        try {
            String downFilePath = minIoDownFile(accessory, urlArr[4]);
            String fileName = accessory.getName();
            String inputPath = downFilePath + fileName;
            String outputPath = downFilePath + fileName + ".pdf";
            int i = fileName.lastIndexOf(".");
            if ((i > -1) && (i < (fileName.length() - 1))) {
                outputPath =  downFilePath + fileName.substring(0, i)  + ".pdf";
            }
            officeToPdf.office2pdf(inputPath, outputPath);
            return outputPath;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    @Override
    public List<BizAccessory> getAccessoryByGroup(String group) {
        QueryFilter<BizAccessory> queryFilter = QueryFilter.build();
        queryFilter.addFilter("GROUP_", group, QueryOP.EQUAL);
        queryFilter.addFilter("IS_DELE_", "0", QueryOP.EQUAL);
        return this.baseMapper.selectPageBySourceId(
                        convert2IPage(new PageBean(0, -1, false)),
                        convert2Wrapper(queryFilter, currentModelClass())).
                getRecords();
    }


    @Override
    public BizAccessory minioUpload(MultipartFile file, String bizPath , HttpServletResponse response) throws Exception {
        BizAccessory accessory = new BizAccessory();
        String fileName = null;
        if (file.isEmpty() || file.getSize() == 0) {
            throw new BaseException("文件为空");
        }
        if (StringUtils.isBlank(fileName)) {
            fileName = file.getOriginalFilename();
        }
        accessory.setName(fileName);
        String name = DigestUtils.md5Hex(UUID.randomUUID().toString()) + fileName.substring(fileName.lastIndexOf("."));
        //如果为空放到默认文件夹
        if (StringUtils.isEmpty(bizPath)){
            bizPath="default";
        }
        String suffix = fileName.substring(fileName.lastIndexOf(".") + 1);
        accessory.setSuffix(suffix);

        List<String> suffixList = Arrays.asList("png,jpg,jpeg,xlsx,xls,pdf,docx,doc,zip,rar,excel,pptx,stp".split(","));
        if(!suffixList.contains(suffix.toLowerCase())){
            throw new BaseException("该文件类型不能上传");
        }

        String url = MinioUtil.getMinIoUrl(bizPath, name);
        // URL MinIO 异步上传
        accessory.setUrl(url);
        String userId = ContextUtil.getCurrentUserId();
        String userName = ContextUtil.getCurrentUserName();
        accessory.setCreateBy(userId);
        accessory.setUpdateBy(userId);
        accessory.setCreateName(userName);
        accessory.setUpdateName(userName);
        accessory.setCreateTime(LocalDateTime.now());
        accessory.setUpdateTime(LocalDateTime.now());
//        accessory.setIsDele("0");
        accessory.setType(handleFileType(accessory.getSuffix()));
        accessory.setFileSize(new DecimalFormat("#,##0.##").format(file.getSize() / 1024.00 / 1024.00) + " MB");
        utils.startMinIOUpload(file, bizPath, name);
        accessory.setBidSignStatus(1);
        return accessory;
    }

    @Override
    public BizAccessory minioUpload(MultipartFile file, String bizPath ,String bucketName, HttpServletResponse response) throws Exception {
        BizAccessory accessory = new BizAccessory();
        String fileName = null;
        if (file.isEmpty() || file.getSize() == 0) {
            throw new BaseException("文件为空");
        }
        if (StringUtils.isBlank(fileName)) {
            fileName = file.getOriginalFilename();
        }
        accessory.setName(fileName);
        String name = DigestUtils.md5Hex(UUID.randomUUID().toString()) + fileName.substring(fileName.lastIndexOf("."));
        //如果为空放到默认文件夹
        if (StringUtils.isEmpty(bizPath)){
            bizPath="default";
        }
        accessory.setSuffix(fileName.substring(fileName.lastIndexOf(".") + 1));

        List<String> suffixList = Arrays.asList("png,jpg,jpeg,xlsx,xls,pdf,docx,doc,zip,rar,excel,pptx,stp".split(","));
        if(!suffixList.contains(accessory.getSuffix())){
            throw new BaseException("该文件类型不能上传");
        }

        String url = MinioUtil.getMinIoUrl(bizPath,name);
        // URL MinIO 异步上传
        accessory.setUrl(url);
        String userId = ContextUtil.getCurrentUserId();
        String userName = ContextUtil.getCurrentUserName();
        accessory.setCreateBy(userId);
        accessory.setUpdateBy(userId);
        accessory.setCreateName(userName);
        accessory.setUpdateName(userName);
        accessory.setCreateTime(LocalDateTime.now());
        accessory.setUpdateTime(LocalDateTime.now());
        accessory.setType(handleFileType(accessory.getSuffix()));
        accessory.setFileSize(new DecimalFormat("#,##0.##").format(file.getSize() / 1024.00 / 1024.00) + " MB");
        utils.startMinIOUpload(file, bizPath, name,bucketName);
        accessory.setBidSignStatus(1);
        return accessory;
    }

    @Override
    public BizAccessory minioUpload(MultipartFile file, String bizPath) throws Exception {
        BizAccessory accessory = new BizAccessory();
        String fileName = file.getOriginalFilename();
        if (file.isEmpty() || file.getSize() == 0) {
            throw new BaseException("文件为空");
        }
        accessory.setName(fileName);
        String name = DigestUtils.md5Hex(UUID.randomUUID().toString()) + fileName.substring(fileName.lastIndexOf("."));
        //如果为空放到默认文件夹
        if (StringUtils.isEmpty(bizPath)){
            bizPath="default";
        }
        String suffix = fileName.substring(fileName.lastIndexOf(".") + 1);
        accessory.setSuffix(suffix);

        List<String> suffixList = Arrays.asList("png,jpg,jpeg,xlsx,xls,pdf,docx,doc,zip,rar,excel,pptx".split(","));
        if(!suffixList.contains(suffix.toLowerCase())){
            throw new BaseException("该文件类型不能上传");
        }

        String url = MinioUtil.getMinIoUrl(bizPath, name);
        // URL MinIO 异步上传
        accessory.setUrl(url);

        String userId = ContextUtil.getCurrentUserId();
        String userName = ContextUtil.getCurrentUserName();
        accessory.setCreateBy(userId);
        accessory.setUpdateBy(userId);
        accessory.setCreateName(userName);
        accessory.setUpdateName(userName);
        accessory.setCreateTime(LocalDateTime.now());
        accessory.setUpdateTime(LocalDateTime.now());
        accessory.setType(handleFileType(accessory.getSuffix()));
        accessory.setFileSize(new DecimalFormat("#,##0.##").format(file.getSize() / 1024.00 / 1024.00) + " MB");
        utils.startMinIOUpload(file, bizPath, name);
        return accessory;
    }

    @Override
    public BizAccessory getFileById(String id) {
        return baseMapper.selectById(id);
    }

    @Override
    public BizAccessory getFileByCreateBy(String id) {
        return baseMapper.selectByCreatorId(id);
    }

    public static Integer handleFileType(String suffix) {
        if ("bmp,jpg,jpeg,png,gif,webp".indexOf(suffix) != -1) {
            return 1;
        }
        if ("docx,doc,xls,xlsx,ppt,pptx,pdf,htl,html,txt".indexOf(suffix) != -1) {
            return 2;
        }
        if ("rar,zip,7z,gz,bz,ace,uha,uda,zpaq".indexOf(suffix) != -1) {
            return 3;
        }
        if ("avi,wmv,mpg,mpeg,mov,rm,ram,swf,flv,mp4".indexOf(suffix) != -1) {
            return 4;
        } else {
            return 5;
        }
    }
}
