package com.artfess.gyl.supplierEva.manager.impl;

import com.artfess.base.exception.ApplicationException;
import com.artfess.base.manager.impl.BaseManagerImpl;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.file.model.DefaultFile;
import com.artfess.file.persistence.manager.FileManager;
import com.artfess.gyl.supplierEva.dao.BizSupplierPerformanceEvaluationDao;
import com.artfess.gyl.supplierEva.dto.EvaluationDTO;
import com.artfess.gyl.supplierEva.dto.ResultEvaDTO;
import com.artfess.gyl.supplierEva.manager.*;
import com.artfess.gyl.supplierEva.model.*;
import com.artfess.gyl.supplierEva.vo.*;
import com.artfess.uc.api.impl.util.ContextUtil;
import com.artfess.uc.api.model.IUser;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.http.util.Asserts;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

/**
 * 供应商绩效考核评价表 服务实现类
 *
 * @author huqi
 * @since 2025-11-12
 */
@Service
public class BizSupplierPerformanceEvaluationManagerImpl extends BaseManagerImpl<BizSupplierPerformanceEvaluationDao, BizSupplierPerformanceEvaluation> implements BizSupplierPerformanceEvaluationManager {

    // 依赖注入
    @Resource
    private BizSupplierPerformanceEvaluationRatingManager ratingManager;
    @Resource
    private BizSupplierPerformanceEvaluationResultManager resultManager;
    @Resource
    private BizSupplierPerformanceEvaluationRatingUserManager ratingUserManager;
    @Resource
    private BizSupplierPerformanceEvaluationIntervalManager intervalManager;
    @Resource
    private BizSupplierPerformanceEvaluationNotificationManager notificationManager;
    @Resource
    private BizSupplierPerformanceEvaluationApprovalRecordManager recordManager;
    @Resource
    private BizSupplierPerformanceEvaluationConfigurationUserManager supplierManager;
    @Resource
    private FileManager fileManager;

    // 定义常量
    private static final List<String> COMPLETED_LEVELS = Arrays.asList("1", "2", "3");
    private static final String STATUS_INIT = "0";
    private static final String STATUS_COMPLETED = "1";
    private static final String STATUS_SUBMITTED = "2";
    private static final String STATUS_PARTIAL_COMPLETED = "3";
    private static final String STATUS_ALL_COMPLETED = "4";


    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean batchCreateEvaluationFromTemplate(EvaluationDTO dto) {
        IUser user = getCurrentUser();
        validateEvaluationData(dto);

        List<BizSupplierPerformanceEvaluationConfigurationUser> supplierList = getSupplierList(dto);
        if (CollectionUtils.isEmpty(supplierList)) {
            throw new ApplicationException("无合格供应商进行分发");
        }

        initEvaluationData(dto, user);
        boolean saveResult = super.save(dto);
        if (!saveResult) {
            return false;
        }

        saveEvaluationResults(dto, supplierList);
        return true;
    }

    /**
     * 获取供应商列表
     */
    private List<BizSupplierPerformanceEvaluationConfigurationUser> getSupplierList(EvaluationDTO dto) {
        LambdaQueryWrapper<BizSupplierPerformanceEvaluationConfigurationUser> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(BizSupplierPerformanceEvaluationConfigurationUser::getValuationConfigurationId,
                        dto.getValuationConfigurationId())
                .select(
                        BizSupplierPerformanceEvaluationConfigurationUser::getSupplierId,
                        BizSupplierPerformanceEvaluationConfigurationUser::getSupplierName,
                        BizSupplierPerformanceEvaluationConfigurationUser::getSupplierCode
                );
        return supplierManager.list(wrapper);
    }

    /**
     * 保存评估结果
     */
    private void saveEvaluationResults(EvaluationDTO dto,
                                       List<BizSupplierPerformanceEvaluationConfigurationUser> supplierList) {
        int totalScore = calculateTotalScore(dto.getDetails());
        List<BizSupplierPerformanceEvaluationResult> results = createEvaluationResults(dto, supplierList, totalScore);
        resultManager.saveBatch(results);

        // 为每个供应商保存评分明细
        results.forEach(result ->
                saveRatingDetailsAndUsers(dto.getId(), result.getId(), dto.getDetails())
        );
    }

    /**
     * 计算总分
     */
    private int calculateTotalScore(List<BizSupplierPerformanceEvaluationRating> details) {
        AtomicReference<Integer> total = new AtomicReference<>(0);
        details.forEach(detail -> {
            if (detail.getScore() != null) {
                total.updateAndGet(v -> v + detail.getScore());
            }
        });
        return total.get();
    }

    /**
     * 创建评估结果列表
     */
    private List<BizSupplierPerformanceEvaluationResult> createEvaluationResults(
            EvaluationDTO dto,
            List<BizSupplierPerformanceEvaluationConfigurationUser> supplierList,
            int totalScore) {
        return supplierList.stream()
                .map(item -> createEvaluationResult(dto, item, totalScore))
                .collect(Collectors.toList());
    }

    /**
     * 创建单个评估结果
     */
    private BizSupplierPerformanceEvaluationResult createEvaluationResult(
            EvaluationDTO dto,
            BizSupplierPerformanceEvaluationConfigurationUser item,
            int totalScore) {
        BizSupplierPerformanceEvaluationResult result = new BizSupplierPerformanceEvaluationResult();
        result.setValuationConfigurationId(dto.getValuationConfigurationId());
        result.setEvaluationId(dto.getId());
        result.setSupplierId(item.getSupplierId());
        result.setSupplierName(item.getSupplierName());
        result.setSupplierCode(item.getSupplierCode());
        result.setTotalScore(totalScore);
        result.setStatus(STATUS_SUBMITTED);
        return result;
    }

    /**
     * 获取当前用户
     */
    private IUser getCurrentUser() {
        IUser user = ContextUtil.getCurrentUser();
        if (user == null) {
            throw new ApplicationException("未找到用户信息");
        }
        return user;
    }

    /**
     * 验证评估数据
     */
    private void validateEvaluationData(EvaluationDTO dto) {
        Asserts.notNull(dto, "考核数据不能为空");
        Asserts.notNull(dto.getDetails(), "考核详情不能为空");
    }

    /**
     * 初始化评估数据
     */
    private void initEvaluationData(EvaluationDTO dto, IUser user) {
        dto.setInitiatorId(user.getUserId());
        dto.setInitiatorAccount(user.getAccount());
        dto.setInitiatorName(user.getFullname());
        dto.setStatus(STATUS_SUBMITTED);
    }

    /**
     * 保存评分明细和用户
     */
    private void saveRatingDetailsAndUsers(String evaluationId, String resultId,
                                           List<BizSupplierPerformanceEvaluationRating> details) {
        List<BizSupplierPerformanceEvaluationRating> savedDetails = saveRatingDetails(evaluationId, resultId, details);
        saveRatingUsers(evaluationId, savedDetails);
    }

    /**
     * 保存评分明细
     */
    private List<BizSupplierPerformanceEvaluationRating> saveRatingDetails(String evaluationId, String resultId,
                                                                           List<BizSupplierPerformanceEvaluationRating> details) {
        details.forEach(item -> {
            item.setId(null);
            item.setPerformanceEvaluationId(evaluationId);
            item.setPerformanceEvaluationResultId(resultId);
            item.setStatus(STATUS_INIT);
        });

        if (!ratingManager.saveBatch(details)) {
            throw new ApplicationException("评分明细保存失败");
        }
        return details;
    }

    /**
     * 保存评分用户
     */
    private void saveRatingUsers(String evaluationId, List<BizSupplierPerformanceEvaluationRating> savedDetails) {
        List<BizSupplierPerformanceEvaluationRatingUser> userList = savedDetails.stream()
                .filter(item -> CollectionUtils.isNotEmpty(item.getUsers()))
                .flatMap(item -> createRatingUsers(evaluationId, item))
                .collect(Collectors.toList());

        if (CollectionUtils.isNotEmpty(userList) && !ratingUserManager.saveBatch(userList)) {
            throw new ApplicationException("评分用户保存失败");
        }
    }

    /**
     * 创建评分用户列表
     */
    private java.util.stream.Stream<BizSupplierPerformanceEvaluationRatingUser> createRatingUsers(
            String evaluationId, BizSupplierPerformanceEvaluationRating item) {
        return item.getUsers().stream()
                .map(sourceUser -> createRatingUser(evaluationId, item, sourceUser));
    }

    /**
     * 创建单个评分用户
     */
    private BizSupplierPerformanceEvaluationRatingUser createRatingUser(
            String evaluationId,
            BizSupplierPerformanceEvaluationRating item,
            BizSupplierPerformanceEvaluationRatingUser sourceUser) {
        BizSupplierPerformanceEvaluationRatingUser user = new BizSupplierPerformanceEvaluationRatingUser();
        user.setEvaluationRatingId(item.getId());
        user.setPerformanceEvaluationId(evaluationId);
        user.setRatingId(sourceUser.getRatingId());
        user.setRatingAccount(sourceUser.getRatingAccount());
        user.setRatingName(sourceUser.getRatingName());
        user.setScore(item.getScore());
        user.setStatus(STATUS_INIT);
        return user;
    }

    @Override
    @Transactional(readOnly = true)
    public EvaluationResultVo getModelById(String id) {
        EvaluationResultVo vo = resultManager.getModel(id);
        if (vo == null) {
            return null;
        }
        loadRelatedData(id, vo);
        return vo;
    }

    /**
     * 加载相关数据
     */
    private void loadRelatedData(String resultId, EvaluationResultVo vo) {
        List<BizSupplierPerformanceEvaluationRating> ratings = loadRatingDetails(resultId);
        List<BizSupplierPerformanceEvaluationNotification> notifications = loadNotifications(resultId);
        List<BizSupplierPerformanceEvaluationApprovalRecord> records = getRecords(resultId);

        setRatingDetails(vo, ratings);
        setNotificationInfo(vo, notifications);
        setApprovalRecords(vo, records);
    }

    /**
     * 设置评分明细
     */
    private void setRatingDetails(EvaluationResultVo vo, List<BizSupplierPerformanceEvaluationRating> ratings) {
        if (CollectionUtils.isNotEmpty(ratings)) {
            loadRatingUsers(ratings);
            vo.setDetails(ratings);
        }
    }

    /**
     * 设置通知信息
     */
    private void setNotificationInfo(EvaluationResultVo vo,
                                     List<BizSupplierPerformanceEvaluationNotification> notifications) {
        if (CollectionUtils.isNotEmpty(notifications)) {
            vo.setNotification(notifications.get(0));
            loadCorrectionReportIfNeeded(vo, notifications.get(0));
        }
    }

    /**
     * 设置审批记录
     */
    private void setApprovalRecords(EvaluationResultVo vo,
                                    List<BizSupplierPerformanceEvaluationApprovalRecord> records) {
        if (CollectionUtils.isNotEmpty(records)) {
            vo.setRecords(records);
        }
    }

    /**
     * 加载评分明细
     */
    private List<BizSupplierPerformanceEvaluationRating> loadRatingDetails(String resultId) {
        return ratingManager.list(new LambdaQueryWrapper<BizSupplierPerformanceEvaluationRating>()
                .eq(BizSupplierPerformanceEvaluationRating::getPerformanceEvaluationResultId, resultId));
    }

    /**
     * 加载评分用户
     */
    private void loadRatingUsers(List<BizSupplierPerformanceEvaluationRating> ratings) {
        List<String> ratingIds = ratings.stream()
                .map(BizSupplierPerformanceEvaluationRating::getId)
                .collect(Collectors.toList());

        if (CollectionUtils.isEmpty(ratingIds)) {
            return;
        }

        List<BizSupplierPerformanceEvaluationRatingUser> ratingUsers = ratingUserManager.list(
                new LambdaQueryWrapper<BizSupplierPerformanceEvaluationRatingUser>()
                        .in(BizSupplierPerformanceEvaluationRatingUser::getEvaluationRatingId, ratingIds));

        if (CollectionUtils.isNotEmpty(ratingUsers)) {
            Map<String, List<BizSupplierPerformanceEvaluationRatingUser>> userMap = ratingUsers.stream()
                    .collect(Collectors.groupingBy(BizSupplierPerformanceEvaluationRatingUser::getEvaluationRatingId));
            ratings.forEach(rating -> rating.setUsers(userMap.get(rating.getId())));
        }
    }

    /**
     * 加载通知信息
     */
    private List<BizSupplierPerformanceEvaluationNotification> loadNotifications(String performanceEvaluationId) {
        return notificationManager.list(new LambdaQueryWrapper<BizSupplierPerformanceEvaluationNotification>()
                .eq(BizSupplierPerformanceEvaluationNotification::getPerformanceEvaluationId, performanceEvaluationId));
    }

    /**
     * 加载整改报告（如果需要）
     */
    private void loadCorrectionReportIfNeeded(EvaluationResultVo vo,
                                              BizSupplierPerformanceEvaluationNotification notification) {
        if (vo.getProcessType() != null && "2".equals(vo.getProcessType())) {
            BizSupplierPerformanceEvaluationCorrectionReport report = notificationManager.getReport(notification.getId());
            if (report != null) {
                vo.setReport(report);
                loadReportFiles(report);
            }
        }
    }

    /**
     * 加载报告文件
     */
    private void loadReportFiles(BizSupplierPerformanceEvaluationCorrectionReport report) {
        List<DefaultFile> files = fileManager.queryFileByBizId(report.getId());
        report.setFiles(files);
    }

    @Override
    @Transactional(readOnly = true)
    public EvaluationOptVo getModelByUser(String id) {
        BizSupplierPerformanceEvaluation evaluation = this.get(id);
        if (evaluation == null) {
            return null;
        }
        EvaluationOptVo vo = new EvaluationOptVo(evaluation);
        IUser user = getCurrentUser();
        List<BizSupplierPerformanceEvaluationRating> ratings = ratingManager.getByUser(user.getAccount(), id);
        vo.setDetails(ratings);
        return vo;
    }

    @Override
    @Transactional(readOnly = true)
    public List<SupplierScore> getUserRatings(String evaluationId, String detailsId) {
        IUser user = ContextUtil.getCurrentUser();
        return baseMapper.getSupplierScores(user.getAccount(), evaluationId, detailsId);
    }

    @Override
    @Transactional(readOnly = true)
    public List<SupplierScore> getUserEvaluations(String ratingId) {
        return baseMapper.getUserEvaluations(ratingId);
    }

    @Override
    @Transactional(readOnly = true)
    public PageList<EvaluationVo> queryEvaluationList(QueryFilter<BizSupplierPerformanceEvaluation> queryFilter) {
        IPage<EvaluationVo> iPage = baseMapper.queryByEva(convert2IPage(queryFilter.getPageBean()),
                convert2Wrapper(queryFilter, currentModelClass()));
        return new PageList<>(iPage);
    }

    @Override
    @Transactional(readOnly = true)
    public PageList<EvaluationVo> queryEvaluationListByUser(QueryFilter<BizSupplierPerformanceEvaluation> queryFilter, String account) {
        IPage<EvaluationVo> iPage = baseMapper.queryByUser(convert2IPage(queryFilter.getPageBean()),
                convert2Wrapper(queryFilter, currentModelClass()), account);
        return new PageList<>(iPage);
    }

    @Override
    @Transactional(readOnly = true)
    public PageList<EvaluationVo> queryByPersonalSum(QueryFilter<BizSupplierPerformanceEvaluation> queryFilter) {
        IPage<EvaluationVo> iPage = baseMapper.queryByPersonalSum(convert2IPage(queryFilter.getPageBean()),
                convert2Wrapper(queryFilter, currentModelClass()));
        return new PageList<>(iPage);
    }

    @Override
    @Transactional(readOnly = true)
    public PageList<CorePerformanceVo> queryBySupplierLevel(QueryFilter<BizSupplierPerformanceEvaluation> queryFilter) {
        IPage<CorePerformanceVo> iPage = baseMapper.countSupplierLevel(convert2IPage(queryFilter.getPageBean()),
                convert2Wrapper(queryFilter, currentModelClass()));
        return new PageList<>(iPage);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean saveUserRating(List<BizSupplierPerformanceEvaluationRatingUser> ratingList) {
        // 1. 参数校验
        if (CollectionUtils.isEmpty(ratingList)) {
            throw new ApplicationException("评分列表不能为空");
        }

        // 2. 校验第一个评分项
        BizSupplierPerformanceEvaluationRatingUser firstRating = ratingList.get(0);
        String id = firstRating.getId();

        // 使用主键查询，性能更好
        BizSupplierPerformanceEvaluationRatingUser user = ratingUserManager.getById(id);
        if (user == null) {
            throw new ApplicationException("未找到评分项");
        }

        if (STATUS_SUBMITTED.equals(user.getStatus())) {
            throw new ApplicationException("已提交评分，无法再进行修改");
        }

        // 3. 准备批量更新数据
        LocalDateTime now = LocalDateTime.now();
        List<String> ratingIds = new ArrayList<>(ratingList.size());

        ratingList.forEach(item -> {
            item.setStatus(STATUS_COMPLETED);
            item.setRatingTime(now);
            ratingIds.add(item.getEvaluationRatingId());
        });

        // 4. 批量更新评分用户记录
        if (!ratingUserManager.updateBatchById(ratingList)) {
            throw new ApplicationException("评分用户更新失败");
        }

        // 5. 更新关联的评分结果状态（优化查询）
        List<BizSupplierPerformanceEvaluationRating> ratings = ratingManager.list(
                new LambdaQueryWrapper<BizSupplierPerformanceEvaluationRating>()
                        .select(BizSupplierPerformanceEvaluationRating::getPerformanceEvaluationResultId)
                        .in(BizSupplierPerformanceEvaluationRating::getId, ratingIds)
        );

        List<String> resultIds = ratings.stream()
                .map(BizSupplierPerformanceEvaluationRating::getPerformanceEvaluationResultId)
                .distinct()
                .collect(Collectors.toList());

        if (!resultIds.isEmpty()) {
            boolean updateResult = resultManager.update(
                    new LambdaUpdateWrapper<BizSupplierPerformanceEvaluationResult>()
                            .set(BizSupplierPerformanceEvaluationResult::getStatus, STATUS_PARTIAL_COMPLETED)
                            .in(BizSupplierPerformanceEvaluationResult::getId, resultIds)
            );
            if (!updateResult) {
                throw new ApplicationException("修改评分结果失败");
            }

            // 6. 更新评分配置状态
            boolean updateEvaluation = lambdaUpdate()
                    .set(BizSupplierPerformanceEvaluation::getStatus, STATUS_PARTIAL_COMPLETED)
                    .eq(BizSupplierPerformanceEvaluation::getId, user.getPerformanceEvaluationId())
                    .update();

            if (!updateEvaluation) {
                throw new ApplicationException("更新评分配置失败");
            }
        }

        return true;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean submitUserRating(String evaId) {
        IUser user = ContextUtil.getCurrentUser();

        List<BizSupplierPerformanceEvaluationRatingUser> supplierUser = ratingUserManager
                .getSupplierByEvaId(evaId, null, user.getAccount());

        if (CollectionUtils.isEmpty(supplierUser)) {
            throw new ApplicationException("当前用户无可提交的评分信息");
        }

        boolean hasUnratedItem = supplierUser.stream()
                .anyMatch(s -> STATUS_INIT.equals(s.getStatus()));
        if (hasUnratedItem) {
            throw new ApplicationException("当前用户存在未评审的得分项无法提交");
        }

        boolean allSubmitted = supplierUser.stream()
                .allMatch(s -> STATUS_SUBMITTED.equals(s.getStatus()));
        if (allSubmitted) {
            throw new ApplicationException("当前用户已提交,请勿二次提交");
        }

        List<String> userIds = supplierUser.stream()
                .map(BizSupplierPerformanceEvaluationRatingUser::getId)
                .collect(Collectors.toList());

        LambdaUpdateWrapper<BizSupplierPerformanceEvaluationRatingUser> updateWrapper = new LambdaUpdateWrapper<>();
        updateWrapper.in(BizSupplierPerformanceEvaluationRatingUser::getId, userIds)
                .set(BizSupplierPerformanceEvaluationRatingUser::getStatus, STATUS_SUBMITTED);

        if (!ratingUserManager.update(updateWrapper)) {
            return false;
        }

        List<BizSupplierPerformanceEvaluationRatingUser> ratingUsers = ratingUserManager.list(
                new LambdaQueryWrapper<BizSupplierPerformanceEvaluationRatingUser>()
                        .eq(BizSupplierPerformanceEvaluationRatingUser::getPerformanceEvaluationId, evaId)
        );

        boolean subMatch = ratingUsers.stream().allMatch(p -> STATUS_SUBMITTED.equals(p.getStatus()));
        if (subMatch) {
            Map<String, BigDecimal> ratingIdToAverageScore = calculateAverageScores(ratingUsers);
            updateRatingItems(ratingIdToAverageScore);
        }
        return true;
    }

    /**
     * 计算平均分
     */
    private Map<String, BigDecimal> calculateAverageScores(List<BizSupplierPerformanceEvaluationRatingUser> ratingUsers) {
        return ratingUsers.stream()
                .filter(r -> STATUS_SUBMITTED.equals(r.getStatus()))
                .filter(r -> r.getEvaluationRatingId() != null)
                .collect(Collectors.groupingBy(
                        BizSupplierPerformanceEvaluationRatingUser::getEvaluationRatingId,
                        Collectors.collectingAndThen(
                                Collectors.toList(),
                                list -> {
                                    if (list.isEmpty()) {
                                        return BigDecimal.ZERO;
                                    }
                                    BigDecimal sum = list.stream()
                                            .map(r -> r.getOperationScore() != null ? r.getOperationScore() : BigDecimal.ZERO)
                                            .reduce(BigDecimal.ZERO, BigDecimal::add);
                                    return sum.divide(BigDecimal.valueOf(list.size()), 2, RoundingMode.HALF_UP);
                                }
                        )
                ));
    }

    /**
     * 更新评分项
     */
    private void updateRatingItems(Map<String, BigDecimal> ratingIdToAverageScore) {
        Set<String> resultIds = new HashSet<>();
        LocalDateTime now = LocalDateTime.now();

        for (Map.Entry<String, BigDecimal> entry : ratingIdToAverageScore.entrySet()) {
            String ratingId = entry.getKey();
            BigDecimal averageScore = entry.getValue();

            LambdaUpdateWrapper<BizSupplierPerformanceEvaluationRating> ratingUpdateWrapper = new LambdaUpdateWrapper<>();
            ratingUpdateWrapper.eq(BizSupplierPerformanceEvaluationRating::getId, ratingId)
                    .set(BizSupplierPerformanceEvaluationRating::getStatus, STATUS_COMPLETED)
                    .set(BizSupplierPerformanceEvaluationRating::getRatingTime, now)
                    .set(BizSupplierPerformanceEvaluationRating::getOperationScore, averageScore);

            if (!ratingManager.update(ratingUpdateWrapper)) {
                throw new ApplicationException("修改评分失败");
            }

            BizSupplierPerformanceEvaluationRating rating = ratingManager.getById(ratingId);
            if (rating != null && StringUtils.isNotBlank(rating.getPerformanceEvaluationResultId())) {
                resultIds.add(rating.getPerformanceEvaluationResultId());
            }
        }

        resultIds.forEach(this::updatePerformanceEvaluations);
    }

    /**
     * 更新绩效评估
     */
    private void updatePerformanceEvaluations(String resultId) {
        BizSupplierPerformanceEvaluationResult result = resultManager.getById(resultId);
        if (result == null) {
            return;
        }

        String currentStatus = result.getStatus();
        if (!STATUS_SUBMITTED.equals(currentStatus) &&
                !STATUS_PARTIAL_COMPLETED.equals(currentStatus)) {
            throw new ApplicationException("已完成评分操作，请刷新页面后进行查看");
        }

        String finalStatus = determineFinalStatus(resultId);
        result.setStatus(STATUS_PARTIAL_COMPLETED);

        if (STATUS_ALL_COMPLETED.equals(finalStatus)) {
            updateCompletedEvaluation(result);
        }

        resultManager.updateById(result);
    }

    /**
     * 确定最终状态
     */
    private String determineFinalStatus(String resultId) {
        List<BizSupplierPerformanceEvaluationRating> unfinishedRatings = getUnfinishedRatings(resultId);
        return unfinishedRatings.isEmpty() ? STATUS_ALL_COMPLETED : STATUS_PARTIAL_COMPLETED;
    }

    /**
     * 更新已完成的绩效评估
     */
    private void updateCompletedEvaluation(BizSupplierPerformanceEvaluationResult evaluation) {
        List<BizSupplierPerformanceEvaluationRating> allRatings = ratingManager.list(
                new LambdaQueryWrapper<BizSupplierPerformanceEvaluationRating>()
                        .eq(BizSupplierPerformanceEvaluationRating::getPerformanceEvaluationResultId, evaluation.getId()));

        if (CollectionUtils.isEmpty(allRatings)) {
            return;
        }

        BigDecimal totalScore = calculateTotalOperationScore(allRatings);
        String level = calculateLevel(evaluation.getValuationConfigurationId(), totalScore);

        if (COMPLETED_LEVELS.contains(level)) {
            evaluation.setStatus(STATUS_COMPLETED);
        } else {
            evaluation.setStatus(STATUS_ALL_COMPLETED);
        }

        evaluation.setOperationScore(totalScore);
        evaluation.setLevel(level);
    }

    /**
     * 计算总操作分数
     */
    private BigDecimal calculateTotalOperationScore(List<BizSupplierPerformanceEvaluationRating> allRatings) {
        if (CollectionUtils.isEmpty(allRatings)) {
            return BigDecimal.ZERO;
        }
        return allRatings.stream()
                .map(BizSupplierPerformanceEvaluationRating::getOperationScore)
                .filter(Objects::nonNull)
                .reduce(BigDecimal.ZERO, BigDecimal::add);
    }

    /**
     * 查询未完成的评分项
     */
    private List<BizSupplierPerformanceEvaluationRating> getUnfinishedRatings(String resultId) {
        return ratingManager.list(new LambdaQueryWrapper<BizSupplierPerformanceEvaluationRating>()
                .eq(BizSupplierPerformanceEvaluationRating::getPerformanceEvaluationResultId, resultId)
                .ne(BizSupplierPerformanceEvaluationRating::getStatus, STATUS_COMPLETED));
    }

    /**
     * 根据总分计算等级
     */
    private String calculateLevel(String configurationId, BigDecimal totalScore) {
        if (StringUtils.isBlank(configurationId) || totalScore == null) {
            return "1";
        }

        String cacheKey = "level_config_" + configurationId;
        Map<String, String> levelCache = getLevelCache(cacheKey);

        double scoreValue = totalScore.doubleValue();
        String scoreKey = String.format("%.2f", scoreValue);

        if (levelCache.containsKey(scoreKey)) {
            return levelCache.get(scoreKey);
        }

        List<BizSupplierPerformanceEvaluationInterval> intervals = intervalManager.list(
                new LambdaQueryWrapper<BizSupplierPerformanceEvaluationInterval>()
                        .eq(BizSupplierPerformanceEvaluationInterval::getPerformanceEvaluationConfigurationId, configurationId)
                        .le(BizSupplierPerformanceEvaluationInterval::getMinScore, scoreValue)
                        .ge(BizSupplierPerformanceEvaluationInterval::getMaxScore, scoreValue));

        String level = intervals.stream()
                .findFirst()
                .map(BizSupplierPerformanceEvaluationInterval::getLevel)
                .orElse("1");

        levelCache.put(scoreKey, level);
        return level;
    }

    /**
     * 获取等级缓存
     */
    private Map<String, String> getLevelCache(String cacheKey) {
        return new ConcurrentHashMap<>();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean submitUserEvaluation(ResultEvaDTO dto) {
        String resultId = dto.getPerformanceEvaluationId();
        BizSupplierPerformanceEvaluationResult result = resultManager.get(resultId);
        if (result == null) {
            throw new ApplicationException("未找到评分结果");
        }
        BizSupplierPerformanceEvaluation evaluation = super.get(result.getEvaluationId());
        if (evaluation == null) {
            throw new ApplicationException("未找到考核模版");
        }
        dto.setLevel(result.getLevel());
        dto.setOperationScore(result.getOperationScore());
        dto.setAssessmentType(evaluation.getAssessmentType());
        dto.setAssessmentYear(evaluation.getAssessmentYear());
        dto.setAssessmentQuarter(evaluation.getAssessmentQuater());
        dto.setAssessmentMonth(evaluation.getAssessmentMonth());
        dto.setValuationConfigurationName(evaluation.getValuationConfigurationName());
        notificationManager.saveEva(dto);

        LambdaUpdateWrapper<BizSupplierPerformanceEvaluationResult> wrapper = new LambdaUpdateWrapper<>();
        wrapper.eq(BizSupplierPerformanceEvaluationResult::getId, resultId)
                .set(BizSupplierPerformanceEvaluationResult::getAssessmentResult, dto.getAssessmentResult())
                .set(BizSupplierPerformanceEvaluationResult::getStatus, STATUS_COMPLETED)
                .set(BizSupplierPerformanceEvaluationResult::getProcessType, dto.getProcessType());
        return resultManager.update(wrapper);
    }

    /**
     * 获取审批记录
     */
    private List<BizSupplierPerformanceEvaluationApprovalRecord> getRecords(String performanceEvaluationId) {
        LambdaQueryWrapper<BizSupplierPerformanceEvaluationApprovalRecord> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(BizSupplierPerformanceEvaluationApprovalRecord::getPerformanceEvaluationId, performanceEvaluationId)
                .groupBy(BizSupplierPerformanceEvaluationApprovalRecord::getApprovalAccount,
                        BizSupplierPerformanceEvaluationApprovalRecord::getStatus)
                .orderByAsc(BizSupplierPerformanceEvaluationApprovalRecord::getCreateTime);
        return recordManager.list(wrapper);
    }

    @Override
    @Transactional
    public boolean remove(List<String> idList) {
        List<BizSupplierPerformanceEvaluationResult> results = resultManager.listByIds(idList);
        if (CollectionUtils.isEmpty(results)) {
            return false;
        }
        boolean anyMatch = results.stream().anyMatch(result -> !result.getStatus().equals(STATUS_SUBMITTED));
        if (anyMatch) {
            throw new ApplicationException("存在已评分的数据,无法删除。");
        }
        return resultManager.removeByIds(idList);
    }

    @Override
    @Transactional
    public boolean updateByIds(List<String> idList, Integer status) {
        LambdaUpdateWrapper<BizSupplierPerformanceEvaluation> evaWrapper = new LambdaUpdateWrapper<>();
        evaWrapper.in(BizSupplierPerformanceEvaluation::getId, idList)
                .set(BizSupplierPerformanceEvaluation::getStatus, status);
        boolean update = super.update(evaWrapper);
        if (!update) {
            return false;
        }
        LambdaUpdateWrapper<BizSupplierPerformanceEvaluationResult> resultWrapper = new LambdaUpdateWrapper<>();
        resultWrapper.in(BizSupplierPerformanceEvaluationResult::getEvaluationId, idList)
                .set(BizSupplierPerformanceEvaluationResult::getStatus, status);
        return resultManager.update(resultWrapper);
    }
}