import { describe, it, expect, vi, beforeEach } from 'vitest'
import { setActivePinia, createPinia } from 'pinia'
import { useUserStore } from '../user'

// Mock API
vi.mock('@/api/system/user', () => ({
    login: vi.fn(),
    loginOut: vi.fn(),
    getCurrentUserDetail: vi.fn()
}))

// Mock cookie utils
vi.mock('@/utils/cookie', () => ({
    getToken: vi.fn(),
    setToken: vi.fn(),
    removeToken: vi.fn()
}))

describe('用户Store测试', () => {
    beforeEach(() => {
        setActivePinia(createPinia())
    })

    it('初始状态正确', () => {
        const store = useUserStore()
        
        expect(store.loginStatus).toBe(false)
        expect(store.userInfo).toBe(null)
        expect(store.userDetail).toBe(null)
        expect(store.isLogin).toBe(false)
    })

    it('登录成功', async () => {
        const { login } = await import('@/api/system/user')
        const { setToken } = await import('@/utils/cookie')
        
        // Mock API 响应
        vi.mocked(login).mockResolvedValue({
            loginStatus: true,
            token: 'mock-token',
            account: 'admin'
        })
        
        const store = useUserStore()
        
        await store.userLogin({ account: 'admin', password: 'password' })
        
        expect(setToken).toHaveBeenCalledWith('mock-token')
        expect(store.loginStatus).toBe(true)
        expect(store.isLogin).toBe(true)
    })

    it('清除用户信息', async () => {
        const { removeToken } = await import('@/utils/cookie')
        const store = useUserStore()
        
        // 设置初始状态
        store.loginStatus = true
        store.userInfo = { name: 'test' } as any
        
        store.clearUser()
        
        expect(removeToken).toHaveBeenCalled()
        expect(store.loginStatus).toBe(false)
        expect(store.userInfo).toBe(null)
    })
})