package com.artfess.examine.consumers;

import com.artfess.base.constants.CodePrefix;
import com.artfess.base.enums.QuestionStateEnum;
import com.artfess.examine.dao.ExamUserRecordDao;
import com.artfess.examine.model.ExamUserRecord;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.redis.connection.Message;
import org.springframework.data.redis.listener.KeyExpirationEventMessageListener;
import org.springframework.data.redis.listener.RedisMessageListenerContainer;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.text.SimpleDateFormat;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.TimeZone;

/**
 * RedisKey键监听以及业务逻辑处理
 *
 * @author: min.wu
 * @date: 2022年11月016日 10:03:49
 */
@Slf4j
@Component
public class RedisTaskListener extends KeyExpirationEventMessageListener {

    @Resource
    private ExamUserRecordDao examUserRecordDao;

    public RedisTaskListener(RedisMessageListenerContainer listenerContainer) {
        super(listenerContainer);
    }

    @Override
    public void onMessage(Message message, byte[] pattern) {
        String expiredKey = message.toString();
        // 只有本业务才执行以下操作
        if (!expiredKey.contains(CodePrefix.PAPER_TASK_KEY.getKey())) {
            return;
        }

        // 将拿到的过期键使用之前拼接时的特殊符号分割成字符数组
        String[] expiredKeyArr = expiredKey.split(":");
        if (expiredKeyArr.length < 1) {
            return;
        }
        String recordId = expiredKeyArr[expiredKeyArr.length - 1];
        ExamUserRecord record = examUserRecordDao.selectById(recordId);
        if (null != record && QuestionStateEnum.inTest.getType().equals(record.getStatus())) {
            record.setStatus(QuestionStateEnum.haveTest.getType());
            record.setEndTime(LocalDateTime.now());

            long time = System.currentTimeMillis() - record.getStartTime().toInstant(ZoneOffset.of("+8")).toEpochMilli();
            //初始化Formatter的转换格式。
            SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
            formatter.setTimeZone(TimeZone.getTimeZone("GMT+00:00"));
            String answerTime = formatter.format(time);
            record.setAnswerTime(answerTime);
            examUserRecordDao.updateById(record);
            log.info("考试超时，自动提交试卷,考生：{},考试记录id：{}", record.getUserName(), record.getId());
        }
    }
}
