package com.artfess.examine.manager.impl;

import com.artfess.base.context.BaseContext;
import com.artfess.base.enums.AccessControlTypeEnum;
import com.artfess.base.enums.FileTypeEnum;
import com.artfess.base.manager.impl.BaseManagerImpl;
import com.artfess.base.query.PageBean;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.util.AuthenticationUtil;
import com.artfess.examine.dao.ExamFileDao;
import com.artfess.examine.dao.ExamMaterialInfoDao;
import com.artfess.examine.manager.ExamMaterialInfoManager;
import com.artfess.examine.manager.ExamMaterialTypeAuthManager;
import com.artfess.examine.model.ExamFile;
import com.artfess.examine.model.ExamMaterialInfo;
import com.artfess.examine.vo.ExamMaterialVo;
import com.artfess.file.util.FileUtils;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * 资料信息表 服务实现类
 *
 * @company 阿特菲斯信息技术有限公司
 * @author min.wu
 * @since 2022-10-19
 */
@Service
public class ExamMaterialInfoManagerImpl extends BaseManagerImpl<ExamMaterialInfoDao, ExamMaterialInfo> implements ExamMaterialInfoManager {

    @Resource
    private ExamFileDao examFileDao;

    @Autowired
    private ExamMaterialTypeAuthManager materialTypeAuthManager;

    @Resource
    BaseContext baseContext;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public String createInfo(ExamMaterialInfo t) {
        QueryWrapper<ExamMaterialInfo> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("type_id_", t.getTypeId());
        queryWrapper.eq("title_", t.getTitle());
        List<ExamMaterialInfo> examMaterialInfos = this.baseMapper.selectList(queryWrapper);
        if(!CollectionUtils.isEmpty(examMaterialInfos)) {
            throw new RuntimeException("该分类下资料名称已重复！");
        }

        int insert = this.baseMapper.insert(t);
        if(insert > 0) {
            processFile(t.getExamFiles(), t.getId());
            ExamMaterialVo examMaterialVo = t.getExamMaterialVo();
            examMaterialVo.setMaterialId(t.getId());
            materialTypeAuthManager.saveAuth(examMaterialVo);
            return t.getId();
        }

        return null;
    }

    private void processFile(List<ExamFile> examFiles, String id) {
        if(CollectionUtils.isEmpty(examFiles)) {
            return;
        }

        QueryWrapper<ExamFile> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("material_id_", id);
        this.examFileDao.delete(queryWrapper);
        examFiles.forEach(examFile -> {
            examFile.setMaterialId(id);
            String suffixName = FileUtils.getSuffixFromFileName(examFile.getFileName());
            examFile.setFileExt(suffixName);
            String fileType = FileTypeEnum.getType(suffixName);
            examFile.setFileType(fileType);
            BigDecimal fileSize = examFile.getFileSizes().divide(new BigDecimal(1024), 2, BigDecimal.ROUND_HALF_UP);
            examFile.setFileSizes(fileSize);
            examFileDao.insert(examFile);
        });

    }

    @Override
    public String updateInfo(ExamMaterialInfo t) {
        QueryWrapper<ExamMaterialInfo> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("type_id_", t.getTypeId());
        queryWrapper.eq("title_", t.getTitle());
        queryWrapper.ne("id_", t.getId());
        List<ExamMaterialInfo> examMaterialInfos = this.baseMapper.selectList(queryWrapper);
        if(!CollectionUtils.isEmpty(examMaterialInfos)) {
            throw new RuntimeException("该分类下资料名称已重复！");
        }

        processFile(t.getExamFiles(), t.getId());

        int insert = this.baseMapper.updateById(t);
        if(insert > 0) {
            ExamMaterialVo examMaterialVo = t.getExamMaterialVo();
            examMaterialVo.setMaterialId(t.getId());
            materialTypeAuthManager.saveAuth(examMaterialVo);
            return t.getId();
        }

        return null;
    }

    @Override
    public PageList<ExamMaterialInfo> findByPage(QueryFilter<ExamMaterialInfo> queryFilter) {

        PageBean pageBean = queryFilter.getPageBean();
        IPage<ExamMaterialInfo> result = baseMapper.queryPage(convert2IPage(pageBean), convert2Wrapper(queryFilter, currentModelClass()));
        return new PageList<ExamMaterialInfo>(result);
    }

    @Override
    public ExamMaterialInfo findById(String id) {
        ExamMaterialInfo examMaterialInfo = this.baseMapper.selectById(id);
        if(null == examMaterialInfo) {
            return new ExamMaterialInfo();
        }
        QueryWrapper<ExamFile> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("material_id_", id);
        List<ExamFile> examFiles = this.examFileDao.selectList(queryWrapper);
        examMaterialInfo.setExamFiles(examFiles);

        ExamMaterialVo byTypeId = materialTypeAuthManager.findByTypeId(id);
        examMaterialInfo.setExamMaterialVo(byTypeId);


        return examMaterialInfo;
    }

    @Override
    public ExamMaterialInfo viewFile(String id) {
        ExamMaterialInfo examMaterialInfo = this.baseMapper.selectById(id);
        if(null == examMaterialInfo) {
            return new ExamMaterialInfo();
        }


        QueryWrapper<ExamFile> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("material_id_", id);
        List<ExamFile> examFiles = this.examFileDao.selectList(queryWrapper);

        ExamMaterialVo byTypeId = materialTypeAuthManager.findByTypeId(id);
        if(AccessControlTypeEnum.ALL.getType().equals(byTypeId.getAccessControlType())){
            examMaterialInfo.setExamFiles(examFiles);
        }else if(AccessControlTypeEnum.ORG.getType().equals(byTypeId.getAccessControlType())) {
            String createOrgId = baseContext.getCurrentOrgId();
            AtomicBoolean flag = new AtomicBoolean(false);
            byTypeId.getAccessControlList().forEach(acc -> {
                if(createOrgId.equals(acc.getAccessControlId())){
                    flag.set(true);
                }
            });
            if(flag.get()) {
                examMaterialInfo.setExamFiles(examFiles);
            }

        }else if(AccessControlTypeEnum.USER.getType().equals(byTypeId.getAccessControlType())) {
            String userId = AuthenticationUtil.getCurrentUserId();
            AtomicBoolean flag = new AtomicBoolean(false);
            byTypeId.getAccessControlList().forEach(acc -> {
                if(userId.equals(acc.getAccessControlId())){
                    flag.set(true);
                }
            });
            if(flag.get()) {
                examMaterialInfo.setExamFiles(examFiles);
            }
        }
        return examMaterialInfo;
    }
}
