package com.artfess.data.controller;


import com.alibaba.fastjson.JSONObject;
import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.query.QueryOP;
import com.artfess.base.util.DictionaryUtils;
import com.artfess.base.util.DmpBeanUtil;
import com.artfess.data.manager.BizEquipmentBaseDataManager;
import com.artfess.data.manager.BizExamIndexDataManager;
import com.artfess.data.manager.BizExamPlanManager;
import com.artfess.data.manager.BizExamSpaceDataManager;
import com.artfess.data.manager.BizTrainEquipmentDataManager;
import com.artfess.data.model.BizExamPlan;
import com.artfess.data.vo.MonthPlanVo;
import com.artfess.data.vo.QuarterPlanVo;
import com.artfess.data.vo.TrainReqVo;
import com.artfess.data.vo.WeekPlanVo;
import com.artfess.data.vo.YearPlanVo;
import com.artfess.examine.manager.ExamEquipmentSysManager;
import com.artfess.poi.util.ExcelUtils;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.List;

/**
 * 训练统计 前端控制器
 *
 * @author min.wu
 * @company 阿特菲斯信息技术有限公司
 * @since 2024-10-19
 */
@Slf4j
@RestController
@Api(tags = "训练统计")
@RequestMapping("/exam/statistical")
@ApiGroup(group = {ApiGroupConsts.GROUP_BIZ})
public class DataStatisticalController {

    @Autowired
    private BizExamPlanManager planManager;

    @Autowired
    private ExamEquipmentSysManager equipmentSysManager;

    @Autowired
    private BizExamSpaceDataManager spaceDataManager;

    @Autowired
    private BizExamIndexDataManager indexDataManager;

    @Autowired
    private BizTrainEquipmentDataManager trainEquipmentDataManager;

    @Autowired
    private BizEquipmentBaseDataManager equipmentBaseDataManager;

    @PostMapping(value = "/planStatistical", produces = {"application/json; charset=utf-8"})
    @ApiOperation("个人训练情况")
    public List<JSONObject> planStatistical() {
        return planManager.planStatistical();
    }

    @PostMapping(value = "/trainObject", produces = {"application/json; charset=utf-8"})
    @ApiOperation("可视化分析-训练对象汇总")
    public JSONObject summary() {
        return equipmentSysManager.summary();
    }

    @PostMapping(value = "/trainSafeguard", produces = {"application/json; charset=utf-8"})
    @ApiOperation("可视化分析-训练保障汇总")
    public JSONObject trainSafeguard() {
        JSONObject jsonObject = new JSONObject(true);
        int spaceCount = spaceDataManager.count();
        jsonObject.put("spaceCount", spaceCount);
        int indexDateCount = indexDataManager.count();
        jsonObject.put("indexDateCount", indexDateCount);
        int trainEquipmentCount = trainEquipmentDataManager.count();
        jsonObject.put("trainEquipmentCount", trainEquipmentCount);
        int equipmentCount = equipmentBaseDataManager.count();
        jsonObject.put("equipmentCount", equipmentCount);
        return jsonObject;
    }

    @PostMapping(value = "/spacePancake", produces = {"application/json; charset=utf-8"})
    @ApiOperation("可视化分析-训练场地饼状图")
    public List<JSONObject> spacePancake(@ApiParam(name = "model", value = "实体信息") @RequestBody TrainReqVo reqVo) {
        return spaceDataManager.spacePancake(reqVo);
    }

    @PostMapping(value = "/indexDataPancake", produces = {"application/json; charset=utf-8"})
    @ApiOperation("可视化分析-训练消耗指标饼状图")
    public List<JSONObject> indexDataPancake(@ApiParam(name = "model", value = "实体信息") @RequestBody TrainReqVo reqVo) {
        return indexDataManager.indexDataPancake(reqVo);
    }

    @PostMapping(value = "/trainEquipmentPancake", produces = {"application/json; charset=utf-8"})
    @ApiOperation("可视化分析-训练器材饼状图")
    public List<JSONObject> trainEquipmentPancake(@ApiParam(name = "model", value = "实体信息") @RequestBody TrainReqVo reqVo) {
        return trainEquipmentDataManager.trainEquipmentPancake(reqVo);
    }

    @PostMapping(value = "/equipmentBasePancake", produces = {"application/json; charset=utf-8"})
    @ApiOperation("可视化分析-装备资源基础饼状图")
    public List<JSONObject> equipmentBasePancake(@ApiParam(name = "model", value = "实体信息") @RequestBody TrainReqVo reqVo) {
        return equipmentBaseDataManager.equipmentBasePancake(reqVo);
    }

    @PostMapping(value = "/trainPlanStylolitic", produces = {"application/json; charset=utf-8"})
    @ApiOperation("可视化分析-训练计划完成情况柱状图")
    public List<JSONObject> trainPlanStylolitic(@ApiParam(name = "model", value = "实体信息") @RequestBody TrainReqVo reqVo) {
        return planManager.trainPlanStylolitic(reqVo);
    }

    @PostMapping(value = "/orgYearTrainPlan", produces = {"application/json; charset=utf-8"})
    @ApiOperation("可视化分析-各单位年度训练计划完成情况柱状图")
    public List<JSONObject> orgYearTrainPlan(@ApiParam(name = "model", value = "实体信息") @RequestBody TrainReqVo reqVo) {
        return planManager.orgYearTrainPlan(reqVo);
    }

    @PostMapping(value = "/monthPlanStatistical", produces = {"application/json; charset=utf-8"})
    @ApiOperation("训练统计")
    public PageList<BizExamPlan> monthPlanStatistical(@ApiParam(name = "queryFilter", value = "分页查询信息") @RequestBody QueryFilter<BizExamPlan> queryFilter) {
        return planManager.monthPlanStatistical(queryFilter);
    }

    @ApiOperation(value = "训练统计导出")
    @PostMapping("/export")
    public void export(HttpServletResponse response, HttpServletRequest request,
                       @ApiParam(name = "queryFilter", value = "分页查询信息") @RequestBody QueryFilter<BizExamPlan> queryFilter) throws Exception {
        PageList<BizExamPlan> page = planManager.monthPlanStatistical(queryFilter);
        queryFilter.addFilter("plan_type_", "3", QueryOP.EQUAL);
        page.getRows().forEach(data -> {
            if (null != DictionaryUtils.findByDictValue("xljb", data.getTrainLevel())) {
                String type = DictionaryUtils.findByDictValue("xljb", data.getTrainLevel()).getName();
                data.setTrainLevel(type);
            }

            if (null != DictionaryUtils.findByDictValue("xllb", data.getTrainType())) {
                String trainType = DictionaryUtils.findByDictValue("xllb", data.getTrainType()).getName();
                data.setTrainType(trainType);
            }

            if (null != DictionaryUtils.findByDictValue("jdmc", data.getQuarter())) {
                String type = DictionaryUtils.findByDictValue("jdmc", data.getQuarter()).getName();
                data.setQuarter(type);
            }

            if (null != DictionaryUtils.findByDictValue("xlfs", data.getWay())) {
                String way = DictionaryUtils.findByDictValue("xlfs", data.getWay()).getName();
                data.setWay(way);
            }
        });

        ExcelUtils<MonthPlanVo> util = new ExcelUtils<MonthPlanVo>(MonthPlanVo.class);
        List<MonthPlanVo> list = DmpBeanUtil.copyList(page.getRows(), MonthPlanVo.class);
        util.exportExcel(response, request, list, "月度训练统计数据");

    }
}
