package com.artfess.examine.manager.impl;

import com.alibaba.fastjson.JSONObject;
import com.artfess.base.enums.ScoreLevelEnum;
import com.artfess.base.enums.TrainTypeEnum;
import com.artfess.base.exception.BaseException;
import com.artfess.base.manager.impl.BaseManagerImpl;
import com.artfess.base.query.PageBean;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.util.ConvertUtil;
import com.artfess.base.util.StringUtil;
import com.artfess.base.util.UniqueIdUtil;
import com.artfess.base.util.WordUtil;
import com.artfess.data.vo.ArchivesManagerVo;
import com.artfess.data.vo.UserExamVo;
import com.artfess.examine.dao.ExamPaperBaseDao;
import com.artfess.examine.dao.ExamSubjectInfoDao;
import com.artfess.examine.dao.ExamUserEvaluationDao;
import com.artfess.examine.dao.ExamUserEvaluationDetailDao;
import com.artfess.examine.dao.ExamUserRecordDao;
import com.artfess.examine.dao.ExamYearAssessmentSubDao;
import com.artfess.examine.dao.ExamYearReportDao;
import com.artfess.examine.manager.ExamUserEvaluationDetailManager;
import com.artfess.examine.manager.ExamUserEvaluationManager;
import com.artfess.examine.model.ExamPaperBase;
import com.artfess.examine.model.ExamSubjectInfo;
import com.artfess.examine.model.ExamUserEvaluation;
import com.artfess.examine.model.ExamUserEvaluationDetail;
import com.artfess.examine.model.ExamUserRecord;
import com.artfess.examine.model.ExamYearAssessmentSub;
import com.artfess.examine.model.ExamYearReport;
import com.artfess.examine.vo.SettingYearVo;
import com.artfess.examine.vo.UserEvaluationVo;
import com.artfess.examine.vo.UserInfoVo;
import com.artfess.file.util.AppFileUtil;
import com.artfess.poi.HeaderNode;
import com.artfess.poi.util.CustomHeader;
import com.artfess.poi.util.FileDownloadUtil;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.google.api.client.util.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 年度考生综合评定表 服务实现类
 *
 * @author min.wu
 * @company 阿特菲斯信息技术有限公司
 * @since 2022-11-28
 */
@Service
public class ExamUserEvaluationManagerImpl extends BaseManagerImpl<ExamUserEvaluationDao, ExamUserEvaluation> implements ExamUserEvaluationManager {

    @Resource
    private ExamYearAssessmentSubDao yearAssessmentSubDao;

    @Resource
    private ExamPaperBaseDao paperBaseDao;

    @Resource
    private ExamUserEvaluationDetailDao userEvaluationDetailDao;

    @Resource
    private ExamUserRecordDao userRecordDao;

    @Resource
    private ExamSubjectInfoDao subjectInfoDao;

    @Resource
    private ExamYearReportDao yearReportDao;

    @Autowired
    private ExamUserEvaluationDetailManager userEvaluationDetailManager;

    @Autowired
    private ExamUserEvaluationManager userEvaluationManager;

    @Override
    public PageList<ExamUserEvaluation> findByPage(QueryFilter<ExamUserEvaluation> queryFilter) {
        PageBean pageBean = queryFilter.getPageBean();
        IPage<ExamUserEvaluation> result = baseMapper.findByPage(convert2IPage(pageBean), convert2Wrapper(queryFilter, currentModelClass()));
        return new PageList<ExamUserEvaluation>(result);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void userEvaluation(ExamUserRecord userRecord) {
        if (StringUtils.isEmpty(userRecord.getPositionId())) {
            return;
        }
        String year = String.valueOf(LocalDateTime.now().getYear());

        QueryWrapper<ExamYearAssessmentSub> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("position_id_", userRecord.getPositionId());
        queryWrapper.eq("year_", year);
        List<ExamYearAssessmentSub> subList = yearAssessmentSubDao.selectList(queryWrapper);

        List<String> subjectIds = subList.stream().map(ExamYearAssessmentSub::getSubjectIds).collect(Collectors.toList());
        String subjectName = subList.stream()
                .map(ExamYearAssessmentSub::getSubjectName)
                .collect(Collectors.joining(",", "", ""));
        if (CollectionUtils.isEmpty(subjectIds)) {
            return;
        }

        ExamPaperBase examPaperBase = paperBaseDao.selectById(userRecord.getPaperId());
        if (null == examPaperBase) {
            return;
        }

        if (!"2".equals(examPaperBase.getPaperType())) {
            return;
        }

        if (StringUtils.isEmpty(examPaperBase.getSubjectId())) {
            return;
        }

        boolean flag = false;
        if (subjectIds.contains(examPaperBase.getSubjectId())) {
            flag = true;
        }

        if (!flag) {
            return;
        }
        //处理当前年度考核成绩是否记录
        processUserEvaluationDetail(examPaperBase, userRecord);

        UserEvaluationVo userEvaluationVo = UserEvaluationVo.builder()
                .userId(userRecord.getUserId())
                .subjectIds(subjectIds)
                .subjectName(subjectName)
                .positionId(userRecord.getPositionId())
                .year(year)
                .build();
        evaluationScore(userEvaluationVo);

    }

    private void evaluationScore(UserEvaluationVo userEvaluationVo) {
        String year = String.valueOf(LocalDateTime.now().getYear());
        //开始计算
        QueryWrapper<ExamUserEvaluationDetail> detailQueryWrapper = new QueryWrapper<>();
        detailQueryWrapper.eq("year_", year);
        detailQueryWrapper.eq("position_id_", userEvaluationVo.getPositionId());
        detailQueryWrapper.eq("user_id_", userEvaluationVo.getUserId());
        detailQueryWrapper.isNotNull("level_");
        List<ExamUserEvaluationDetail> list = userEvaluationDetailDao.selectList(detailQueryWrapper);

        int totalCount = 0;

        QueryWrapper<ExamUserEvaluation> query = new QueryWrapper<>();
        query.eq("position_id_", userEvaluationVo.getPositionId());
        query.eq("year_", String.valueOf(LocalDateTime.now().getYear()));
        query.eq("user_id_", userEvaluationVo.getUserId());
        ExamUserEvaluation examUserEvaluation = this.baseMapper.selectOne(query);
        if (!CollectionUtils.isEmpty(userEvaluationVo.getSubjectIds())) {
            totalCount = userEvaluationVo.getSubjectIds().size();
        }
        Integer level = null;
        int inCount = list.size();

        if (null == examUserEvaluation) {
            examUserEvaluation = new ExamUserEvaluation();
            examUserEvaluation.setYear(String.valueOf(LocalDateTime.now().getYear()));
            examUserEvaluation.setPositionId(userEvaluationVo.getPositionId());
            examUserEvaluation.setUserId(userEvaluationVo.getUserId());
            examUserEvaluation.setUserName(userEvaluationVo.getUserName());
            examUserEvaluation.setInCount(1);
            // 如果是线下课目 则没有课目id 手动计算课目总数
            if (CollectionUtils.isEmpty(userEvaluationVo.getSubjectIds())) {
                totalCount = 1;
                examUserEvaluation.setNotCount(0);
            } else {
                examUserEvaluation.setNotCount(userEvaluationVo.getSubjectIds().size() - 1);
            }
            examUserEvaluation.setTotalCount(totalCount);
            examUserEvaluation.setSubjectName(userEvaluationVo.getSubjectName());
            this.baseMapper.insert(examUserEvaluation);
        } else {
            if (CollectionUtils.isEmpty(userEvaluationVo.getSubjectIds())) {
                totalCount = examUserEvaluation.getTotalCount() + 1;
            }
            if (inCount < totalCount) {
                examUserEvaluation.setInCount(inCount);
                examUserEvaluation.setNotCount(totalCount - inCount);
            }
            examUserEvaluation.setTotalCount(totalCount);
            examUserEvaluation.setLevel(null);
            examUserEvaluation.setEvaluationTime(null);

            if (!CollectionUtils.isEmpty(list) && inCount == totalCount) {
                examUserEvaluation.setEvaluationTime(LocalDateTime.now());
                Map<Integer, List<ExamUserEvaluationDetail>> map = list.stream().collect(Collectors.groupingBy(ExamUserEvaluationDetail::getLevel));
                int goodCount = 0;
                int lhCount = 0;
                int jgCount = 0;
                int bjgCount = 0;
                if (map.containsKey(ScoreLevelEnum.GOOD.getType())) {
                    goodCount = map.get(ScoreLevelEnum.GOOD.getType()).size();
                }
                if (map.containsKey(ScoreLevelEnum.LH.getType())) {
                    lhCount = map.get(ScoreLevelEnum.LH.getType()).size();
                }
                if (map.containsKey(ScoreLevelEnum.JG.getType())) {
                    jgCount = map.get(ScoreLevelEnum.JG.getType()).size();
                }
                if (map.containsKey(ScoreLevelEnum.BJG.getType())) {
                    bjgCount = map.get(ScoreLevelEnum.BJG.getType()).size();
                }
                //优秀-所有训练课目|成绩均为良好以上,其中优秀率不低于50%;或者所有训练课目成绩均为及格以上，其中优秀率不低于80%。
                if (((lhCount + goodCount) == totalCount && totalCount * 0.5 <= goodCount) || (jgCount + lhCount + goodCount == totalCount && totalCount * 0.8 <= goodCount)) {
                    level = ScoreLevelEnum.GOOD.getType();
                }
                //良好——所有训练课目成绩均为及格以上,其中优良率不低于50%。.
                else if (jgCount + lhCount + goodCount == totalCount && totalCount * 0.5 <= goodCount + lhCount) {
                    level = ScoreLevelEnum.LH.getType();
                }
                //及格——90%以上的训练课目成绩为及格以上,或者50%以上的训练课目成绩为优秀、80%以上的训练课目成绩为及格以上。
                else if ((totalCount * 0.5 <= goodCount && totalCount * 0.8 <= jgCount + lhCount + goodCount) || totalCount * 0.9 <= jgCount + lhCount + goodCount) {
                    level = ScoreLevelEnum.JG.getType();
                } else {
                    level = ScoreLevelEnum.BJG.getType();
                }
                examUserEvaluation.setLevel(level);

            }
            examUserEvaluation.setSubjectName(userEvaluationVo.getSubjectName());
            this.baseMapper.updateById(examUserEvaluation);
        }
    }

    private void processUserEvaluationDetail(ExamPaperBase examPaperBase, ExamUserRecord userRecord) {
        if (StringUtils.isEmpty(examPaperBase.getEndStatus()) || "0".equals(examPaperBase.getEndStatus())) {
            return;
        }

        String year = String.valueOf(LocalDateTime.now().getYear());
        QueryWrapper<ExamUserEvaluationDetail> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("subject_id_", examPaperBase.getSubjectId());
        queryWrapper.eq("year_", year);
        queryWrapper.eq("position_id_", userRecord.getPositionId());
        queryWrapper.eq("user_id_", userRecord.getUserId());
        ExamUserEvaluationDetail examUserEvaluationDetail = userEvaluationDetailDao.selectOne(queryWrapper);
        if (null == examUserEvaluationDetail) {
            examUserEvaluationDetail = new ExamUserEvaluationDetail();
            examUserEvaluationDetail.setPositionId(userRecord.getPositionId());
            examUserEvaluationDetail.setSubjectId(examPaperBase.getSubjectId());
            examUserEvaluationDetail.setUserId(userRecord.getUserId());
            examUserEvaluationDetail.setUserName(userRecord.getUserName());
            examUserEvaluationDetail.setYear(year);
            examUserEvaluationDetail.setRecordId(userRecord.getId());
            examUserEvaluationDetail.setLevel(userRecord.getLevel());
            userEvaluationDetailDao.insert(examUserEvaluationDetail);
        } else {
            examUserEvaluationDetail.setRecordId(userRecord.getId());
            examUserEvaluationDetail.setLevel(userRecord.getLevel());
            userEvaluationDetailDao.updateById(examUserEvaluationDetail);
        }
    }

    @Override
    public PageList<ExamUserEvaluation> myYearPage(QueryFilter<ExamUserEvaluation> queryFilter) {
        PageBean pageBean = queryFilter.getPageBean();
        IPage<ExamUserEvaluation> result = baseMapper.myYearPage(convert2IPage(pageBean), convert2Wrapper(queryFilter, currentModelClass()));
        return new PageList<ExamUserEvaluation>(result);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean addSubject(ExamUserEvaluation t) {
        Assert.hasText(t.getId(), "请选择要增加课目的考生");
        //如果没有新增课目直接跳过
        if (StringUtil.isEmpty(t.getSubjectId())) {
            return true;
        }
        ExamUserEvaluation examUserEvaluation = this.baseMapper.selectById(t.getId());
        List<String> subjectIds = Arrays.asList(t.getSubjectId().split(","));
        subjectIds.forEach(subjectId -> {
            if (examUserEvaluation.getSubjectId().contains(subjectId)) {
                throw new BaseException("增加的课目中有已经加入年度考核的课目，请重新检查");
            }
        });
        examUserEvaluation.setSubjectId(examUserEvaluation.getSubjectId() + "," + t.getSubjectId());
        examUserEvaluation.setSubjectName(examUserEvaluation.getSubjectName() + "," + t.getSubjectName());
        int i = this.baseMapper.updateById(examUserEvaluation);
        if (i > 0) {
            UserEvaluationVo userEvaluationVo = UserEvaluationVo.builder()
                    .userId(examUserEvaluation.getUserId())
                    .subjectIds(Arrays.asList(examUserEvaluation.getSubjectId().split(",")))
                    .subjectName(examUserEvaluation.getSubjectName())
                    .positionId(examUserEvaluation.getPositionId())
                    .year(examUserEvaluation.getYear())
                    .build();
            evaluationScore(userEvaluationVo);
            return true;
        }
        //重新计算分数
        return false;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean delSubject(ExamUserEvaluation t) {
        Assert.hasText(t.getId(), "请选择要删除课目的考生");
        //如果没有删除课目直接跳过
        if (StringUtil.isEmpty(t.getSubjectId())) {
            return true;
        }
        ExamUserEvaluation examUserEvaluation = this.baseMapper.selectById(t.getId());
        String afterSubjectId = examUserEvaluation.getSubjectId();
        List<String> afterSubjectIds = new ArrayList(Arrays.asList(afterSubjectId.split(",")));
        List<String> afterSubjectNames = new ArrayList(Arrays.asList(examUserEvaluation.getSubjectName().split(",")));
        List<String> subjectIds = Arrays.asList(t.getSubjectId().split(","));

        //检查要删除的课目是否已存在，不存在则返回false
        subjectIds.forEach(subjectId -> {
            if (!afterSubjectId.contains(subjectId)) {
                throw new BaseException("删除的课目中未找到对应课目，请重新检查");
            }
            afterSubjectIds.remove(subjectId);
        });
        //检查要删除的课目是否已存在，不存在则返回false
        Arrays.asList(t.getSubjectName().split(",")).forEach(subjectName -> {
            afterSubjectNames.remove(subjectName);
        });

        examUserEvaluation.setSubjectId(String.join(",", afterSubjectIds));
        examUserEvaluation.setSubjectName(String.join(",", afterSubjectNames));
        int i = this.baseMapper.updateById(examUserEvaluation);
        if (i > 0) {
            QueryWrapper<ExamUserEvaluationDetail> queryWrapper = new QueryWrapper<>();
            queryWrapper.in("subject_id_", subjectIds);
            queryWrapper.eq("year_", examUserEvaluation.getYear());
            queryWrapper.eq("position_id_", examUserEvaluation.getPositionId());
            queryWrapper.eq("user_id_", examUserEvaluation.getUserId());
            userEvaluationDetailDao.delete(queryWrapper);
            UserEvaluationVo userEvaluationVo = UserEvaluationVo.builder()
                    .userId(examUserEvaluation.getUserId())
                    .subjectIds(Arrays.asList(examUserEvaluation.getSubjectId().split(",")))
                    .subjectName(examUserEvaluation.getSubjectName())
                    .positionId(examUserEvaluation.getPositionId())
                    .year(examUserEvaluation.getYear())
                    .build();
            evaluationScore(userEvaluationVo);
            return true;
        }
        //重新计算分数
        return false;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean settingYear(SettingYearVo t) {
        Assert.hasText(t.getId(), "请选择要设置年度考核成绩的考生");
        Assert.hasText(t.getRecordId(), "请选择要设置年度考核的成绩信息");
        ExamUserRecord userRecord = userRecordDao.selectById(t.getRecordId());
        Assert.notNull(userRecord, "考试记录不存在");
        ExamPaperBase examPaperBase = paperBaseDao.selectById(userRecord.getPaperId());
        if (null == examPaperBase) {
            return false;
        }
        ExamUserEvaluation examUserEvaluation = this.baseMapper.selectById(t.getId());
        QueryWrapper<ExamUserEvaluationDetail> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("subject_id_", examPaperBase.getSubjectId());
        queryWrapper.eq("year_", examUserEvaluation.getYear());
        queryWrapper.eq("position_id_", examUserEvaluation.getPositionId());
        queryWrapper.eq("user_id_", examUserEvaluation.getUserId());

        ExamUserEvaluationDetail examUserEvaluationDetail = userEvaluationDetailDao.selectOne(queryWrapper);
        if (null == examUserEvaluationDetail) {
            examUserEvaluationDetail = new ExamUserEvaluationDetail();
            examUserEvaluationDetail.setPositionId(userRecord.getPositionId());
            examUserEvaluationDetail.setSubjectId(examPaperBase.getSubjectId());
            examUserEvaluationDetail.setUserId(userRecord.getUserId());
            examUserEvaluationDetail.setUserName(userRecord.getUserName());
            examUserEvaluationDetail.setYear(examUserEvaluation.getYear());
            examUserEvaluationDetail.setRecordId(userRecord.getId());
            examUserEvaluationDetail.setLevel(userRecord.getLevel());
            examUserEvaluationDetail.setScore(String.valueOf(userRecord.getSysScore()));
            examUserEvaluationDetail.setType("1");
            examUserEvaluationDetail.setSubjectName(examPaperBase.getSubjectName());
            userEvaluationDetailDao.insert(examUserEvaluationDetail);
        } else {
            examUserEvaluationDetail.setScore(String.valueOf(userRecord.getSysScore()));
            examUserEvaluationDetail.setType("1");
            examUserEvaluationDetail.setSubjectName(examPaperBase.getSubjectName());
            examUserEvaluationDetail.setRecordId(userRecord.getId());
            examUserEvaluationDetail.setLevel(userRecord.getLevel());
            userEvaluationDetailDao.updateById(examUserEvaluationDetail);
        }

        UserEvaluationVo userEvaluationVo = UserEvaluationVo.builder()
                .userId(examUserEvaluation.getUserId())
                .subjectIds(Arrays.asList(examUserEvaluation.getSubjectId().split(",")))
                .subjectName(examUserEvaluation.getSubjectName())
                .positionId(examUserEvaluation.getPositionId())
                .year(examUserEvaluation.getYear())
                .build();
        evaluationScore(userEvaluationVo);
        return true;
    }

    @Override
    public List<ExamUserEvaluationDetail> evaluationDetail(String id) {
        ExamUserEvaluation examUserEvaluation = this.baseMapper.selectById(id);
        if (null == examUserEvaluation) {
            return Lists.newArrayList();
        }
        if (StringUtil.isEmpty(examUserEvaluation.getSubjectId())) {
            return Lists.newArrayList();
        }
        List<String> subjectIds = new ArrayList<>(Arrays.asList(examUserEvaluation.getSubjectId().split(",")));
        QueryWrapper<ExamUserEvaluationDetail> queryWrapper = new QueryWrapper<>();
        queryWrapper.in("subject_id_", subjectIds);
        queryWrapper.eq("year_", examUserEvaluation.getYear());
        queryWrapper.eq("position_id_", examUserEvaluation.getPositionId());
        queryWrapper.eq("user_id_", examUserEvaluation.getUserId());
        List<ExamUserEvaluationDetail> detailList = userEvaluationDetailDao.selectList(queryWrapper);
        Map<String, ExamUserEvaluationDetail> detailMap = detailList.stream().collect(Collectors.toMap(item -> item.getSubjectId(), item -> item));
        QueryWrapper<ExamSubjectInfo> query = new QueryWrapper<>();
        query.in("id_", subjectIds);
        List<ExamSubjectInfo> subjectInfoList = subjectInfoDao.selectList(query);
        Map<String, ExamSubjectInfo> sujectMap = subjectInfoList.stream().collect(Collectors.toMap(item -> item.getId(), item -> item));
        List<ExamUserEvaluationDetail> list = Lists.newArrayList();
        subjectIds.forEach(subjectId -> {
            ExamUserEvaluationDetail examUserEvaluationDetail = new ExamUserEvaluationDetail();
            examUserEvaluationDetail.setPositionId(examUserEvaluation.getPositionId());
            examUserEvaluationDetail.setSubjectId(subjectId);
            examUserEvaluationDetail.setUserId(examUserEvaluation.getUserId());
            examUserEvaluationDetail.setUserName(examUserEvaluation.getUserName());
            examUserEvaluationDetail.setYear(examUserEvaluation.getYear());
            if (null != detailMap && detailMap.containsKey(subjectId)) {
                ExamUserEvaluationDetail detail = detailMap.get(subjectId);
                if (StringUtils.isNotEmpty(detail.getRecordId()) && null != detail.getLevel()) {
                    examUserEvaluationDetail.setStatus(1);
                } else {
                    examUserEvaluationDetail.setStatus(0);
                }
                examUserEvaluationDetail.setRecordId(detail.getRecordId());
                examUserEvaluationDetail.setLevel(detail.getLevel());

            }
            if (null != sujectMap && sujectMap.containsKey(subjectId)) {
                ExamSubjectInfo examSubjectInfo = sujectMap.get(subjectId);
                examUserEvaluationDetail.setSubjectName(examSubjectInfo.getName());
            }
            list.add(examUserEvaluationDetail);

        });
        return list;
    }

    @Override
    public String getReportStatus() {
        String year = String.valueOf(LocalDateTime.now().getYear());
        QueryWrapper<ExamYearReport> yearReportQueryWrapper = new QueryWrapper<>();
        yearReportQueryWrapper.eq("year_", year);
        Integer count = yearReportDao.selectCount(yearReportQueryWrapper);
        if (count > 0) {
            return "1";
        }
        return "0";
    }

    @Override
    public List<JSONObject> getSubjectInfo(SettingYearVo t) {
        String year = String.valueOf(LocalDateTime.now().getYear());
        t.setYear(year);
        return this.baseMapper.getSubjectInfo(t);
    }

    @Override
    public List<JSONObject> getPositionPaper(SettingYearVo t) {
        return this.baseMapper.getPositionPaper(t);
    }

    @Override
    public void export(String orgId, HttpServletResponse response, String fileName) throws IOException {
        Map<String, Integer> rowMap = Maps.newHashMap();
        Map<String, Integer> columnMap = Maps.newHashMap();
        String year = String.valueOf(LocalDateTime.now().getYear());
        //获取当前组织下所有考生
        List<UserInfoVo> userList = this.userEvaluationDetailDao.getYearUserList(orgId, year);
        if (CollectionUtils.isEmpty(userList)) {
            throw new BaseException("该组织下没有考生");
        }
        HeaderNode headerNode = null;
        List<HeaderNode> list = Lists.newArrayList();
        headerNode = new HeaderNode();
        headerNode.setColumn(0);
        headerNode.setHeaderName("序号");
        headerNode.setRow(1);
        list.add(headerNode);

        headerNode = new HeaderNode();
        headerNode.setColumn(1);
        headerNode.setHeaderName("姓名");
        headerNode.setRow(1);
        list.add(headerNode);

        headerNode = new HeaderNode();
        headerNode.setColumn(2);
        headerNode.setHeaderName("性别");
        headerNode.setRow(1);
        list.add(headerNode);

        headerNode = new HeaderNode();
        headerNode.setColumn(3);
        headerNode.setHeaderName("年龄");
        headerNode.setRow(1);
        list.add(headerNode);

        headerNode = new HeaderNode();
        headerNode.setColumn(4);
        headerNode.setHeaderName("部职别");
        headerNode.setRow(1);
        list.add(headerNode);

        //获取当前岗位所有年度成绩
        List<ExamUserEvaluationDetail> examUserEvaluationDetails = this.userEvaluationDetailDao.findByYearParams(orgId, year);
        if (CollectionUtils.isEmpty(examUserEvaluationDetails)) {
            throw new BaseException("该岗位下没有年度考核成绩");
        }
        int useColumn = 4;
        //第一步 获取导入的线下训练课目
        List<JSONObject> subjectNames = this.baseMapper.offineInfo(orgId, year);
        if (!CollectionUtils.isEmpty(subjectNames)) {
            //线下导入成绩课目
            Integer xxksCount = 0;
            for (JSONObject json : subjectNames) {
                xxksCount++;
                headerNode = new HeaderNode();
                useColumn += 1;
                headerNode.setColumn(useColumn);
                headerNode.setHeaderName("课目" + ConvertUtil.intToChineseNum(xxksCount));
                headerNode.setRow(2);
                headerNode.setOverNodeCount(2);
                headerNode.setOverNode(false);
                list.add(headerNode);

                headerNode = new HeaderNode();
                headerNode.setColumn(useColumn);
                headerNode.setHeaderName("成绩");
                headerNode.setRow(3);
                list.add(headerNode);
                columnMap.put(json.getString("subjectname") + "-成绩", useColumn);
                useColumn += 1;
                headerNode = new HeaderNode();
                headerNode.setColumn(useColumn);
                headerNode.setHeaderName("评定");
                headerNode.setRow(3);
                list.add(headerNode);
            }

            headerNode = new HeaderNode();
            headerNode.setColumn(useColumn - (subjectNames.size() * 2) + 1);
            headerNode.setHeaderName("军事训练");
            headerNode.setRow(1);
            headerNode.setOverNodeCount(subjectNames.size() * 2);
            headerNode.setOverNode(false);
            list.add(headerNode);
        }

        //第二步 获取共同训练课目
        List<String> zygtxlSubjectNames = this.baseMapper.zygtxlSubjectNames(orgId, TrainTypeEnum.zygtxl.getType());
        //线下导入成绩课目
        Map<String, Integer> zygtxlColumnMap = Maps.newHashMap();
        for (int i = 0; i < zygtxlSubjectNames.size(); i++) {
            headerNode = new HeaderNode();
            useColumn += 1;
            headerNode.setColumn(useColumn);
            String s = ConvertUtil.intToChineseNum(i + 1);
            headerNode.setHeaderName("课目" + s);
            headerNode.setRow(2);
            headerNode.setOverNodeCount(2);
            headerNode.setOverNode(false);
            list.add(headerNode);

            headerNode = new HeaderNode();
            headerNode.setColumn(useColumn);
            headerNode.setHeaderName("成绩");
            headerNode.setRow(3);
            list.add(headerNode);
            zygtxlColumnMap.put("课目" + s, useColumn);
            //处理课目名称
            useColumn += 1;
            headerNode = new HeaderNode();
            headerNode.setColumn(useColumn);
            headerNode.setHeaderName("评定");
            headerNode.setRow(3);
            list.add(headerNode);
        }
        headerNode = new HeaderNode();
        headerNode.setColumn(useColumn - (zygtxlSubjectNames.size() * 2) + 1);
        headerNode.setHeaderName("共同训练");
        headerNode.setRow(1);
        headerNode.setOverNodeCount(zygtxlSubjectNames.size() * 2);
        headerNode.setOverNode(false);
        list.add(headerNode);

        //第三步 获取当前岗位岗位训练课目
        Map<String, Integer> gwxlColumnMap = Maps.newHashMap();
        List<String> gwxlSubjectNames = this.baseMapper.gwxlSubjectNames(orgId, TrainTypeEnum.gwxl.getType());
        for (int i = 0; i < gwxlSubjectNames.size(); i++) {
            headerNode = new HeaderNode();
            useColumn += 1;
            headerNode.setColumn(useColumn);
            String s = ConvertUtil.intToChineseNum(i + 1);
            headerNode.setHeaderName("课目" + s);
            headerNode.setRow(2);
            headerNode.setOverNodeCount(2);
            headerNode.setOverNode(false);

            list.add(headerNode);
            headerNode = new HeaderNode();
            headerNode.setColumn(useColumn);
            headerNode.setHeaderName("成绩");
            headerNode.setRow(3);
            list.add(headerNode);
            gwxlColumnMap.put("课目" + s, useColumn);

            useColumn += 1;
            headerNode = new HeaderNode();
            headerNode.setColumn(useColumn);
            headerNode.setHeaderName("评定");
            headerNode.setRow(3);
            list.add(headerNode);
        }
        headerNode = new HeaderNode();
        headerNode.setColumn(useColumn - (gwxlSubjectNames.size() * 2) + 1);
        headerNode.setHeaderName("岗位训练");
        headerNode.setRow(1);
        headerNode.setOverNodeCount(gwxlSubjectNames.size() * 2);
        headerNode.setOverNode(false);
        list.add(headerNode);

        headerNode = new HeaderNode();
        headerNode.setColumn(useColumn + 1);
        headerNode.setHeaderName("总评");
        headerNode.setRow(1);
        list.add(headerNode);

        headerNode = new HeaderNode();
        headerNode.setColumn(0);
        headerNode.setHeaderName(year + "年训练成绩统计表");
        headerNode.setRow(0);
        headerNode.setOverNodeCount(useColumn + 2);
        headerNode.setOverNode(false);
        list.add(headerNode);

        int row = 3;
        int size = 0;
        for (UserInfoVo user : userList) {
            row++;
            size++;
            headerNode = new HeaderNode();
            headerNode.setColumn(0);
            headerNode.setHeaderName(size + "");
            headerNode.setRow(row);
            list.add(headerNode);

            headerNode = new HeaderNode();
            headerNode.setColumn(1);
            headerNode.setHeaderName(user.getUserName());
            headerNode.setRow(row);
            list.add(headerNode);

            headerNode = new HeaderNode();
            headerNode.setColumn(2);
            headerNode.setHeaderName(user.getSex());
            headerNode.setRow(row);
            list.add(headerNode);

            headerNode = new HeaderNode();
            headerNode.setColumn(3);
            headerNode.setHeaderName("年龄");
            headerNode.setRow(row);
            list.add(headerNode);

            headerNode = new HeaderNode();
            headerNode.setColumn(4);
            headerNode.setHeaderName(user.getOrgName());
            headerNode.setRow(row);
            list.add(headerNode);
            rowMap.put(user.getUserId(), headerNode.getRow());

            headerNode = new HeaderNode();
            headerNode.setColumn(useColumn + 1);
            headerNode.setHeaderName(ScoreLevelEnum.getDesc(user.getLevel()));
            headerNode.setRow(row);
            list.add(headerNode);
            rowMap.put(user.getUserId(), headerNode.getRow());
        }

        for (ExamUserEvaluationDetail detail : examUserEvaluationDetails) {
            if (!rowMap.containsKey(detail.getUserId())) {
                continue;
            }
            int currentRow = rowMap.get(detail.getUserId());
            boolean flag = false;
            Integer scoreColumn = null;
            Integer levelColumn = null;
            if (columnMap.containsKey(detail.getSubjectName() + "-成绩")) {
                scoreColumn = columnMap.get(detail.getSubjectName() + "-成绩");
                levelColumn = scoreColumn + 1;
                flag = true;
            }

            for (Map.Entry<String, Integer> entry : zygtxlColumnMap.entrySet()) {
                String key = entry.getKey();
                if (StringUtils.isNotEmpty(detail.getSubjectName()) && detail.getSubjectName().contains(key)) {
                    scoreColumn = entry.getValue();
                    levelColumn = entry.getValue() + 1;
                    flag = true;
                }
            }

            for (Map.Entry<String, Integer> entry : gwxlColumnMap.entrySet()) {
                String key = entry.getKey();
                if (StringUtils.isNotEmpty(detail.getSubjectName()) && detail.getSubjectName().contains(key)) {
                    scoreColumn = entry.getValue();
                    levelColumn = entry.getValue() + 1;
                    flag = true;
                }
            }

            if (!flag) {
                continue;
            }

            headerNode = new HeaderNode();
            headerNode.setColumn(scoreColumn);
            headerNode.setHeaderName(detail.getScore());
            headerNode.setRow(currentRow);
            list.add(headerNode);

            headerNode = new HeaderNode();
            headerNode.setColumn(levelColumn);
            headerNode.setHeaderName(ScoreLevelEnum.getDesc(detail.getLevel()));
            headerNode.setRow(currentRow);
            list.add(headerNode);
        }
        list.sort(Comparator.comparing(HeaderNode::getRow));
        CustomHeader.export(list, response, fileName);

    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean batchSettingYear(SettingYearVo t) {
        Assert.hasText(t.getPositionId(), "请选择要设置年度考核成绩的岗位");
        Assert.hasText(t.getPaperId(), "请选择要设置年度考核的试卷");
        Assert.hasText(t.getSubjectId(), "请选择要设置年度考核的课目");
        ExamSubjectInfo examSubjectInfo = subjectInfoDao.selectById(t.getSubjectId());
        QueryWrapper<ExamUserRecord> query = new QueryWrapper<>();
        query.eq("position_id_", t.getPositionId());
        query.eq("paper_id_", t.getPaperId());
        List<ExamUserRecord> userRecords = userRecordDao.selectList(query);
        Map<String, ExamUserRecord> userRecordMap = userRecords.stream().collect(Collectors.toMap(item -> item.getUserId(), item -> item));
        String year = String.valueOf(LocalDateTime.now().getYear());
        List<UserInfoVo> userList = this.userEvaluationDetailDao.getUserList(t.getPositionId());
        List<ExamUserEvaluationDetail> details = Lists.newArrayList();
        userList.forEach(userInfoVo -> {
            ExamUserRecord examUserRecord = null;
            if (userRecordMap.containsKey(userInfoVo.getUserId())) {
                examUserRecord = userRecordMap.get(userInfoVo.getUserId());
            }
            QueryWrapper<ExamUserEvaluationDetail> queryWrapper = new QueryWrapper<>();
            queryWrapper.eq("subject_id_", t.getSubjectId());
            queryWrapper.eq("year_", year);
            queryWrapper.eq("position_id_", t.getPositionId());
            queryWrapper.eq("user_id_", userInfoVo.getUserId());
            ExamUserEvaluationDetail examUserEvaluationDetail = userEvaluationDetailDao.selectOne(queryWrapper);
            if (null == examUserEvaluationDetail) {
                examUserEvaluationDetail = new ExamUserEvaluationDetail();
                examUserEvaluationDetail.setPositionId(t.getPositionId());
                examUserEvaluationDetail.setSubjectId(t.getSubjectId());
                examUserEvaluationDetail.setUserId(userInfoVo.getUserId());
                examUserEvaluationDetail.setUserName(userInfoVo.getUserName());
                examUserEvaluationDetail.setYear(year);
                if (null != examSubjectInfo) {
                    examUserEvaluationDetail.setSubjectName(examSubjectInfo.getName());
                }
                if (null != examUserRecord) {
                    examUserEvaluationDetail.setRecordId(examUserRecord.getId());
                    examUserEvaluationDetail.setLevel(examUserRecord.getLevel());
                    examUserEvaluationDetail.setType("1");
                    examUserEvaluationDetail.setScore(String.valueOf(examUserRecord.getSysScore()));
                } else {
                    examUserEvaluationDetail.setLevel(ScoreLevelEnum.BJG.getType());
                    examUserEvaluationDetail.setType("1");
                    examUserEvaluationDetail.setScore("0");
                }
                details.add(examUserEvaluationDetail);
            } else {
                if (null != examUserRecord) {
                    examUserEvaluationDetail.setRecordId(examUserRecord.getId());
                    examUserEvaluationDetail.setLevel(examUserRecord.getLevel());
                    examUserEvaluationDetail.setType("1");
                    examUserEvaluationDetail.setScore(String.valueOf(examUserRecord.getSysScore()));
                } else {
                    examUserEvaluationDetail.setLevel(ScoreLevelEnum.BJG.getType());
                    examUserEvaluationDetail.setType("1");
                    examUserEvaluationDetail.setScore("0");
                }
                details.add(examUserEvaluationDetail);
            }
            QueryWrapper<ExamUserEvaluation> userEvaluationQueryWrapper = new QueryWrapper<>();
            userEvaluationQueryWrapper.eq("subject_id_", t.getSubjectId());
            userEvaluationQueryWrapper.eq("year_", year);
            userEvaluationQueryWrapper.eq("position_id_", t.getPositionId());
            userEvaluationQueryWrapper.eq("user_id_", userInfoVo.getUserId());
            ExamUserEvaluation examUserEvaluation = baseMapper.selectOne(userEvaluationQueryWrapper);
            if (null == examUserEvaluation) {
                examUserEvaluation = new ExamUserEvaluation();
                examUserEvaluation.setYear(t.getYear());
                examUserEvaluation.setPositionId(t.getPositionId());
                examUserEvaluation.setUserId(userInfoVo.getUserId());
                examUserEvaluation.setUserName(userInfoVo.getUserName());
                examUserEvaluation.setInCount(0);
                examUserEvaluation.setNotCount(1);
                examUserEvaluation.setTotalCount(1);
                examUserEvaluation.setSubjectName(examUserEvaluationDetail.getSubjectName());
                examUserEvaluation.setSubjectId(examUserEvaluationDetail.getSubjectId());
            }
            UserEvaluationVo userEvaluationVo = UserEvaluationVo.builder()
                    .userId(userInfoVo.getUserId())
                    .subjectIds(Arrays.asList(examUserEvaluation.getSubjectId().split(",")))
                    .subjectName(examUserEvaluationDetail.getSubjectName())
                    .positionId(t.getPositionId())
                    .year(year)
                    .build();
            evaluationScore(userEvaluationVo);
        });
        boolean b = userEvaluationDetailManager.saveOrUpdateBatch(details);
        return b;
    }

    @Override
    public boolean importExcel(List<ExamUserEvaluationDetail> detail, String positionId) {
        String year = String.valueOf(LocalDateTime.now().getYear());
        List<UserInfoVo> userList = this.userEvaluationDetailDao.getUserList(positionId);
        Map<String, UserInfoVo> userInfoVoMap = userList.stream().collect(Collectors.toMap(item -> item.getUserName(), item -> item));
        detail.forEach(userEvaluation -> {
            if (userInfoVoMap.containsKey(userEvaluation.getUserName())) {
                userEvaluation.setUserId(userInfoVoMap.get(userEvaluation.getUserName()).getUserId());
            }
            QueryWrapper<ExamUserEvaluationDetail> queryWrapper = new QueryWrapper<>();
            queryWrapper.eq("user_id_", userEvaluation.getUserId());
            queryWrapper.eq("subject_name_", userEvaluation.getSubjectName());
            queryWrapper.eq("type_", "2");
            List<ExamUserEvaluationDetail> details = userEvaluationDetailDao.selectList(queryWrapper);
            if (!CollectionUtils.isEmpty(details)) {
                throw new BaseException("该岗位下" + userEvaluation.getUserName() + "已导入" + userEvaluation.getSubjectName() + "线下成绩");
            }
            userEvaluation.setUserName(userEvaluation.getUserName());
            userEvaluation.setPositionId(positionId);
            userEvaluation.setType("2");
            userEvaluation.setYear(year);
            userEvaluation.setLevel(ScoreLevelEnum.getType(userEvaluation.getLevelDesc()));
            UserEvaluationVo userEvaluationVo = UserEvaluationVo.builder()
                    .userId(userEvaluation.getUserId())
                    .subjectName(userEvaluation.getSubjectName())
                    .positionId(userEvaluation.getPositionId())
                    .userName(userEvaluation.getUserName())
                    .year(year)
                    .build();
            evaluationScore(userEvaluationVo);
        });
        boolean b = userEvaluationDetailManager.saveOrUpdateBatch(detail);
        return b;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean delPositionSubject(String subjectIds, String positionId, String year) {
        ExamSubjectInfo examSubjectInfo = subjectInfoDao.selectById(subjectIds);
        if (null == examSubjectInfo) {
            return true;
        }

        QueryWrapper<ExamUserEvaluation> query = new QueryWrapper<>();
        query.eq("position_id_", positionId);
        query.eq("year_", year);
        query.like("subject_id_", subjectIds);
        query.gt("in_count_", 0);
        List<ExamUserEvaluation> examUserEvaluations = userEvaluationManager.list(query);
        if (!CollectionUtils.isEmpty(examUserEvaluations)) {
            throw new BaseException("当前岗位该课目已参与考核中，不能进行删除");
        }

        QueryWrapper<ExamYearAssessmentSub> subQuery = new QueryWrapper<>();
        subQuery.eq("position_id_", positionId);
        subQuery.eq("year_", year);
        subQuery.eq("subject_ids_", subjectIds);
        yearAssessmentSubDao.delete(subQuery);

        QueryWrapper<ExamUserEvaluation> UserEvaluationQuery = new QueryWrapper<>();
        UserEvaluationQuery.eq("position_id_", positionId);
        UserEvaluationQuery.eq("year_", year);
        examUserEvaluations = userEvaluationManager.list(UserEvaluationQuery);
        List<ExamUserEvaluation> updateUserEvaluationList = Lists.newArrayList();
        List<String> needDelIds = Lists.newArrayList();
        examUserEvaluations.forEach(examUserEvaluation -> {
            List<String> ids = Lists.newArrayList(Arrays.asList(examUserEvaluation.getSubjectId().split(",")));
            List<String> names = Lists.newArrayList(Arrays.asList(examUserEvaluation.getSubjectName().split(",")));
            if (!CollectionUtils.isEmpty(ids)) {
                ids.removeIf(subjectIds::equals);
            }
            if (!CollectionUtils.isEmpty(names)) {
                names.removeIf(examSubjectInfo.getName()::equals);
            }
            if (CollectionUtils.isEmpty(ids) || CollectionUtils.isEmpty(names)) {
                needDelIds.add(examUserEvaluation.getId());
            } else {
                examUserEvaluation.setSubjectId(ids.stream().collect(Collectors.joining(",", "", "")));
                examUserEvaluation.setSubjectName(names.stream().collect(Collectors.joining(",", "", "")));
                updateUserEvaluationList.add(examUserEvaluation);
            }

        });
        //修改考核课目
        this.saveOrUpdateBatch(updateUserEvaluationList);
        //删除没有课目的考核信息
        this.removeByIds(needDelIds);

        QueryWrapper<ExamUserEvaluationDetail> detailQueryWrapper = new QueryWrapper<>();
        detailQueryWrapper.in("subject_id_", subjectIds);
        detailQueryWrapper.eq("year_", year);
        detailQueryWrapper.eq("position_id_", positionId);
        int delete = userEvaluationDetailDao.delete(detailQueryWrapper);
        return true;
    }

    @Override
    public PageList<ArchivesManagerVo> positionArchives(QueryFilter<ExamUserEvaluation> queryFilter) {
        PageBean pageBean = queryFilter.getPageBean();
        IPage<ArchivesManagerVo> result = baseMapper.positionArchives(convert2IPage(pageBean), convert2Wrapper(queryFilter, currentModelClass()));
        result.getRecords().forEach(vo -> {
            vo.setId(UniqueIdUtil.getSuid());
            if (0 == vo.getPassCount()) {
                vo.setPassRate("0%");
            } else {
                vo.setPassRate(new BigDecimal(vo.getPassCount()).divide(new BigDecimal(vo.getShouldCount()), 2, BigDecimal.ROUND_HALF_UP)
                        .multiply(new BigDecimal(100)) + "%");
            }

            if (0 == vo.getNoPassCount()) {
                vo.setNoPassRate("0%");
            } else {
                vo.setNoPassRate(new BigDecimal(vo.getNoPassCount()).divide(new BigDecimal(vo.getShouldCount()), 2, BigDecimal.ROUND_HALF_UP)
                        .multiply(new BigDecimal(100)) + "%");
            }

            if (0 == vo.getActualCount()) {
                vo.setReferRate("0%");
            } else {
                vo.setReferRate(new BigDecimal(vo.getActualCount()).divide(new BigDecimal(vo.getShouldCount()), 2, BigDecimal.ROUND_HALF_UP)
                        .multiply(new BigDecimal(100)) + "%");
            }

        });
        return new PageList<>(result);
    }

    @Override
    public PageList<ArchivesManagerVo> orgArchives(QueryFilter<ExamUserEvaluation> queryFilter) {
        PageBean pageBean = queryFilter.getPageBean();
        IPage<ArchivesManagerVo> result = baseMapper.orgArchives(convert2IPage(pageBean), convert2Wrapper(queryFilter, currentModelClass()));
        result.getRecords().forEach(vo -> {
            vo.setId(UniqueIdUtil.getSuid());

            if (0 == vo.getActualCount()) {
                vo.setReferRate("0%");
            } else {
                vo.setReferRate(new BigDecimal(vo.getActualCount()).divide(new BigDecimal(vo.getShouldCount()), 2, BigDecimal.ROUND_HALF_UP)
                        .multiply(new BigDecimal(100)) + "%");
            }

        });
        return new PageList<>(result);
    }

    @Override
    public PageList<ArchivesManagerVo> subjectArchives(QueryFilter<ExamUserEvaluation> queryFilter) {
        PageBean pageBean = queryFilter.getPageBean();
        IPage<ArchivesManagerVo> result = baseMapper.subjectArchives(convert2IPage(pageBean), convert2Wrapper(queryFilter, currentModelClass()));
        Set<String> postionIds = result.getRecords().stream().map(ArchivesManagerVo::getPositionId).collect(Collectors.toSet());
        List<ArchivesManagerVo> shouldCountList = Lists.newArrayList();
        if (!CollectionUtils.isEmpty(postionIds)) {
            shouldCountList = this.baseMapper.getShouldCountMap(postionIds);
        }
        Map<String, ArchivesManagerVo> shouldCountMap = Maps.newHashMap();
        if (!CollectionUtils.isEmpty(shouldCountList)) {
            shouldCountMap = shouldCountList.stream().collect(Collectors.toMap(item -> item.getPositionId(), item -> item));
        }
        Map<String, ArchivesManagerVo> evaluationMap = Maps.newHashMap();
        if (!CollectionUtils.isEmpty(result.getRecords())) {
            List<ArchivesManagerVo> list = this.baseMapper.getEvaluationList(result.getRecords().get(0).getYear());
            list.forEach(vo -> {
                evaluationMap.put(vo.getPositionId() + vo.getSubjectId() + vo.getYear(), vo);
            });
        }

        Map<String, ArchivesManagerVo> finalShouldCountMap = shouldCountMap;
        result.getRecords().forEach(data -> {
            if (!CollectionUtils.isEmpty(finalShouldCountMap) && null != finalShouldCountMap.get(data.getPositionId())) {
                data.setShouldCount(finalShouldCountMap.get(data.getPositionId()).getShouldCount());
            }
            if (!CollectionUtils.isEmpty(evaluationMap) && null != evaluationMap.get(data.getPositionId() + data.getSubjectId() + data.getYear())) {
                ArchivesManagerVo archivesManagerVo = evaluationMap.get(data.getPositionId() + data.getSubjectId() + data.getYear());
                data.setActualCount(archivesManagerVo.getActualCount());
                data.setNoPassCount(archivesManagerVo.getNoPassCount());
                data.setPassCount(archivesManagerVo.getPassCount());
            }
            if (0 == data.getActualCount()) {
                data.setReferRate("0%");
            } else {
                data.setReferRate(new BigDecimal(data.getActualCount()).divide(new BigDecimal(data.getShouldCount()), 2, BigDecimal.ROUND_HALF_UP)
                        .multiply(new BigDecimal(100)) + "%");
            }
            if (0 == data.getPassCount()) {
                data.setPassRate("0%");
            } else {
                data.setPassRate(new BigDecimal(data.getPassCount()).divide(new BigDecimal(data.getShouldCount()), 2, BigDecimal.ROUND_HALF_UP)
                        .multiply(new BigDecimal(100)) + "%");
            }
        });
        return new PageList<>(result);
    }

    @Override
    public void subjectArchivesExport(QueryFilter<ExamUserEvaluation> queryFilter, HttpServletResponse response) {
        PageList<ArchivesManagerVo> pageList = this.subjectArchives(queryFilter);
        JSONObject dataMap = new JSONObject();
        dataMap.put("list", pageList.getRows());
        if (CollectionUtils.isEmpty(pageList.getRows())) {
            dataMap.put("year", LocalDate.now().getYear());
        } else {
            dataMap.put("year", pageList.getRows().get(0).getYear());
        }
        String fileId = UniqueIdUtil.getSuid();
        //文件唯一名称
        String fileOnlyName = "课目档案.doc";
        String ftlPath = AppFileUtil.createFilePath("ftl" + File.separator + fileId, fileOnlyName);
        /** 生成word */
        WordUtil.createWord(response, dataMap, "课目档案.ftl", ftlPath, fileOnlyName);
        try {
            FileDownloadUtil.fileDownload(response, new File(ftlPath), fileOnlyName);
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        } catch (Exception e) {
            e.printStackTrace();
        }

    }

    @Override
    public void orgArchivesExport(QueryFilter<ExamUserEvaluation> queryFilter, HttpServletResponse response) {
        PageList<ArchivesManagerVo> pageList = this.orgArchives(queryFilter);
        pageList.getRows().forEach(vo -> {
            if (StringUtils.isNotEmpty(vo.getLevel())) {
                vo.setLevel(ScoreLevelEnum.getDesc(Integer.parseInt(vo.getLevel())));
            }

        });
        JSONObject dataMap = new JSONObject();
        dataMap.put("list", pageList.getRows());
        if (CollectionUtils.isEmpty(pageList.getRows())) {
            dataMap.put("year", LocalDate.now().getYear());
        } else {
            dataMap.put("year", pageList.getRows().get(0).getYear());
        }
        String fileId = UniqueIdUtil.getSuid();
        //文件唯一名称
        String fileOnlyName = "单位档案.doc";
        String ftlPath = AppFileUtil.createFilePath("ftl" + File.separator + fileId, fileOnlyName);
        /** 生成word */
        WordUtil.createWord(response, dataMap, "单位档案.ftl", ftlPath, fileOnlyName);
        try {
            FileDownloadUtil.fileDownload(response, new File(ftlPath), fileOnlyName);
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    public void positionArchivesExport(QueryFilter<ExamUserEvaluation> queryFilter, HttpServletResponse response) {
        PageList<ArchivesManagerVo> pageList = this.positionArchives(queryFilter);
        JSONObject dataMap = new JSONObject();
        dataMap.put("list", pageList.getRows());
        if (CollectionUtils.isEmpty(pageList.getRows())) {
            dataMap.put("year", LocalDate.now().getYear());
        } else {
            dataMap.put("year", pageList.getRows().get(0).getYear());
        }
        String fileId = UniqueIdUtil.getSuid();
        //文件唯一名称
        String fileOnlyName = "岗位档案.doc";
        String ftlPath = AppFileUtil.createFilePath("ftl" + File.separator + fileId, fileOnlyName);
        /** 生成word */
        WordUtil.createWord(response, dataMap, "岗位档案.ftl", ftlPath, fileOnlyName);
        try {
            FileDownloadUtil.fileDownload(response, new File(ftlPath), fileOnlyName);
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    public PageList<UserExamVo> userQualityAnalyse(QueryFilter<ExamUserEvaluation> queryFilter) {
        PageBean pageBean = queryFilter.getPageBean();
        IPage<UserExamVo> result = baseMapper.userQualityAnalyse(convert2IPage(pageBean), convert2Wrapper(queryFilter, currentModelClass()));
        result.getRecords().forEach(exam -> {
            String subjectNames = userEvaluationDetailManager.getSubjectNames(exam.getPositionId(), exam.getYear(), exam.getUserId());
            if (!StringUtils.isNotEmpty(subjectNames)) {
                return;
            }
            String[] split = subjectNames.split(",");
            for (int i = 0; i < split.length; i++) {
                if (i == 0) {
                    exam.setSubjectName1(split[i]);
                }
                if (i == 1) {
                    exam.setSubjectName2(split[i]);
                }
                if (i == 2) {
                    exam.setSubjectName3(split[i]);
                }
                if (i == 3) {
                    exam.setSubjectName4(split[i]);
                }

            }
        });
        return new PageList<UserExamVo>(result);
    }

    /**
     * 设置考生职业教育分数以及总体评价
     *
     * @param t 考试记录
     * @return 设置成功与否
     */
    @Override
    public boolean setUserWorkEvaluation(ExamUserEvaluation t) {
        if (t != null && StringUtils.isNotEmpty(t.getId())) {
            ExamUserEvaluation userEvaluation = baseMapper.selectById(t.getId());
            if (userEvaluation != null) {
                userEvaluation.setVocationalEducationScore(t.getVocationalEducationScore());
                Integer vocationalEducationScore = t.getVocationalEducationScore();
                //计算公式：90分以上是优秀，60~69分为及格，70~90分为良好，低于60分不及格4
                if (vocationalEducationScore >= 90) {
                    userEvaluation.setVocationalEducationEvaluation(ScoreLevelEnum.GOOD.getDesc());
                } else if (vocationalEducationScore >= 70 && vocationalEducationScore < 90) {
                    userEvaluation.setVocationalEducationEvaluation(ScoreLevelEnum.LH.getDesc());
                } else if (vocationalEducationScore >= 60 && vocationalEducationScore < 70) {
                    userEvaluation.setVocationalEducationEvaluation(ScoreLevelEnum.JG.getDesc());
                } else {
                    userEvaluation.setVocationalEducationEvaluation(ScoreLevelEnum.BJG.getDesc());
                }
                userEvaluation.setVocationalEducationEvaluation(t.getVocationalEducationEvaluation());
                userEvaluation.setOverallEvaluationDescription(t.getOverallEvaluationDescription());

            }
        }
        return false;
    }

}
