package com.artfess.cqlt.controller;


import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.controller.BaseController;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.valid.AddGroup;
import com.artfess.base.valid.UpdateGroup;
import com.artfess.cqlt.manager.QfEnterpriseInfoManager;
import com.artfess.cqlt.model.QfEnterpriseInfo;
import com.artfess.cqlt.vo.EnterpriseInfoVo;
import com.artfess.i18n.util.I18nUtil;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RestController;

import java.util.HashMap;
import java.util.List;

/**
 * 境外企业信息 前端控制器
 *
 * @company 阿特菲斯信息技术有限公司
 * @author lxk
 * @since 2023-02-10
 */
@Slf4j
@RestController
@Api(tags = "境外企业信息 ")
@RequestMapping("/qfEnterpriseInfo/v1/")
@ApiGroup(group = {ApiGroupConsts.GROUP_BIZ})
public class QfEnterpriseInfoController extends BaseController<QfEnterpriseInfoManager, QfEnterpriseInfo> {


    @PostMapping("/insertInfo")
    @ApiOperation("新增企业信息")
    public CommonResult<String> insertInfo(@ApiParam(name="model", value="实体信息") @RequestBody @Validated({AddGroup.class}) QfEnterpriseInfo t) {
        String id = baseService.newInsertTree(t);
        if(!StringUtils.isNotBlank(id)) {
            return new CommonResult<String>(false, I18nUtil.getMessage("option.fail", LocaleContextHolder.getLocale()), null);
        }
        return new CommonResult<String>(true, I18nUtil.getMessage("option.success", LocaleContextHolder.getLocale()), null);
    }

    @Override
    @PostMapping("/updateById")
    @ApiOperation("编辑企业信息")
    public CommonResult<String> updateById(@ApiParam(name="model", value="实体信息") @RequestBody @Validated({UpdateGroup.class})  QfEnterpriseInfo t) {
        QfEnterpriseInfo byId = baseService.getById(t.getId());
        String id = baseService.newUpdateTree(t, byId.getName());
        if(!StringUtils.isNotBlank(id)) {
            return new CommonResult<String>(false, I18nUtil.getMessage("update.fail", LocaleContextHolder.getLocale()), null);
        }
        return new CommonResult<String>(true, I18nUtil.getMessage("option.success", LocaleContextHolder.getLocale()), null);
    }

    @PostMapping("/getTree")
    @ApiOperation("获取企业信息树形结构")
    public CommonResult<List<QfEnterpriseInfo>> getTree() {
        return baseService.getTree();
    }


    @Override
    @GetMapping("deleteById/{id}")
    @ApiOperation("根据id删除")
    public CommonResult<String> deleteById(@ApiParam(name="id", value="实体id") @PathVariable String id) {
        boolean result = baseService.removeById(id);
        if(!result) {
            return new CommonResult<String>(false, I18nUtil.getMessage("delete.fail", LocaleContextHolder.getLocale()), null);
        }
        return new CommonResult<String>(true, I18nUtil.getMessage("option.success", LocaleContextHolder.getLocale()), null);
    }


    @PostMapping("/getList")
    @ApiOperation("企业分页查询")
    public CommonResult<PageList<QfEnterpriseInfo>> getList(@RequestBody QueryFilter<QfEnterpriseInfo> queryFilter) {
        return new CommonResult<PageList<QfEnterpriseInfo>>(true, I18nUtil.getMessage("option.success", LocaleContextHolder.getLocale()), super.query(queryFilter));
    }

    @PostMapping("/move")
    @ApiOperation("移动企业的上下级")
    public CommonResult<String> move(@RequestBody QfEnterpriseInfo entity) {
        boolean result = baseService.move(entity);
        if(!result) {
            return new CommonResult<String>(false, I18nUtil.getMessage("delete.fail", LocaleContextHolder.getLocale()), null);
        }
        return new CommonResult<String>(true, I18nUtil.getMessage("option.success", LocaleContextHolder.getLocale()), null);
    }

    @RequestMapping(value = "updateSequence", method = RequestMethod.POST, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "批量修改排序号", notes = "批量修改排序号")
    public CommonResult<String> updateSequence(@ApiParam(name = "params", value = "排序参数：Key：ID，Value：排序号") @RequestBody HashMap<String, Integer> params) throws Exception {
        if(!params.isEmpty()){
            baseService.updateSequence(params);
        }
        return new CommonResult<String>(true, I18nUtil.getMessage("option.success", LocaleContextHolder.getLocale()), null);
    }

    @PostMapping("/getAreaTree")
    @ApiOperation("获取按洲别划分的企业树")
    public CommonResult<List<EnterpriseInfoVo>> getAreaTree(@ApiParam(name="queryFilter", value="分页查询信息") @RequestBody QueryFilter<QfEnterpriseInfo> queryFilter) {
        List<EnterpriseInfoVo> list = baseService.getAreaTree(queryFilter);
        return CommonResult.success(list, null);
    }

    @GetMapping("findByCode/{code}")
    @ApiOperation("根据code获取企业信息")
    public CommonResult<String> findByCode(@ApiParam(name="code", value="实体id") @PathVariable String code) {
        QfEnterpriseInfo enterpriseInfo= baseService.findByCode(code);
        return CommonResult.success(enterpriseInfo, null);
    }
}
