package com.artfess.cqlt.manager.impl;

import com.alibaba.fastjson.JSONObject;
import com.artfess.base.manager.impl.BaseManagerImpl;
import com.artfess.cqlt.dao.QfEuroRatesDDao;
import com.artfess.cqlt.dao.QfEuroRatesMDao;
import com.artfess.cqlt.manager.QfEuroRatesDManager;
import com.artfess.cqlt.model.QfEuroRatesD;
import com.artfess.cqlt.model.QfEuroRatesM;
import com.artfess.cqlt.vo.EuroRatesRespVo;
import com.artfess.cqlt.vo.ReportReqVo;
import com.artfess.i18n.util.I18nUtil;
import com.artfess.poi.util.CustomHeader;
import com.artfess.poi.util.HeaderNode;
import com.google.api.client.util.Lists;
import com.google.common.collect.Maps;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 资金--欧元同业拆借利率详情表 服务实现类
 *
 * @author min.wu
 * @company 阿特菲斯信息技术有限公司
 * @since 2023-07-17
 */
@Service
public class QfEuroRatesDManagerImpl extends BaseManagerImpl<QfEuroRatesDDao, QfEuroRatesD> implements QfEuroRatesDManager {

    @Resource
    private QfEuroRatesMDao ratesMDao;


    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean batchUpdate(QfEuroRatesM t) {
        Assert.hasText(t.getId(), I18nUtil.getMessage("QfOperationKpiM.reportId", LocaleContextHolder.getLocale()));
        QfEuroRatesM euroRatesM = ratesMDao.selectById(t.getId());
        List<QfEuroRatesD> list = t.getList();
        list.forEach(detail -> {
            detail.setMainId(t.getId());
            detail.setFillDate(euroRatesM.getFillDate());
        });
        boolean b = this.saveOrUpdateBatch(list);
        return b;
    }

    @Override
    public List<JSONObject> detailQuery(List<QfEuroRatesD> list) {
        if (CollectionUtils.isEmpty(list)) {
            return Lists.newArrayList();
        }
        String mainId = list.get(0).getMainId();
        List<JSONObject> detailList = Lists.newArrayList();
        List<String> monthList = this.baseMapper.getMonthList(mainId);
        JSONObject detail = new JSONObject(true);
        detail.put("type", "");
        monthList.forEach(month -> {
            detail.put(month + "月", month + "月");
        });
        detailList.add(detail);

        Map<String, List<QfEuroRatesD>> collect = list.stream().collect(Collectors.groupingBy(QfEuroRatesD::getType));
        collect.forEach((type, v) -> {
            JSONObject data = new JSONObject(true);
            data.put("type", type);
            for (QfEuroRatesD pld : v) {
                data.put(pld.getMonth() + "月", pld.getFillData() + "%");
            }
            detailList.add(data);
        });
        return detailList;
    }

    @Override
    public List<EuroRatesRespVo> euroRatesAnalysis(ReportReqVo t) {
        return this.baseMapper.euroRatesAnalysis(t);
    }

    @Override
    public void export(HttpServletRequest request, HttpServletResponse response, List<QfEuroRatesD> list, String sheetName) throws IOException {
        if (CollectionUtils.isEmpty(list)) {
            return;
        }
        String mainId = list.get(0).getMainId();

        List<String> monthList = this.baseMapper.getMonthList(mainId);

        Map<String, Integer> rowMap = Maps.newHashMap();
        Map<String, Integer> columMap = Maps.newHashMap();
        List<HeaderNode> headerNodeList = Lists.newArrayList();

        HeaderNode headerNode = new HeaderNode();
        headerNode.setRow(0);
        headerNode.setColumn(0);
        headerNode.setHeaderName("");
        headerNodeList.add(headerNode);

        Integer column = 1;
        Integer row = 1;
        for (String month : monthList) {
            headerNode = new HeaderNode();
            headerNode.setRow(0);
            headerNode.setColumn(column);
            headerNode.setHeaderName(month +"月");
            headerNodeList.add(headerNode);
            columMap.put(month, headerNode.getColumn());
            column++;
        }
        Map<String, List<QfEuroRatesD>> collect = list.stream().collect(Collectors.groupingBy(QfEuroRatesD::getType));

        for (Map.Entry<String, List<QfEuroRatesD>> entry : collect.entrySet()) {
            String type = entry.getKey();
            headerNode = new HeaderNode();
            headerNode.setRow(row);
            headerNode.setColumn(0);
            headerNode.setHeaderName(type);
            headerNodeList.add(headerNode);
            rowMap.put(type, headerNode.getRow());
            row++;
        }

        for (Map.Entry<String, List<QfEuroRatesD>> entry : collect.entrySet()) {
            List<QfEuroRatesD> value = entry.getValue();
            value.forEach(detail->{
                HeaderNode node = new HeaderNode();
                node.setRow(rowMap.get(detail.getType()));
                node.setColumn(columMap.get(detail.getMonth()));
                node.setHeaderName(detail.getFillData().toString());
                headerNodeList.add(node);
            });
        }

        String date = new SimpleDateFormat("yyyyMMddHHmmss").format(new Date());
        String fileName = String.format(sheetName + "-%s", date);
        CustomHeader.export(headerNodeList, response, fileName, sheetName);
    }

}
