package com.artfess.cqlt.controller;

import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.controller.BaseController;
import com.artfess.base.exception.RequiredException;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.valid.AddGroup;
import com.artfess.base.valid.UpdateGroup;
import com.artfess.cqlt.manager.QfOperationSopDManager;
import com.artfess.cqlt.manager.QfOperationSopMManager;
import com.artfess.cqlt.model.QfOperationSopD;
import com.artfess.cqlt.model.QfOperationSopM;
import com.artfess.i18n.util.I18nUtil;
import com.artfess.poi.util.ExcelUtils;
import com.artfess.poi.util.FileDownloadUtil;
import com.artfess.uc.api.impl.util.ContextUtil;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.core.io.ClassPathResource;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.time.LocalDate;
import java.util.Arrays;
import java.util.List;

/**
 * 运营--新项目收入贡献填报主表 前端控制器
 *
 * @company 阿特菲斯信息技术有限公司
 * @author 黎沐华
 * @since 2023-02-21
 */
@Slf4j
@RestController
@Api(tags = "运营--新项目收入贡献填报主表")
@RequestMapping("/qf/operation/sop/")
@ApiGroup(group = {ApiGroupConsts.GROUP_BIZ})
public class QfOperationSopMController extends BaseController<QfOperationSopMManager, QfOperationSopM> {

    @Autowired
    private QfOperationSopDManager operationKpiDManager;

    @Override
    @PostMapping("/")
    @ApiOperation("添加实体的接口")
    public CommonResult<String> create(@ApiParam(name = "model", value = "实体信息") @RequestBody @Validated({AddGroup.class}) QfOperationSopM t) {
        t.setFillUser(ContextUtil.getCurrentUserId());
        t.setFillDate(LocalDate.now());
        boolean result = baseService.insertInfo(t);
        if (!result) {
            return new CommonResult<String>(false, I18nUtil.getMessage("option.fail", LocaleContextHolder.getLocale()), null);
        }

        return new CommonResult<String>(true, I18nUtil.getMessage("option.success", LocaleContextHolder.getLocale()), t.getId());
    }

    @Override
    @PutMapping("/")
    @ApiOperation("更新实体")
    public CommonResult<String> updateById(@ApiParam(name = "model", value = "实体信息") @RequestBody @Validated({UpdateGroup.class}) QfOperationSopM t) {
        boolean result = baseService.updateInfo(t);
        if (!result) {
            return new CommonResult<String>(false, I18nUtil.getMessage("option.fail", LocaleContextHolder.getLocale()), null);
        }
        return new CommonResult<String>(true, I18nUtil.getMessage("option.success", LocaleContextHolder.getLocale()), null);
    }

    @PutMapping("/updateStatus")
    @ApiOperation("生效")
    public CommonResult<String> updateStatus(@ApiParam(name = "model", value = "实体信息") @RequestBody QfOperationSopM t) {
        boolean result = baseService.updateStatus(t);
        if (!result) {
            return new CommonResult<String>(false, I18nUtil.getMessage("option.fail", LocaleContextHolder.getLocale()), null);
        }
        return new CommonResult<String>(true, I18nUtil.getMessage("option.success", LocaleContextHolder.getLocale()), null);
    }


    @Override
    @DeleteMapping("/{id}")
    @ApiOperation("根据id删除")
    public CommonResult<String> deleteById(@ApiParam(name = "id", value = "实体id") @PathVariable String id) {
        boolean result = baseService.removeById(id);
        if (!result) {
            return new CommonResult<String>(false, I18nUtil.getMessage("delete.fail", LocaleContextHolder.getLocale()), null);
        }
        return new CommonResult<String>(true, I18nUtil.getMessage("option.success", LocaleContextHolder.getLocale()), null);
    }

    @Override
    @DeleteMapping("/")
    @ApiOperation("根据id集合批量删除")
    public CommonResult<String> deleteByIds(@ApiParam(name = "ids", value = "实体集合") @RequestParam String... ids) {
        boolean result = baseService.removeByIds(Arrays.asList(ids));
        if (!result) {
            return new CommonResult<String>(false, I18nUtil.getMessage("delete.fail", LocaleContextHolder.getLocale()), null);
        }
        return new CommonResult<String>(true, I18nUtil.getMessage("option.success", LocaleContextHolder.getLocale()), null);
    }

    /**
     * @param file
     * @param mainId
     * @return
     */
    @ApiOperation(value = "新项目收入贡献数据导入")
    @PostMapping("/importExcel")
    public CommonResult<String> importExcel(@RequestParam("file") MultipartFile file, @RequestParam(value = "mainId", required = false) String mainId) {
        try {
            ExcelUtils<QfOperationSopD> excelUtil = new ExcelUtils<>(QfOperationSopD.class);
            List<QfOperationSopD> list = excelUtil.importExcel(null, file.getInputStream());
            boolean result = baseService.importExcel(list, mainId);
            if (!result) {
                throw new IllegalArgumentException(I18nUtil.getMessage("import.fail", LocaleContextHolder.getLocale()));
            }
            return new CommonResult<>();
        } catch (Exception e) {
            throw new IllegalArgumentException(I18nUtil.getMessage("import.fail", LocaleContextHolder.getLocale()) + "," + e.getMessage());
        }
    }

    /**
     * 下载导入模板
     *
     * @param response
     * @return
     */
    @ApiOperation(value = "下载导入模板")
    @GetMapping("/downModel")
    public void downTemplate(HttpServletResponse response) {
        try {
            ClassPathResource classPathResource = new ClassPathResource("model/sopDetail.xlsx");
            FileDownloadUtil.fileDownload(response, classPathResource.getInputStream(), "新项目收入贡献数据导入模板.xlsx");
        } catch (Exception e) {
            response.setCharacterEncoding("utf-8");
            throw new RequiredException(I18nUtil.getMessage("resources.noexist", LocaleContextHolder.getLocale()));
        }
    }

    @ApiOperation(value = "新项目收入贡献数据导出")
    @PostMapping("/export")
    public void export(HttpServletRequest request, HttpServletResponse response, @ApiParam(name = "queryFilter", value = "分页查询信息") @RequestBody QueryFilter<QfOperationSopD> queryFilter) throws Exception {
        PageList<QfOperationSopD> page = operationKpiDManager.query(queryFilter);
        ExcelUtils<QfOperationSopD> util = new ExcelUtils<QfOperationSopD>(QfOperationSopD.class);
        util.exportExcel(response, request, page.getRows(), "新项目收入贡献数据信息");
    }

}
