import { addCompRouter } from './compRouter'
import router from '@/router'
import { nextTick } from 'vue'

// 获取全局router实例

interface NavigationOptions {
    title: string
    parentRoutePath: string
    parentName: string
    path: string
    name: string
    componentUrl: any
}

/**
 * 公共路由跳转方法
 * @param options 路由配置选项
 */
export const navigateToRoute = async (
    options: NavigationOptions & { routeParams?: Record<string, any> }
) => {
    await addCompRouter({
        title: options.title,
        parentRoutePath: options.parentRoutePath,
        parentName: options.parentName,
        path: options.path,
        name: options.name,
        componentUrl: options.componentUrl
    })

    // 确保路由已添加后跳转，并传递参数
    await nextTick()
    router.push({
        name: options.name,
        query: options.routeParams || {}
    })
}

/**
 * 从basePath提取parentName
 * @param basePath 基础路径
 */
const extractParentName = (basePath: string): string => {
    const segments = basePath.split('/').filter(Boolean)
    return segments[segments.length - 1]
}

interface BaseHandlerOptions {
    basePath: string
    componentUrl: any
    parentName?: string
    routeParams?: Record<string, any>
}

/**
 * 创建新增页面跳转方法
 * @param options 基础选项
 */
export const createAddHandler = async (
    options: BaseHandlerOptions & { params?: Record<string, any> }
) => {
    const { basePath, componentUrl, parentName, routeParams = {}, params = {} } = options
    const name = parentName || extractParentName(basePath)
    const routeName = `${name}Add`

    await navigateToRoute({
        title: '新增',
        parentRoutePath: 'platLayout',
        parentName: name,
        path: `${basePath}/Add`,
        name: routeName,
        componentUrl,
        routeParams: { ...routeParams, ...params }
    })
}

/**
 * 创建编辑页面跳转方法
 * @param options 基础选项
 */
export const createEditHandler = async (
    options: BaseHandlerOptions & { params?: Record<string, any> }
) => {
    const { basePath, componentUrl, parentName, routeParams = {}, params = {} } = options
    const name = parentName || extractParentName(basePath)
    const routeName = `${name}Edit`

    await navigateToRoute({
        title: '编辑',
        parentRoutePath: 'platLayout',
        parentName: name,
        path: `${basePath}/Edit`,
        name: routeName,
        componentUrl,
        routeParams: { ...routeParams, ...params }
    })
}

/**
 * 创建详情页面跳转方法
 * @param options 基础选项
 */
export const createDetailHandler = async (
    options: BaseHandlerOptions & { params?: Record<string, any> }
) => {
    const { basePath, componentUrl, parentName, routeParams = {}, params = {} } = options
    const name = parentName || extractParentName(basePath)
    const routeName = `${name}Detail`

    await navigateToRoute({
        title: '详情',
        parentRoutePath: 'platLayout',
        parentName: name,
        path: `${basePath}/Detail`,
        name: routeName,
        componentUrl,
        routeParams: { ...routeParams, ...params }
    })
}

interface CustomHandlerOptions {
    basePath: string
    componentUrl: any
    pageType: string
    title: string
    parentName?: string
    parentRoutePath?: string
    routeParams?: Record<string, any>
}

/**
 * 创建自定义页面跳转方法
 * @param options 自定义选项
 */
export const createCustomHandler = async (
    options: CustomHandlerOptions & { params?: Record<string, any> }
) => {
    const {
        basePath,
        componentUrl,
        pageType,
        title,
        parentName,
        parentRoutePath = 'platLayout',
        routeParams = {},
        params = {}
    } = options

    const name = parentName || extractParentName(basePath)
    // const routeName = `${name}${pageType.charAt(0).toUpperCase() + pageType.slice(1)}`
    const routeName = pageType
    await navigateToRoute({
        title,
        parentRoutePath,
        parentName: name,
        path: `${basePath}/${pageType}`,
        name: routeName,
        componentUrl,
        routeParams: { ...routeParams, ...params }
    })
}
