package com.artfess.examine.manager.impl;

import com.alibaba.fastjson.JSON;
import com.artfess.base.constants.CodePrefix;
import com.artfess.base.enums.PaperWayTypeEnum;
import com.artfess.base.enums.QuestionStateEnum;
import com.artfess.base.enums.QuestionTypeEnum;
import com.artfess.base.manager.impl.BaseManagerImpl;
import com.artfess.base.query.PageBean;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.util.AuthenticationUtil;
import com.artfess.examine.dao.ExamImitateRecordDao;
import com.artfess.examine.dao.ExamPaperBaseDao;
import com.artfess.examine.dao.ExamQuestionsOptionDao;
import com.artfess.examine.manager.ExamImitateRecordDetailManager;
import com.artfess.examine.manager.ExamImitateRecordManager;
import com.artfess.examine.manager.ExamPaperBaseManager;
import com.artfess.examine.manager.ExamPaperSettingManager;
import com.artfess.examine.model.ExamImitateRecord;
import com.artfess.examine.model.ExamImitateRecordDetail;
import com.artfess.examine.model.ExamPaperBase;
import com.artfess.examine.model.ExamPaperSetting;
import com.artfess.examine.model.ExamQuestionsInfo;
import com.artfess.examine.model.ExamQuestionsOption;
import com.artfess.examine.vo.ExamReqVo;
import com.artfess.examine.vo.MyExamInfoVo;
import com.artfess.examine.vo.QuestionOptionReqVo;
import com.artfess.examine.vo.QuestionsInfoVo;
import com.artfess.examine.vo.SubmitAnswerReqVo;
import com.artfess.redis.util.RedisUtil;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.List;
import java.util.Map;
import java.util.TimeZone;
import java.util.stream.Collectors;

/**
 * 考生模拟考试记录（人员考试成绩） 服务实现类
 *
 * @company 阿特菲斯信息技术有限公司
 * @author min.wu
 * @since 2022-10-19
 */
@Slf4j
@Service
public class ExamImitateRecordManagerImpl extends BaseManagerImpl<ExamImitateRecordDao, ExamImitateRecord> implements ExamImitateRecordManager {

    @Autowired
    private ExamPaperBaseManager paperBaseManager;

    @Autowired
    private ExamImitateRecordDetailManager imitateRecordDetailManager;

    @Autowired
    private ExamPaperSettingManager paperSettingManager;

    @Resource
    private ExamQuestionsOptionDao questionsOptionDao;

    @Resource
    private ExamPaperBaseDao paperBaseDao;

    @Autowired
    private RedisUtil redisUtil;

    @Override
    public void startPaper(String id) {


    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void createImitate(ExamPaperBase examPaperBase) {

        QueryWrapper<ExamImitateRecord> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("paper_base_id_", examPaperBase.getId());
        baseMapper.delete(queryWrapper);

        //随机出卷每个考生试卷都是根据规则随机生成
        ExamImitateRecord imitateRecord = new ExamImitateRecord();
        imitateRecord.setPaperBaseId(examPaperBase.getId());
        imitateRecord.setUserId(AuthenticationUtil.getCurrentUserId());
        imitateRecord.setUserName(AuthenticationUtil.getCurrentUsername());
        imitateRecord.setStatus(QuestionStateEnum.toBeAnswer.getType());
        baseMapper.insert(imitateRecord);

        processUserRecordDetail(imitateRecord, examPaperBase);
    }

    @Override
    public MyExamInfoVo getUserRecord(String id) {
        ExamImitateRecord imitateRecord = baseMapper.selectById(id);
        Assert.notNull(imitateRecord, "考试任务不存在，请联系管理员");
        ExamReqVo reqVo = new ExamReqVo();
        reqVo.setRecordId(id);
        List<QuestionsInfoVo> questionList = baseMapper.getQuestionList(reqVo);
        if(CollectionUtils.isEmpty(questionList)) {
            throw new RuntimeException("当前试卷未找到对应题目信息");
        }
        List<String> questionIds = questionList.stream().map(QuestionsInfoVo::getQuestionId).collect(Collectors.toList());
        QueryWrapper<ExamQuestionsOption> optionQueryWrapper = new QueryWrapper<>();
        optionQueryWrapper.in("question_id_", questionIds);
        optionQueryWrapper.orderByAsc("option_key_");
        List<ExamQuestionsOption> examQuestionsOptions = questionsOptionDao.selectList(optionQueryWrapper);

        Map<String, List<ExamQuestionsOption>> map = examQuestionsOptions.stream().collect(Collectors.groupingBy(ExamQuestionsOption::getQuestionId));
        questionList.forEach(question -> {
            if(StringUtils.isEmpty(question.getResult())) {
                question.setActualScore(BigDecimal.ZERO);
            }
            if(!map.containsKey(question.getQuestionId())){
                return;
            }
            question.setOptions(map.get(question.getQuestionId()));
        });
        MyExamInfoVo myExamInfoVo = new MyExamInfoVo();
        BeanUtils.copyProperties(imitateRecord, myExamInfoVo);
        reqVo.setRecordId(imitateRecord.getId());
        myExamInfoVo.setQuestionsInfoVos(questionList);
        myExamInfoVo.setRecordId(imitateRecord.getId());
        return myExamInfoVo;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public MyExamInfoVo startExam(ExamReqVo reqVo) {
        Assert.hasText(reqVo.getPaperId(), "考试试卷id不能为空");
        QueryWrapper<ExamImitateRecord> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("user_id_", reqVo.getUserId());
        queryWrapper.eq("paper_base_id_", reqVo.getPaperId());
        ExamImitateRecord imitateRecord = baseMapper.selectOne(queryWrapper);
        Assert.notNull(imitateRecord, "考试任务不存在，请联系管理员");

        QueryWrapper<ExamPaperSetting> query = new QueryWrapper<>();
        query.eq("paper_id_", imitateRecord.getPaperBaseId());
        ExamPaperSetting paperSetting = paperSettingManager.getOne(query);
        ExamPaperBase examPaperBase = paperBaseDao.selectById(imitateRecord.getPaperBaseId());
        if (QuestionStateEnum.haveTest.getType().equals(imitateRecord.getStatus())) {
            throw new RuntimeException("当前试卷您已交卷，不能再次考试");
        }

        if (QuestionStateEnum.finish.getType().equals(imitateRecord.getStatus())) {
            throw new RuntimeException("当前试卷已阅卷，不能再次考试");
        }

        if (QuestionStateEnum.zuobi.getType().equals(imitateRecord.getStatus())) {
            throw new RuntimeException("考试中有作弊行为，已被强制交卷");
        }

        MyExamInfoVo myExamInfoVo = new MyExamInfoVo();
        BeanUtils.copyProperties(imitateRecord, myExamInfoVo);
        reqVo.setRecordId(imitateRecord.getId());
        myExamInfoVo.setTotalNumber(examPaperBase.getTotalNumber());
        myExamInfoVo.setTotalScore(examPaperBase.getTotalScore());
        //总时长（毫秒）
        BigDecimal totalTime = paperSetting.getTimeLength().multiply(new BigDecimal(60)).multiply(new BigDecimal(1000));
        //第一次考试
        //1.创建答题redis定时任务
        //2.修改考试任务状态，更新开始考试时间
        imitateRecord.setStatus(QuestionStateEnum.inTest.getType());
        //考虑接口延迟
        imitateRecord.setStartTime(LocalDateTime.now());
        //修改或创建考试活动任务并修改剩余考试时间
        myExamInfoVo.setTimeLength(totalTime);
        createUserTask(reqVo, paperSetting.getTimeLength());
        baseMapper.updateById(imitateRecord);
        List<QuestionsInfoVo> questionList = baseMapper.getQuestionList(reqVo);
        List<String> questionIds = questionList.stream().map(QuestionsInfoVo::getQuestionId).collect(Collectors.toList());
        QueryWrapper<ExamQuestionsOption> optionQueryWrapper = new QueryWrapper<>();
        optionQueryWrapper.in("question_id_", questionIds);
        optionQueryWrapper.orderByAsc("option_key_");
        List<ExamQuestionsOption> examQuestionsOptions = questionsOptionDao.selectList(optionQueryWrapper);

        Map<String, List<ExamQuestionsOption>> map = examQuestionsOptions.stream().collect(Collectors.groupingBy(ExamQuestionsOption::getQuestionId));
        questionList.forEach(question -> {
            question.setRightOption(null);
            question.setResult(null);
            if(!map.containsKey(question.getQuestionId())){
                return;
            }

            question.setOptions(map.get(question.getQuestionId()));
        });
        myExamInfoVo.setQuestionsInfoVos(questionList);
        myExamInfoVo.setRecordId(imitateRecord.getId());
        return myExamInfoVo;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void submitAnswer(SubmitAnswerReqVo reqVo) {
        ExamImitateRecord imitateRecord = baseMapper.selectById(reqVo.getRecordId());
        Assert.notNull(imitateRecord, "模拟考试不存在，请联系管理员");
        QueryWrapper<ExamPaperSetting> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("paper_id_", imitateRecord.getPaperBaseId());
        ExamPaperSetting paperSetting = paperSettingManager.getOne(queryWrapper);
        Assert.notNull(paperSetting, "考试信息不存在");
        if (QuestionStateEnum.haveTest.getType().equals(imitateRecord.getStatus())) {
            throw new RuntimeException("当前试卷您已交卷，不能再次考试");
        }

        if (QuestionStateEnum.finish.getType().equals(imitateRecord.getStatus())) {
            throw new RuntimeException("当前试卷已阅卷，不能再次考试");
        }

        if (QuestionStateEnum.zuobi.getType().equals(imitateRecord.getStatus())) {
            throw new RuntimeException("考试中有作弊行为，已被强制交卷");
        }


        //已答的不做修改
        List<QuestionOptionReqVo> options = reqVo.getQuestionOptionList();
        //将用户回答的问题以及问题答案放入map中，跟正确答案作对比
        Map<String,String> userOptionNumber = Maps.newHashMap();
        options.forEach(option -> {
            if(userOptionNumber.containsKey(option.getQuestionId())){
                userOptionNumber.put(option.getQuestionId(), userOptionNumber.get(option.getQuestionId()) + "," + option.getResult());
            }else{
                userOptionNumber.put(option.getQuestionId(), option.getResult());
            }
        });

        if(reqVo.getStatus() == 1) {
            imitateRecord.setStatus(QuestionStateEnum.inTest.getType());
        }

        //判断是否交卷
        if (null != reqVo.getStatus() && reqVo.getStatus() == 1) {
            imitateRecord.setStatus(QuestionStateEnum.inTest.getType());
            long time = System.currentTimeMillis() - imitateRecord.getStartTime().toInstant(ZoneOffset.of("+8")).toEpochMilli();
            //初始化Formatter的转换格式。
            SimpleDateFormat formatter = new SimpleDateFormat("HH:mm:ss");
            formatter.setTimeZone(TimeZone.getTimeZone("GMT+00:00"));
            String answerTime = formatter.format(time);
            imitateRecord.setAnswerTime(answerTime);
            imitateRecord.setEndTime(LocalDateTime.now());
        }
        //如果是试卷则给答题内容打分
        calculateScore(userOptionNumber, imitateRecord);
    }

    @Override
    public PageList<ExamImitateRecord> trainingPaperQuery(QueryFilter<ExamImitateRecord> queryFilter) {
        PageBean pageBean = queryFilter.getPageBean();
        IPage<ExamImitateRecord> result = baseMapper.trainingPaperQuery(convert2IPage(pageBean), convert2Wrapper(queryFilter, currentModelClass()));

        return new PageList<ExamImitateRecord>(result);
    }

    @Override
    public List<QuestionsInfoVo> errorQuestionsList(String id) {

        List<QuestionsInfoVo> questionList = this.baseMapper.errorQuestionsList(id);
        List<String> questionIds = questionList.stream().map(QuestionsInfoVo::getQuestionId).collect(Collectors.toList());
        QueryWrapper<ExamQuestionsOption> optionQueryWrapper = new QueryWrapper<>();
        optionQueryWrapper.in("question_id_", questionIds);
        optionQueryWrapper.orderByAsc("option_key_");

        List<ExamQuestionsOption> examQuestionsOptions = questionsOptionDao.selectList(optionQueryWrapper);
        Map<String, List<ExamQuestionsOption>> map = examQuestionsOptions.stream().collect(Collectors.groupingBy(ExamQuestionsOption::getQuestionId));
        questionList.forEach(question -> {
            if (!map.containsKey(question.getQuestionId())) {
                return;
            }
            question.setOptions(map.get(question.getQuestionId()));
        });
        questionList.sort((o1, o2) -> o1.getType().compareTo(o2.getType()));
        return questionList;
    }

    /**
     * 给用户回答的问题进行打分
     * @param userOptionNumber
     * @param record
     */
    private void calculateScore(Map<String, String> userOptionNumber, ExamImitateRecord record) {
        log.info("问题选项：{}", userOptionNumber);
        //如果已经生成试卷题目则直接返回生成的记录
        QueryWrapper<ExamImitateRecordDetail> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("record_id_", record.getId());
        List<ExamImitateRecordDetail> list = imitateRecordDetailManager.getBaseMapper().selectList(queryWrapper);
        Integer totalScore = 0;
        for (ExamImitateRecordDetail question : list){
            String result = userOptionNumber.get(question.getQuestionId());
            question.setResult(result);
            //未回答题目直接跳过
            if(!userOptionNumber.containsKey(question.getQuestionId())){
                continue;
            }
            if(QuestionTypeEnum.radio.getType().equals(question.getQuestionType())
                    || QuestionTypeEnum.multi.getType().equals(question.getQuestionType())
                    || QuestionTypeEnum.judge.getType().equals(question.getQuestionType())){
                if(result.equals(question.getRightKey())){
                    question.setActualScore(question.getScore());
                    question.setIsRight("1");
                }else{
                    question.setActualScore(BigDecimal.ZERO);
                    question.setIsRight("0");
                }
                totalScore += question.getActualScore().intValue();
            }
            imitateRecordDetailManager.updateById(question);
        }
        log.info("问题详情：{}", JSON.toJSONString(list));
        record.setTotalScore(new BigDecimal(totalScore));
        baseMapper.updateById(record);
    }

    private void processUserRecordDetail(ExamImitateRecord imitateRecord, ExamPaperBase examPaperBase) {
        //随机出卷每个考生试卷都是根据规则随机生成
        if(PaperWayTypeEnum.sjcj.getType().equals(examPaperBase.getWayType())) {
            List<ExamQuestionsInfo> examQuestionsInfos = paperBaseManager.processQuestionList(examPaperBase);
            createUserRecordDetail(imitateRecord, examQuestionsInfos, examPaperBase);

        } else {
            List<ExamQuestionsInfo> examQuestionsInfos = paperBaseManager.processQuestionList(examPaperBase);

            createUserRecordDetail(imitateRecord, examQuestionsInfos, examPaperBase);
        }
    }

    private void createUserRecordDetail(ExamImitateRecord record, List<ExamQuestionsInfo> examQuestionsInfos, ExamPaperBase examPaperBase) {
        //保存考生试卷信息
        //保存考生题目信息
        List<ExamImitateRecordDetail> details = Lists.newArrayList();
        examQuestionsInfos.forEach(question->{
            ExamImitateRecordDetail userRecordDetail = new ExamImitateRecordDetail();
            userRecordDetail.setRecordId(record.getId());
            userRecordDetail.setQuestionId(question.getId());
            userRecordDetail.setQuestionType(question.getType());
            userRecordDetail.setRightKey(question.getRightOption());
            userRecordDetail.setScore(question.getScore());
            details.add(userRecordDetail);
        });

        if(!CollectionUtils.isEmpty(details)) {
            imitateRecordDetailManager.saveBatch(details);
        }
    }

    private void createUserTask(ExamReqVo reqVo, BigDecimal timeLength) {
        String key = CodePrefix.PAPER_TASK_KEY.getKey() + ":" + reqVo.getRecordId();
        Long time = timeLength.longValue() * 60;
        redisUtil.set(key, JSON.toJSON(reqVo), time);
    }

    private void getBalanceAwswerTime(ExamReqVo reqVo) {

        String key = CodePrefix.PAPER_TASK_KEY.getKey() + ":" + reqVo.getRecordId();
        Object taskInfo = redisUtil.get(key);
        Assert.notNull(taskInfo, "考试已结束");
    }
}
