package com.artfess.examine.controller;


import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.controller.BaseController;
import com.artfess.base.enums.QuestionTypeEnum;
import com.artfess.base.enums.ResponseErrorEnums;
import com.artfess.base.exception.RequiredException;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.Direction;
import com.artfess.base.query.FieldSort;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.valid.AddGroup;
import com.artfess.base.valid.UpdateGroup;
import com.artfess.examine.manager.ExamQuestionsInfoManager;
import com.artfess.examine.model.ExamQuestionsInfo;
import com.artfess.examine.vo.SubjectQuestionTypeVo;
import com.artfess.poi.util.ExcelUtils;
import com.artfess.poi.util.FileDownloadUtil;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.core.io.ClassPathResource;
import org.springframework.util.Assert;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.List;

/**
 * 题库试题信息 前端控制器
 *
 * @company 阿特菲斯信息技术有限公司
 * @author min.wu
 * @since 2022-10-19
 */
@Slf4j
@RestController
@Api(tags = "基础数据-题库管理")
@RequestMapping("/exam/questions/info/")
@ApiGroup(group = {ApiGroupConsts.GROUP_BIZ})
public class ExamQuestionsInfoController extends BaseController<ExamQuestionsInfoManager, ExamQuestionsInfo> {

    @Override
    @PostMapping(value="/query", produces={"application/json; charset=utf-8" })
    @ApiOperation("分页查询结果")
    public PageList<ExamQuestionsInfo> query(@ApiParam(name="queryFilter", value="分页查询信息") @RequestBody QueryFilter<ExamQuestionsInfo> queryFilter) {
        List<FieldSort> sorter = queryFilter.getSorter();
        sorter.add(new FieldSort("q.type_", Direction.ASC));
        return baseService.findByPage(queryFilter);
    }

    @Override
    @PostMapping("/")
    @ApiOperation("添加实体的接口")
    public CommonResult<String> create(@ApiParam(name="model", value="实体信息") @RequestBody @Validated({AddGroup.class}) ExamQuestionsInfo t) {
        String id = baseService.createInfo(t);
        if(!StringUtils.isNotBlank(id)) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, null);
        }
        return new CommonResult<>();
    }

    @Override
    @PutMapping("/")
    @ApiOperation("更新实体")
    public CommonResult<String> updateById(@ApiParam(name="model", value="实体信息") @RequestBody @Validated({UpdateGroup.class})  ExamQuestionsInfo t) {
        String id = baseService.updateInfo(t);
        if(!StringUtils.isNotBlank(id)) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, "更新实体失败");
        }
        return new CommonResult<>();
    }

    @Override
    @GetMapping("/{id}")
    @ApiOperation("根据id查询实体")
    public ExamQuestionsInfo getById(@ApiParam(name="id", value="实体id") @PathVariable String id) {
        return baseService.findById(id);
    }

    @GetMapping("/findBySubjectId/{subjectId}")
    @ApiOperation("根据科目id获取题型统计情况")
    public List<SubjectQuestionTypeVo> findBySubjectId(@ApiParam(name="subjectId", value="科目id") @PathVariable String subjectId) {
        return baseService.findBySubjectId(subjectId);
    }

    @ApiOperation(value = "导入")
    @PostMapping("/importExcel")
    public CommonResult<String> importExcel(@RequestParam("file") MultipartFile file, @RequestParam(value = "subjectId", required = false) String subjectId) {
        try {
            Assert.hasText(subjectId, "请选择课目");
            ExcelUtils<ExamQuestionsInfo> excelUtil = new ExcelUtils<>(ExamQuestionsInfo.class);
            List<ExamQuestionsInfo> list = excelUtil.importQuestionExcel(null, file.getInputStream());
            list.forEach(question -> {
                question.setSubId(subjectId);
                boolean result = baseService.importExcel(question);
                if(!result) {
                    throw new IllegalArgumentException("导入失败");
                }
            });
            return new CommonResult<>();
        } catch (Exception e) {
            throw new IllegalArgumentException("导入失败," + e.getMessage());
        }
    }

    @ApiOperation(value = "导出")
    @PostMapping("/export")
    public void export(HttpServletResponse response, HttpServletRequest request,
                       @ApiParam(name="queryFilter", value="分页查询信息") @RequestBody QueryFilter<ExamQuestionsInfo> queryFilter) throws Exception {
        PageList<ExamQuestionsInfo> page = baseService.query(queryFilter);
        page.getRows().forEach(question->{
            question.setType(QuestionTypeEnum.getDesc(question.getType()));
        });
        ExcelUtils<ExamQuestionsInfo> util = new ExcelUtils<ExamQuestionsInfo>(ExamQuestionsInfo.class);
        util.exportExcel(response, request, page.getRows(), "问题信息");
    }

    /**
     * 下载导入模板
     *
     * @param response
     * @return
     */
    @ApiOperation(value = "下载导入模板")
    @GetMapping("/downModel")
    public void downTemplate(HttpServletResponse response) {
        try {
            ClassPathResource classPathResource = new ClassPathResource("model/questionInfo.xlsx");
            FileDownloadUtil.fileDownload(response, classPathResource.getInputStream(), "题库模板.xlsx");
        } catch (Exception e) {
            response.setCharacterEncoding("utf-8");
            throw new RequiredException("你所下载的资源不存在");
        }
    }

}
