package com.artfess.yhxt.basedata.controller;

import com.artfess.yhxt.basedata.manager.RouteManager;
import com.artfess.yhxt.basedata.model.Route;
import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.controller.BaseController;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.PageBean;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.query.QueryOP;
import com.artfess.base.util.BeanUtils;
import com.artfess.base.util.JsonUtil;
import com.artfess.base.util.StringUtil;
import com.artfess.poi.util.ExcelUtil;
import com.artfess.uc.api.impl.util.ContextUtil;
import com.artfess.uc.api.model.IUser;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;
import java.util.*;
/**
 * 路线前端控制器
 *
 * @company 阿特菲斯信息技术有限公司
 * @author wangping
 * @since 2021-07-30
 */
@RestController
@RequestMapping(value = "/route/routemanage/v1")
@Api(tags = "路线管理")
@ApiGroup(group = {ApiGroupConsts.GROUP_BIZ})
public class RouteController extends BaseController<RouteManager, Route> {
    @Resource
    RouteManager routeManager;

    /**
     * 路线管理列表(分页条件查询)数据
     *
     * @param queryFilter
     * @return
     * @throws Exception PageJson
     * @throws
     */
    @PostMapping("/list")
    @ApiOperation(value = "路线管理数据列表", httpMethod = "POST", notes = "获取路线管理列表")
    public PageList<Route> list(@ApiParam(name = "queryFilter", value = "查询对象") @RequestBody QueryFilter<Route> queryFilter) throws Exception {
        return routeManager.query(queryFilter);
    }

    /**
     * 路线管理明细页面
     *
     * @param id
     * @return
     * @throws Exception ModelAndView
     */
    @GetMapping(value = "/get/{id}")
    @ApiOperation(value = "路线管理数据详情", httpMethod = "GET", notes = "路线管理数据详情")
    public Route get(@ApiParam(name = "id", value = "业务对象主键", required = true) @PathVariable String id) throws Exception {
        return routeManager.get(id);
    }

    /**
     * 新增路线管理
     *
     * @param Route
     * @return
     * @throws Exception
     * @throws
     */
    @PostMapping(value = "/save")
    @ApiOperation(value = "新增,更新路线管理数据", httpMethod = "POST", notes = "新增,更新路线管理数据")
    public CommonResult<String> save(@ApiParam(name = "Route", value = "路线管理业务对象", required = true) @RequestBody Route Route) throws Exception {
        String msg = "添加路线管理成功";
        if (StringUtil.isEmpty(Route.getId())) {
            Route.setIsDele("0");
            HashMap<String, Object> map = new HashMap<>();
            IUser user= ContextUtil.getCurrentUser();
            if(ObjectUtils.isNotEmpty(user)){
                map.put("tenant_id_", user.getTenantId());
            }
            int nextSequence =baseService.getNextSequence(map);
            Route.setSn(nextSequence);
            routeManager.create(Route);
        } else {
            routeManager.update(Route);
            msg = "更新路线管理成功";
        }
        return new CommonResult<String>(msg);
    }

    /**
     * 删除路线管理记录
     *
     * @param id
     * @return
     * @throws Exception
     * @throws
     */
    @DeleteMapping(value = "/remove/{id}")
    @ApiOperation(value = "删除路线管理记录", httpMethod = "DELETE", notes = "删除路线管理记录")
    public CommonResult<String> remove(@ApiParam(name = "id", value = "业务主键", required = true) @PathVariable String id) throws Exception {
        routeManager.remove(id);
        return new CommonResult<String>(true, "删除成功");
    }

    /**
     * 批量删除路线管理记录
     *
     * @param ids
     * @return
     * @throws Exception
     * @throws
     */
    @DeleteMapping(value = "/removes")
    @ApiOperation(value = "批量删除路线管理记录", httpMethod = "DELETE", notes = "批量删除路线管理记录")
    public CommonResult<String> removes(@ApiParam(name = "ids", value = "多个主键之间用逗号分隔", required = true) @RequestParam String... ids) throws Exception {
        routeManager.removeByIds(Arrays.asList(ids));
        return new CommonResult<String>(true, "删除成功");
    }

    @RequestMapping(value="/getList",method = RequestMethod.POST, produces={"application/json; charset=utf-8" })
    @ApiOperation(value = "不分页查询路线信息",httpMethod = "POST")
    public PageList<Route> getList(@ApiParam(name="queryFilter", value="不分页查询信息") @RequestBody QueryFilter<Route> queryFilter) {
        queryFilter.addFilter("is_dele_", "0", QueryOP.EQUAL);
        PageBean pageBean = queryFilter.getPageBean();
        pageBean.setPageSize(PageBean.WITHOUT_PAGE);
        pageBean.setPage(1);
        queryFilter.setPageBean(pageBean);
        PageList<Route>   route = baseService.queryRoute(queryFilter);
        return route;
    }

    @RequestMapping(value="/getJson",method = RequestMethod.POST, produces={"application/json; charset=utf-8" })
    @ApiOperation(value = "分页查询路线信息",httpMethod = "POST")
    public PageList<Route> getJson(@ApiParam(name="queryFilter", value="分页查询信息") @RequestBody QueryFilter<Route> queryFilter) {
        queryFilter.addFilter("is_dele_", "0", QueryOP.EQUAL);
        PageList<Route> pageList = baseService.queryRoute(queryFilter);
        return  pageList;
    }

    @RequestMapping(value = "/updateBySn", method = RequestMethod.POST, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "修改排序号", httpMethod = "POST")
    public CommonResult<String> updateBySn(@ApiParam(name = "params", value = "排序参数：Key：ID，Value：排序号") @RequestBody HashMap<String, Integer> params) throws Exception {
        baseService.updateSequence(params);
        return new CommonResult<>(true, "排序完成");
    }

    @RequestMapping(value = "updateBatch", method = RequestMethod.POST, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "逻辑删除", httpMethod = "POST", notes = "逻辑删除")
    public CommonResult<String> updateBatchRoute(@ApiParam(name = "id", value = "路线ID", required = true) @RequestParam(value = "id", required = true) String id) throws Exception {

        if (StringUtils.isNotBlank(id))
            routeManager.updateRoute(id);
        return new CommonResult<>(true, "删除成功");
    }

    @RequestMapping(value = "updateRoute", method = RequestMethod.DELETE, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "批量逻辑删除", httpMethod = "DELETE", notes = "批量逻辑删除")
    public CommonResult<String> updateRoute(
            @ApiParam(name = "ids", value = "路线ID集合以，隔开", required = true)
            @RequestParam(value = "ids", required = true) String ids) throws Exception {
        if (StringUtils.isNotBlank(ids)) {
            List<String> idsList = Arrays.asList(ids.split(","));
            UpdateWrapper<Route> updateWrapper = new UpdateWrapper<>();
            updateWrapper.in("id_", idsList);
            updateWrapper.set("is_dele_", "1");
            routeManager.update(null, updateWrapper);

        }

        return new CommonResult<>(true, "批量删除成功");
    }

    @RequestMapping(value="route/export" ,method = RequestMethod.GET, produces = {"application/json; charset=utf-8"})
    @ApiOperation(value = "路线导出", httpMethod = "GET", notes = "路线导出")
    public void export(HttpServletResponse response, @ApiParam(name="ids",value="ids", required = true)@RequestParam String ids) throws 	Exception{
        QueryFilter filter = QueryFilter.build().withPage(new PageBean(1, PageBean.WITHOUT_PAGE));
        filter.addFilter("id", ids.split(","), QueryOP.IN);
        PageList<Route> routes = routeManager.query(filter);
        List<Map<String,Object>> list = new ArrayList<>();
        if(BeanUtils.isNotEmpty(routes.getRows())){
            for (Route route : routes.getRows()) {
                list.add(JsonUtil.toMap(JsonUtil.toJson(route)));
            }
        }
        Map<String,String> exportMap = new LinkedHashMap<>();
        exportMap.put("number","路线编码");
        exportMap.put("name","路线名称");
        exportMap.put("start","起点");
        exportMap.put("end","终点");
        HSSFWorkbook exportFile = ExcelUtil.exportExcel("路线列表", 22, exportMap, list);
        ExcelUtil.downloadExcel(exportFile,"路线导出",response);
    }
}
