package com.artfess.cqxy.contract.controller;

import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.controller.BaseController;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.util.BeanUtils;
import com.artfess.cqxy.contract.manager.ItemManager;
import com.artfess.cqxy.contract.model.Item;
import com.artfess.cqxy.processManagermant.model.ProgressManageDefinite;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.util.Assert;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.Arrays;
import java.util.List;

/**
 * 合同明细表(BizContractItem)表控制层
 *
 * @author 黎沐华
 * @since 2022-02-26 12:57:01
 */
@Slf4j
@RestController
@Api(tags = "合同管理-合同明细接口")
@ApiGroup(group = {ApiGroupConsts.GROUP_BIZ})
@RequestMapping("/biz/contract/item/v1")
public class ItemController extends BaseController<ItemManager, Item> {
    /**
     * 保存或修改数据
     * @param item 实体对象
     * @return 新增或更新结果
     */
    @PostMapping("/save")
    @ApiOperation(value = "S-保存或修改数据",httpMethod = "POST")
    public CommonResult insert(@RequestBody Item item) {
        log.debug("合同明细-保存或修改数据接口请求参数：{}", item);
        Assert.notNull(item.getContractId(),"合同ID不能为空");
        boolean add = StringUtils.isEmpty(item.getId());
        boolean save = baseService.saveOrUpdate(item);
        String msg = (add ? "添加" : "修改")+(save ? "成功" : "失败");
        log.debug("合同明细-保存或修改数据接口响应结果：{}", msg);
        return new CommonResult(save, msg);
    }

    /**
     * 读取Excel数据
     * @param file Excel文件
     */
    @PostMapping("/readExcelData")
    @ApiOperation(value = "S-读取Excel数据")
    public CommonResult<List<Item>> readExcelData (@ApiParam("上传文件") @RequestParam("file") MultipartFile file){
        log.debug("读取Excel数据接口请求参数：{}",file);
        List<Item> result = baseService.readExcelData(file);
        return new CommonResult<List<Item>>(true,"读取成功",result);
    }

    /**
     * 批量导入Excel数据
     * @param file Excel文件
     */
    @PostMapping("/importExcelData")
    @ApiOperation(value = "S-批量导入Excel数据")
    public CommonResult importExcelData (@ApiParam("上传文件") @RequestParam("file") MultipartFile file, String contractId){
        log.debug("批量导入Excel数据接口请求参数：{}",file, contractId);
        baseService.importExcelData(file, contractId);
        return new CommonResult<>("导入成功");
    }

    /**
     * 导出数据到Excel
     * @param queryFilter 通用查询器
     * @param response 响应对象
     * @return 通用结果
     * @throws IOException IO异常
     */
    @PostMapping("/exportDataToExcel")
    @ApiOperation(value = "S-导出数据到Excel",notes = "参数说明：查询器只使到了query组传参，传入CONTRACT_ID_导出该合同下的数据，否则将导出所有数据")
    public CommonResult exportDataToExcel (@ApiParam(name = "queryFilter",value = "通用查询器")@RequestBody QueryFilter<Item> queryFilter, HttpServletResponse response) throws IOException {
        log.debug("导出数据到Excel接口请求参数：{}",queryFilter,response);
        baseService.exportDatatoExcel(queryFilter,response);
        return new CommonResult<>("导出成功");
    }

    /**
     * 删除数据
     * @param ids 主键集合
     * @return 删除结果
     */
    @PostMapping("/delete")
    @ApiOperation(value = "S-删除数据",httpMethod = "POST")
    public CommonResult delete(@ApiParam(name = "ids",value = "ID集合") @RequestParam("ids") String ids) {
        log.debug("合同明细-删除数据接口请求参数：{}", ids);
        Assert.notNull(ids,"主键集合不能为空");
        boolean state = baseService.deleteByIds(Arrays.asList(ids.split(",")));
        String msg = state ? "删除成功" : "删除失败";
        log.debug("合同明细-删除数据接口响应结果：{}", msg);
        return new CommonResult<String>(state, msg);
    }

    /**
     * 通过主键查询单条数据
     * @param id 主键
     * @return 单条数据
     */
    @PostMapping("/detail/{id}")
    @ApiOperation(value = "S-根据ID查询详情",httpMethod = "POST")
    public CommonResult<Item> selectOne(@ApiParam(name = "id",value = "主键ID") @PathVariable String id) {
        log.debug("合同明细-根据ID查询详情 接口请求参数：{}", id);
        Assert.notNull(id,"主键不能为空");
        Item result = baseService.getById(id);
        boolean state = !BeanUtils.isEmpty(result);
        log.debug("合同明细-根据ID查询详情 接口响应结果：{}", result);
        return new CommonResult<Item>(state, state?"操作成功":"该ID没有对应数据", result);
    }

    /**
     * 分页查询所有数据
     * @param queryFilter 通用查询器
     * @return 分页数据
     */
    @PostMapping("/page")
    @ApiOperation(value = "S-分页查询所有数据",notes = "参数说明：在通用查询器中的query对象组传入查询参数，传入 PROJECT_ID_ 项目ID 以切换项目，所有参数都需要带上表别名——主表为bcp、副表为pm",httpMethod = "POST")
    public CommonResult<PageList<Item>> selectAll(@ApiParam(name = "queryFilter",value = "通用查询器")@RequestBody QueryFilter<Item> queryFilter) {
        log.debug("合同明细-分页查询所有数据接口请求参数：{}", queryFilter);
        PageList<Item> result = baseService.queryAllByPage(queryFilter);
        log.debug("合同明细-分页查询所有数据接口响应结果：{}", result);
        return new CommonResult<>(true,"操作成功",result);
    }
}

