export interface ConnectFailResult {
    /**
     * - 1：未知错误
     * - 2：网络连接已经存在
     * - 3：URL参数为空
     * - 4：无法识别的URL格式
     * - 5：URL必须以ws或者wss开头
     * - 6：连接服务器超时
     * - 7：服务器返回的https证书无效
     * - 8：服务端返回协议头无效
     * - 9：WebSocket请求没有指定Sec-WebSocket-Protocol请求头
     * - 10：网络连接没有打开，无法发送消息
     * - 11：消息发送失败
     * - 12：无法申请更多内存来读取网络数据
     */
    readonly error: 1 | 2 | 3 | 4 | 5 | 6 | 7 | 8 | 9 | 10 | 11 | 12;
}
export interface ConnectSocketOptions {
    url: string;
    data?: string;
    header?: Record<string, string>;
    method?: 'OPTIONS' | 'GET' | 'HEAD' | 'POST' | 'PUT' | 'DELETE' | 'TRACE' | 'CONNECT';
    complete?(res?: ConnectFailResult): void;
}
export interface SendSocketMessageOptions {
    data: string;
    isBuffer?: boolean;
    complete?(): void;
}
export interface HttpRequestParams {
    /**
     * 目标服务器url
     */
    url: string;
    /**
     * 设置请求的 HTTP 头
     */
    headers?: Record<string, string>;
    /**
     * 默认GET，目前支持GET/POST
     */
    method?: 'GET' | 'POST';
    /**
     * 请求参数。
     *
     * 传给服务器的数据最终会是 String 类型，如果 data 不是 String 类型，会被转换成 String 。转换规则如下：
     * - 若方法为GET，会将数据转换成 query string： encodeURIComponent(k)=encodeURIComponent(v)&encodeURIComponent(k)=encodeURIComponent(v)...
     * - 若方法为 POST 且 headers['content-type'] 为 application/json ，会对数据进行 JSON 序列化
     * - 若方法为 POST 且 headers['content-type'] 为 application/x-www-form-urlencoded ，会将数据转换成 query string： encodeURIComponent(k)=encodeURIComponent(v)&encodeURIComponent(k)=encodeURIComponent(v)...
     */
    data?: Record<string, any>;
    /**
     * 超时时间，单位ms，默认30000
     */
    timeout?: number;
    /**
     * 期望返回的数据格式，默认json，支持json，text，base64
     */
    dataType?: 'json' | 'text' | 'base64';
    complete?(res?: HttpRequestSuccessResult): void;
}
export interface HttpRequestSuccessResult {
    /**
     * 响应数据，格式取决于请求时的 dataType 参数
     */
    readonly data?: any;
    /**
     * 响应码
     *
     * - 11：无权跨域
     * - 12：网络出错
     * - 13：超时
     * - 14：解码失败
     * - 15：HTTP错误
     * - 16：请求已被停止/服务端限流
     */
    readonly status?: 11 | 12 | 13 | 14 | 19;
    /**
     * 响应头
     */
    readonly headers?: Readonly<Record<string, string>>;
}
export interface UploadFileSuccessResult {
    /**
     * 开发者服务器返回的数据
     */
    readonly data: string;
    /**
     * HTTP状态码
     */
    readonly statusCode: number;
    /**
     * 服务器返回的 header
     */
    readonly header: Readonly<Record<string, string>>;
}
export interface UploadFileOptions {
    /**
     * 开发者服务器 url
     */
    url: string;
    /**
     * 要上传文件资源的路径
     */
    filePath: string;
    /**
     * 文件对应的 key , 开发者在服务器端通过这个 key 可以获取到文件二进制内容
     */
    fileName: string;
    /**
     * 文件类型，image / video / audio
     */
    fileType?: 'image' | 'video' | 'audio';
    /**
     * HTTP 请求 Header , header 中不能设置 Referer
     */
    header?: Record<string, string>;
    /**
     * HTTP 请求中其他额外的 form data
     */
    formData?: Record<string, any>;
}
export default interface Network {
    connectSocket(options: ConnectSocketOptions): Promise<void>;
    onSocketOpen(callback: (res?: any) => void): void;
    offSocketOpen(callback: (res?: any) => void): void;
    onSocketError(callback: (res?: any) => void): void;
    offSocketError(callback: (res?: any) => void): void;
    sendSocketMessage(options: SendSocketMessageOptions): Promise<{
        success: boolean;
    }>;
    onSocketMessage(callback: (res?: {
        data: string;
        isBuffer: boolean;
    }) => void): Promise<void>;
    offSocketMessage(): void;
    closeSocket(options?: {
        complete?(): void;
    }): void;
    onSocketClose(callback: (res?: any) => void): void;
    offSocketClose(): void;
    httpRequest(options: HttpRequestParams): Promise<HttpRequestSuccessResult>;
    request(options: HttpRequestParams): Promise<HttpRequestSuccessResult>;
}
