package com.artfess.application.controller;

import com.artfess.application.model.MsgTemplate;
import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.controller.BaseController;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.query.QueryOP;
import com.artfess.base.util.StringUtil;
import com.artfess.sysConfig.persistence.manager.SysLogsSettingsManager;
import com.artfess.sysConfig.persistence.model.SysLogsSettings;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.web.bind.annotation.*;

import java.util.Map;

/**
 * <pre>
 * 描述：日志配置 控制器类
 * 构建组：artfess
 * 作者:baseli
 * 日期:2021年6月9日15:46:28
 * </pre>
 */
@RestController
@RequestMapping(value = "/logs/sysLogsSettings/v1")
@Api(tags = "日志配置")
@ApiGroup(group = {ApiGroupConsts.GROUP_APPLICATION})
public class SysLogsSettingsController extends BaseController<SysLogsSettingsManager, SysLogsSettings> {
    /**
     * 日志配置列表(分页条件查询)数据
     *
     * @param queryFilter
     * @return
     * @throws Exception PageJson
     * @throws
     */
    @PostMapping("/list")
    @ApiOperation(value = "日志配置数据列表", httpMethod = "POST", notes = "获取日志配置列表")
    public PageList<SysLogsSettings> list(@ApiParam(name = "queryFilter", value = "查询对象") @RequestBody QueryFilter<SysLogsSettings> queryFilter) throws Exception {
        return baseService.query(queryFilter);
    }

    /**
     * 日志配置明细页面
     *
     * @param id
     * @return
     * @throws Exception ModelAndView
     */
    @GetMapping(value = "/get/{id}")
    @ApiOperation(value = "日志配置数据详情", httpMethod = "GET", notes = "日志配置数据详情")
    public SysLogsSettings get(@ApiParam(name = "id", value = "业务对象主键", required = true) @PathVariable String id) throws Exception {
        return baseService.get(id);
    }

    /**
     * 新增日志配置
     *
     * @param sysLogsSettings
     * @return
     * @throws Exception
     * @throws
     */
    @PostMapping(value = "save")
    @ApiOperation(value = "新增,更新日志配置数据", httpMethod = "POST", notes = "新增,更新日志配置数据")
    public CommonResult<String> save(@ApiParam(name = "sysLogsSettings", value = "日志配置业务对象", required = true) @RequestBody SysLogsSettings sysLogsSettings) throws Exception {
        String msg = "添加日志配置成功";
        //验证登录日志是否存在
        //验证错误日志是否存在
        //验证模块 + 操作日志是否存在
        boolean isExist = false;
        if(ObjectUtils.isNotEmpty(sysLogsSettings)){
            QueryFilter queryFilter = QueryFilter.build().withDefaultPage();
            String id = sysLogsSettings.getId();
            Integer logType = sysLogsSettings.getLogType();
            String moduleType = sysLogsSettings.getModuleType();
            if(StringUtils.isNotBlank(id)){
                queryFilter.addFilter("id_",  id, QueryOP.NOT_IN);
            }
            if(null != logType){
                queryFilter.addFilter("LOG_TYPE_",  logType, QueryOP.EQUAL);
            }
            if(StringUtils.isNotBlank(moduleType)){
                queryFilter.addFilter("module_type_",  moduleType, QueryOP.EQUAL);
            }
            PageList<MsgTemplate> query = baseService.query(queryFilter);
            isExist = query.getRows().size() > 0;
        }
        if (isExist) {
            return new CommonResult(false, "配置信息已经存在,添加失败!", null);
        } else {
            if (StringUtil.isEmpty(sysLogsSettings.getId())) {
                baseService.create(sysLogsSettings);
                msg = "添加日志配置成功";
            } else {
                baseService.update(sysLogsSettings);
                msg = "更新日志配置成功";
            }
        }
        return new CommonResult<String>(msg);
    }

    /**
     * 删除日志配置记录
     *
     * @param id
     * @return
     * @throws Exception
     * @throws
     */
    @DeleteMapping(value = "remove/{id}")
    @ApiOperation(value = "删除日志配置记录", httpMethod = "DELETE", notes = "删除日志配置记录")
    public CommonResult<String> remove(@ApiParam(name = "id", value = "业务主键", required = true) @PathVariable String id) throws Exception {
        baseService.remove(id);
        return new CommonResult<String>(true, "删除成功");
    }

    /**
     * 批量删除日志配置记录
     *
     * @param ids
     * @return
     * @throws Exception
     * @throws
     */
    @DeleteMapping(value = "/removes")
    @ApiOperation(value = "批量删除日志配置记录", httpMethod = "DELETE", notes = "批量删除日志配置记录")
    public CommonResult<String> removes(@ApiParam(name = "ids", value = "业务主键数组,多个业务主键之间用逗号分隔", required = true) @RequestParam String... ids) throws Exception {
        baseService.removeByIds(ids);
        return new CommonResult<String>(true, "批量删除成功");
    }

    /**
     * 获取日志配置状态的Map
     *
     * @return
     * @throws Exception
     */
    @GetMapping("/getSysLogsSettingStatusMap")
    @ApiOperation(value = "获取日志配置状态的Map集合", httpMethod = "GET", notes = "获取日志配置状态的Map集合")
    public Map<String, String> getSysLogsSettingStatusMap() throws Exception {
        return baseService.getSysLogsSettingStatusMap();
    }
}
