package com.artfess.examine.controller;


import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.controller.BaseController;
import com.artfess.base.enums.ResponseErrorEnums;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.valid.AddGroup;
import com.artfess.base.valid.UpdateGroup;
import com.artfess.examine.manager.ExamFileManager;
import com.artfess.examine.manager.ExamMaterialInfoManager;
import com.artfess.examine.model.ExamFile;
import com.artfess.examine.model.ExamMaterialInfo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

/**
 * 资料信息表 前端控制器
 *
 * @company 阿特菲斯信息技术有限公司
 * @author min.wu
 * @since 2022-10-19
 */
@Slf4j
@RestController
@Api(tags = "基础数据-资料信息")
@RequestMapping("/exam/materialType/info/")
@ApiGroup(group = {ApiGroupConsts.GROUP_BIZ})
public class ExamMaterialInfoController extends BaseController<ExamMaterialInfoManager, ExamMaterialInfo> {

    @Autowired
    private ExamFileManager fileManager;


    @Override
    @PostMapping("/")
    @ApiOperation("添加实体的接口")
    public CommonResult<String> create(@ApiParam(name="model", value="实体信息") @RequestBody @Validated({AddGroup.class}) ExamMaterialInfo t) {
        String id = baseService.createInfo(t);
        if(!StringUtils.isNotBlank(id)) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, null);
        }
        return new CommonResult<>();
    }

    @Override
    @PutMapping("/")
    @ApiOperation("更新实体")
    public CommonResult<String> updateById(@ApiParam(name="model", value="实体信息") @RequestBody @Validated({UpdateGroup.class})  ExamMaterialInfo t) {
        String id = baseService.updateInfo(t);
        if(!StringUtils.isNotBlank(id)) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, "更新实体失败");
        }
        return new CommonResult<>();
    }

    @Override
    @DeleteMapping("/{id}")
    @ApiOperation("根据文件id删除资料附件信息")
    public CommonResult<String> deleteById(@ApiParam(name="id", value="实体id") @PathVariable String id) {
        ExamFile examFile = fileManager.get(id);
        if(null == examFile) {
            return new CommonResult<>("删除成功");
        }
        boolean result = fileManager.removeById(id);
        if(!result) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, "删除附件失败");
        }
        return new CommonResult<>();
    }

    @Override
    @PostMapping(value="/query", produces={"application/json; charset=utf-8" })
    @ApiOperation("分页查询结果")
    public PageList<ExamMaterialInfo> query(@ApiParam(name="queryFilter", value="分页查询信息") @RequestBody QueryFilter<ExamMaterialInfo> queryFilter) {
        return baseService.findByPage(queryFilter);
    }

    @Override
    @GetMapping("/{id}")
    @ApiOperation("根据id查询实体")
    public ExamMaterialInfo getById(@ApiParam(name="id", value="实体id") @PathVariable String id) {
        return baseService.findById(id);
    }

    @GetMapping("/viewFile/{id}")
    @ApiOperation("查看文件")
    public ExamMaterialInfo viewFile(@ApiParam(name="id", value="实体id") @PathVariable String id) {
        return baseService.viewFile(id);
    }

}
