package com.artfess.data.controller;


import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.controller.BaseController;
import com.artfess.base.model.CommonResult;
import com.artfess.base.valid.AddGroup;
import com.artfess.data.dto.EditPersonalProfileDto;
import com.artfess.data.dto.PersonalProfileExportDTO;
import com.artfess.data.manager.BizUserBasicManager;
import com.artfess.data.model.BizUserBasic;
import com.artfess.data.vo.PersonalProfileVo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.util.Arrays;

/**
 * 个人信息基本情况表 前端控制器
 *
 * @company 阿特菲斯信息技术有限公司
 * @author 超级管理员
 * @since 2024-12-27
 */
@Slf4j
@RestController
@Api(tags = "个人中心-个人档案")
@RequestMapping("/biz/personalProfile")
@ApiGroup(group = {ApiGroupConsts.GROUP_BIZ})
public class BizUserBasicController extends BaseController<BizUserBasicManager, BizUserBasic> {


    private final BizUserBasicManager bizUserBasicManager;

    public BizUserBasicController(BizUserBasicManager bizUserBasicManager) {
        super();
        this.bizUserBasicManager = bizUserBasicManager;
    }

    /**
     * 获取个人档案
     *
     * @return 个人档案
     */
    @ApiOperation("获取个人档案(自己账号)")
    @GetMapping
    public CommonResult<PersonalProfileVo> getPersonalProfile() {
        PersonalProfileVo personalProfileVo = bizUserBasicManager.getPersonalProfile();
        return CommonResult.success(personalProfileVo, "获取个人档案成功");
    }

    /**
     * 获取个人档案
     *
     * @return 个人档案
     */
    @ApiOperation("根据年份和用户ID获取个人档案)")
    @GetMapping(value = "/{year}/{userId}", produces = {"application/json; charset=utf-8"})
    public CommonResult<PersonalProfileVo> getPersonalProfile(@PathVariable("year") String year, @PathVariable("userId") String userId) {
        PersonalProfileVo personalProfileVo = bizUserBasicManager.getPersonalProfile(year, userId);
        return CommonResult.success(personalProfileVo, "获取个人档案成功");
    }

    /**
     * 编辑个人档案
     */
    @ApiOperation("编辑个人档案")
    @PutMapping
    public CommonResult<String> create(@ApiParam(name = "model", value = "实体信息") @RequestBody @Validated({AddGroup.class}) EditPersonalProfileDto editPersonalProfileDto) {
        bizUserBasicManager.editPersonalProfile(editPersonalProfileDto);
        return new CommonResult<>();
    }

    /**
     * 上传个人图片，返回URL
     */
    @ApiOperation("上传个人图片，返回URL")
    @PostMapping(value = "/uploadImage")
    public CommonResult<String> uploadImage(@ApiParam(name = "file", value = "文件") @RequestParam("file") MultipartFile file) {
        String url = bizUserBasicManager.uploadImage(file);
        return CommonResult.success(url, "上传个人图片成功");
    }

    /**
     * 获取文件流
     */
    @ApiOperation("获取文件流")
    @GetMapping(value = "/download/{fileName}")
    public void download(@PathVariable("fileName") String fileName, HttpServletResponse response) {
        bizUserBasicManager.download(fileName, response);
    }


    @PostMapping(value = "/export")
    @ApiOperation("导出勾选用户的个人档案")
    public void export(HttpServletResponse response, @ApiParam(name = "PersonalProfileExportDTO", value = "档案导出DTO") @RequestBody PersonalProfileExportDTO dto) {
        baseService.export(dto.getIds(),dto.getYear(), response);
    }
}
