package com.artfess.cqlt.controller;


import cn.hutool.core.bean.BeanUtil;
import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.controller.BaseController;
import com.artfess.base.exception.RequiredException;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.Direction;
import com.artfess.base.query.FieldSort;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.util.DateUtils;
import com.artfess.base.valid.AddGroup;
import com.artfess.base.valid.UpdateGroup;
import com.artfess.cqlt.manager.QfOperationKpiDManager;
import com.artfess.cqlt.manager.QfOperationKpiMManager;
import com.artfess.cqlt.model.QfOperationCompanyCipM;
import com.artfess.cqlt.model.QfOperationKpiD;
import com.artfess.cqlt.model.QfOperationKpiM;
import com.artfess.cqlt.vo.FaReportRespVo;
import com.artfess.cqlt.vo.FaTargetRespVo;
import com.artfess.cqlt.vo.KpiDetailVo;
import com.artfess.cqlt.vo.ReportReqVo;
import com.artfess.i18n.util.I18nUtil;
import com.artfess.poi.util.ExcelUtils;
import com.artfess.poi.util.FileDownloadUtil;
import com.artfess.uc.api.impl.util.ContextUtil;
import com.google.common.collect.Lists;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.core.io.ClassPathResource;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.time.LocalDate;
import java.util.Arrays;
import java.util.List;

/**
 * 运营--KPI数据填报主表 前端控制器
 *
 * @author min.wu
 * @company 阿特菲斯信息技术有限公司
 * @since 2023-02-14
 */
@Slf4j
@RestController
@Api(tags = "运营--KPI数据填报")
@RequestMapping("/qf/operation/kpi/")
@ApiGroup(group = {ApiGroupConsts.GROUP_BIZ})
public class QfOperationKpiMController extends BaseController<QfOperationKpiMManager, QfOperationKpiM> {

    @Autowired
    private QfOperationKpiDManager operationKpiDManager;

    @Override
    @PostMapping(value="/query", produces={"application/json; charset=utf-8" })
    @ApiOperation("分页查询结果")
    public PageList<QfOperationKpiM> query(@ApiParam(name="queryFilter", value="分页查询信息") @RequestBody QueryFilter<QfOperationKpiM> queryFilter) {
        List<FieldSort> sorter = queryFilter.getSorter();
        sorter.add(new FieldSort("fill_month_", Direction.ASC));
        queryFilter.setSorter(sorter);
        return baseService.query(queryFilter);
    }

    @Override
    @PostMapping("/")
    @ApiOperation("添加实体的接口")
    public CommonResult<String> create(@ApiParam(name = "model", value = "实体信息") @RequestBody @Validated({AddGroup.class}) QfOperationKpiM t) {
        t.setFillUser(ContextUtil.getCurrentUserId());
        t.setFillDate(LocalDate.now());
        t.setFillQuarter(DateUtils.getQuarter(t.getFillMonth()));
        boolean result = baseService.insertInfo(t);
        if (!result) {
            return new CommonResult<String>(false, I18nUtil.getMessage("option.fail", LocaleContextHolder.getLocale()), null);
        }

        return new CommonResult<String>(true, I18nUtil.getMessage("option.success", LocaleContextHolder.getLocale()), t.getId());
    }

    @Override
    @PutMapping("/")
    @ApiOperation("更新实体")
    public CommonResult<String> updateById(@ApiParam(name = "model", value = "实体信息") @RequestBody @Validated({UpdateGroup.class}) QfOperationKpiM t) {
        boolean result = baseService.updateInfo(t);
        if (!result) {
            return new CommonResult<String>(false, I18nUtil.getMessage("option.fail", LocaleContextHolder.getLocale()), null);
        }
        return new CommonResult<String>(true, I18nUtil.getMessage("option.success", LocaleContextHolder.getLocale()), null);
    }

    @PutMapping("/updateStatus")
    @ApiOperation("生效")
    public CommonResult<String> updateStatus(@ApiParam(name = "model", value = "实体信息") @RequestBody QfOperationKpiM t) {
        boolean result = baseService.updateStatus(t);
        if (!result) {
            return new CommonResult<String>(false, I18nUtil.getMessage("option.fail", LocaleContextHolder.getLocale()), null);
        }
        return new CommonResult<String>(true, I18nUtil.getMessage("option.success", LocaleContextHolder.getLocale()), null);
    }


    @Override
    @DeleteMapping("/{id}")
    @ApiOperation("根据id删除")
    public CommonResult<String> deleteById(@ApiParam(name = "id", value = "实体id") @PathVariable String id) {
        boolean result = baseService.removeById(id);
        if (!result) {
            return new CommonResult<String>(false, I18nUtil.getMessage("delete.fail", LocaleContextHolder.getLocale()), null);
        }
        return new CommonResult<String>(true, I18nUtil.getMessage("option.success", LocaleContextHolder.getLocale()), null);
    }

    @Override
    @DeleteMapping("/")
    @ApiOperation("根据id集合批量删除")
    public CommonResult<String> deleteByIds(@ApiParam(name = "ids", value = "实体集合") @RequestParam String... ids) {
        boolean result = baseService.removeByIds(Arrays.asList(ids));
        if (!result) {
            return new CommonResult<String>(false, I18nUtil.getMessage("delete.fail", LocaleContextHolder.getLocale()), null);
        }
        return new CommonResult<String>(true, I18nUtil.getMessage("option.success", LocaleContextHolder.getLocale()), null);
    }

    /**
     * @param file
     * @param mainId
     * @return
     */
    @ApiOperation(value = "KPI数据导入")
    @PostMapping("/importExcel")
    public CommonResult<String> importExcel(@RequestParam("file") MultipartFile file, @RequestParam(value = "mainId", required = false) String mainId) throws Exception {
        try {
            ExcelUtils<QfOperationKpiD> excelUtil = new ExcelUtils<>(QfOperationKpiD.class);
            List<QfOperationKpiD> list = excelUtil.importExcel(null, file.getInputStream());
            boolean result = baseService.importExcel(list, mainId);
            if (!result) {
                throw new IllegalArgumentException(I18nUtil.getMessage("import.fail", LocaleContextHolder.getLocale()));
            }
            return new CommonResult<>();
        } catch (Exception e) {
            throw new IllegalArgumentException(I18nUtil.getMessage("import.fail", LocaleContextHolder.getLocale()) + "," + e.getMessage());
        }
    }

    /**
     * 下载导入模板
     *
     * @param response
     * @return
     */
    @ApiOperation(value = "下载导入模板")
    @GetMapping("/downModel")
    public void downTemplate(HttpServletResponse response) {
        try {
            ClassPathResource classPathResource = new ClassPathResource("model/kpiDetail.xlsx");
            FileDownloadUtil.fileDownload(response, classPathResource.getInputStream(), "KPI数据导入模板.xlsx");
        } catch (Exception e) {
            response.setCharacterEncoding("utf-8");
            throw new RequiredException(I18nUtil.getMessage("resources.noexist", LocaleContextHolder.getLocale()));
        }
    }

    @ApiOperation(value = "KPI数据导出")
    @PostMapping("/export")
    public void export(HttpServletRequest request, HttpServletResponse response, @ApiParam(name = "queryFilter", value = "分页查询信息") @RequestBody QueryFilter<QfOperationKpiD> queryFilter) throws Exception {
        PageList<QfOperationKpiD> page = operationKpiDManager.query(queryFilter);
        List<KpiDetailVo> list = Lists.newArrayList();
        page.getRows().forEach(detail -> {

            KpiDetailVo kpiDetailVo = new KpiDetailVo();
            BeanUtil.copyProperties(detail, kpiDetailVo);

            // 百分比格式，不改VO，导出工具类已经根据%处理，这一行保留，不做数据放大
            if (null != detail.getTrExfiAct()) {
                kpiDetailVo.setTrExfiActStr(detail.getTrExfiAct() + "%");
            }
            if (null != detail.getOee24ExEpdmAct()) {
                kpiDetailVo.setOee24ExEpdmActStr(detail.getOee24ExEpdmAct() + "%");
            }
            if (null != detail.getQrExEpdmAct()) {
                kpiDetailVo.setQrExEpdmActStr(detail.getQrExEpdmAct() + "%");
            }
            list.add(kpiDetailVo);
        });
        ExcelUtils<KpiDetailVo> util = new ExcelUtils<KpiDetailVo>(KpiDetailVo.class);
        util.exportExcel(response, request, list, "KPI数据信息");
    }

    @PostMapping("/oneLevelData")
    @ApiOperation(value = "运行板块指标一级界面统计接口", response = FaTargetRespVo.class)
    public CommonResult<String> oneLevelData(@ApiParam(name = "model", value = "请求参数") @RequestBody ReportReqVo t) {
        List<FaTargetRespVo> data = baseService.data(t);
        return CommonResult.success(data, null);
    }

    @PostMapping("/fromUnderData")
    @ApiOperation("运行板块指标包含当前指标的年度、季度、月度报表)")
    public CommonResult<String> fromUnderData(@ApiParam(name = "model", value = "请求参数") @RequestBody ReportReqVo t) {
        List<FaTargetRespVo> data = baseService.fromUnderData(t);
        return CommonResult.success(data, null);
    }

    @PostMapping("/enterpriseData")
    @ApiOperation("根据指标id获取对应运行板块所有企业统计信息（所有企业）")
    public CommonResult<String> enterpriseData(@ApiParam(name = "model", value = "请求参数") @RequestBody ReportReqVo t) {
        List<FaReportRespVo> data = baseService.enterpriseData(t);
        return CommonResult.success(data, null);
    }


}
