package com.artfess.cqlt.controller;


import cn.hutool.core.bean.BeanUtil;
import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.controller.BaseController;
import com.artfess.base.exception.RequiredException;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.valid.AddGroup;
import com.artfess.base.valid.UpdateGroup;
import com.artfess.cqlt.manager.QfInvestExpenditureDManager;
import com.artfess.cqlt.manager.QfInvestExpenditureMManager;
import com.artfess.cqlt.model.QfInvestExpenditureD;
import com.artfess.cqlt.model.QfInvestExpenditureM;
import com.artfess.cqlt.vo.InvestExpendVo;
import com.artfess.i18n.util.I18nUtil;
import com.artfess.poi.util.ExcelUtils;
import com.artfess.poi.util.FileDownloadUtil;
import com.artfess.uc.api.impl.util.ContextUtil;
import com.google.common.collect.Lists;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.core.io.ClassPathResource;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.time.LocalDate;
import java.util.Arrays;
import java.util.List;

/**
 * 投资--萨固密集团资本性支出年度预算与完成情况 前端控制器
 *
 * @company 阿特菲斯信息技术有限公司
 * @author 黎沐华
 * @since 2023-02-21
 */
@Slf4j
@RestController
@Api(tags = "投资--萨固密集团资本性支出年度预算与完成情况")
@RequestMapping("/qf/invest/expenditure/v1/")
@ApiGroup(group = {ApiGroupConsts.GROUP_BIZ})
public class QfInvestExpenditureMController extends BaseController<QfInvestExpenditureMManager, QfInvestExpenditureM> {

    @Autowired
    private QfInvestExpenditureDManager investExpenditureDManager;

    @Override
    @PostMapping("/")
    @ApiOperation("添加实体的接口")
    public CommonResult<String> create(@ApiParam(name = "model", value = "实体信息") @RequestBody @Validated({AddGroup.class}) QfInvestExpenditureM t) {
        t.setFillUser(ContextUtil.getCurrentUserId());
        t.setFillDate(LocalDate.now());
        boolean result = baseService.insertInfo(t);
        if (!result) {
            return new CommonResult<String>(false, I18nUtil.getMessage("option.fail", LocaleContextHolder.getLocale()), null);
        }

        return new CommonResult<String>(true, I18nUtil.getMessage("option.success", LocaleContextHolder.getLocale()), t.getId());
    }

    @Override
    @PutMapping("/")
    @ApiOperation("更新实体")
    public CommonResult<String> updateById(@ApiParam(name = "model", value = "实体信息") @RequestBody @Validated({UpdateGroup.class}) QfInvestExpenditureM t) {
        boolean result = baseService.updateInfo(t);
        if (!result) {
            return new CommonResult<String>(false, I18nUtil.getMessage("option.fail", LocaleContextHolder.getLocale()), null);
        }
        return new CommonResult<String>(true, I18nUtil.getMessage("option.success", LocaleContextHolder.getLocale()), null);
    }

    @PutMapping("/updateStatus")
    @ApiOperation("生效")
    public CommonResult<String> updateStatus(@ApiParam(name = "model", value = "实体信息") @RequestBody QfInvestExpenditureM t) {
        boolean result = baseService.updateStatus(t);
        if (!result) {
            return new CommonResult<String>(false, I18nUtil.getMessage("option.fail", LocaleContextHolder.getLocale()), null);
        }
        return new CommonResult<String>(true, I18nUtil.getMessage("option.success", LocaleContextHolder.getLocale()), null);
    }


    @Override
    @DeleteMapping("/{id}")
    @ApiOperation("根据id删除")
    public CommonResult<String> deleteById(@ApiParam(name = "id", value = "实体id") @PathVariable String id) {
        boolean result = baseService.removeById(id);
        if (!result) {
            return new CommonResult<String>(false, I18nUtil.getMessage("delete.fail", LocaleContextHolder.getLocale()), null);
        }
        return new CommonResult<String>(true, I18nUtil.getMessage("option.success", LocaleContextHolder.getLocale()), null);
    }

    @Override
    @DeleteMapping("/")
    @ApiOperation("根据id集合批量删除")
    public CommonResult<String> deleteByIds(@ApiParam(name = "ids", value = "实体集合") @RequestParam String... ids) {
        boolean result = baseService.removeByIds(Arrays.asList(ids));
        if (!result) {
            return new CommonResult<String>(false, I18nUtil.getMessage("delete.fail", LocaleContextHolder.getLocale()), null);
        }
        return new CommonResult<String>(true, I18nUtil.getMessage("option.success", LocaleContextHolder.getLocale()), null);
    }

    /**
     * @param file
     * @param mainId
     * @return
     */
    @ApiOperation(value = "年度预算与完成情况数据导入")
    @PostMapping("/importExcel")
    public CommonResult<String> importExcel(@RequestParam("file") MultipartFile file, @RequestParam(value = "mainId", required = false) String mainId) {
        try {
            ExcelUtils<QfInvestExpenditureD> excelUtil = new ExcelUtils<>(QfInvestExpenditureD.class);
            List<QfInvestExpenditureD> list = excelUtil.importExcel(null, file.getInputStream());
            boolean result = baseService.importExcel(list, mainId);
            if (!result) {
                throw new IllegalArgumentException(I18nUtil.getMessage("import.fail", LocaleContextHolder.getLocale()));
            }
            return new CommonResult<>();
        } catch (Exception e) {
            throw new IllegalArgumentException(I18nUtil.getMessage("import.fail", LocaleContextHolder.getLocale()) + "," + e.getMessage());
        }
    }

    /**
     * 下载导入模板
     *
     * @param response
     * @return
     */
    @ApiOperation(value = "下载导入模板")
    @GetMapping("/downModel")
    public void downTemplate(HttpServletResponse response) {
        try {
            ClassPathResource classPathResource = new ClassPathResource("model/expenditureDetail.xlsx");
            FileDownloadUtil.fileDownload(response, classPathResource.getInputStream(), "年度预算与完成情况数据导入模板.xlsx");
        } catch (Exception e) {
            response.setCharacterEncoding("utf-8");
            throw new RequiredException(I18nUtil.getMessage("resources.noexist", LocaleContextHolder.getLocale()));
        }
    }

    @ApiOperation(value = "年度预算与完成情况数据导出")
    @PostMapping("/export")
    public void export(HttpServletRequest request, HttpServletResponse response, @ApiParam(name = "queryFilter", value = "分页查询信息") @RequestBody QueryFilter<QfInvestExpenditureD> queryFilter) throws Exception {
        PageList<QfInvestExpenditureD> page = investExpenditureDManager.query(queryFilter);
        List<InvestExpendVo> list = Lists.newArrayList();
        page.getRows().forEach(detail -> {
            InvestExpendVo newInvestExpendVo = new InvestExpendVo();
            BeanUtil.copyProperties(detail, newInvestExpendVo);
            // 百分比格式，不改VO，导出工具类已经根据%处理，这一行保留，不做数据放大
            newInvestExpendVo.setCompletionRateStr(detail.getCompletionRate() + "%");
            list.add(newInvestExpendVo);
        });
        ExcelUtils<QfInvestExpenditureD> util = new ExcelUtils<QfInvestExpenditureD>(QfInvestExpenditureD.class);
        util.exportExcel(response, request, page.getRows(), "年度预算与完成情况数据信息");
    }

}
