package com.artfess.cqlt.controller;


import com.alibaba.fastjson.JSONObject;
import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.controller.BaseController;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.valid.AddGroup;
import com.artfess.base.valid.UpdateGroup;
import com.artfess.cqlt.manager.QfFinanceSasacCompanyDManager;
import com.artfess.cqlt.manager.QfFinanceSasacCompanyMManager;
import com.artfess.cqlt.model.QfFinanceSasacCompanyD;
import com.artfess.cqlt.model.QfFinanceSasacCompanyM;
import com.artfess.i18n.util.I18nUtil;
import com.artfess.uc.api.impl.util.ContextUtil;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.time.LocalDate;
import java.util.Arrays;
import java.util.List;

/**
 * 财务--国资委子企业季度填报主表 前端控制器
 *
 * @author min.wu
 * @company 阿特菲斯信息技术有限公司
 * @since 2023-03-02
 */
@Slf4j
@RestController
@Api(tags = "财务--国资委子企业季度填报主表")
@RequestMapping("/qf/financeSasac/company/")
@ApiGroup(group = {ApiGroupConsts.GROUP_BIZ})
public class QfFinanceSasacCompanyMController extends BaseController<QfFinanceSasacCompanyMManager, QfFinanceSasacCompanyM> {

    @Autowired
    private QfFinanceSasacCompanyDManager financeSasacCompanyDManager;

    @Override
    @PostMapping("/")
    @ApiOperation("添加实体的接口")
    public CommonResult<String> create(@ApiParam(name = "model", value = "实体信息") @RequestBody @Validated({AddGroup.class}) QfFinanceSasacCompanyM t) {
        t.setFillUser(ContextUtil.getCurrentUserId());
        t.setFillDate(LocalDate.now());
        boolean result = baseService.insertInfo(t);
        if (!result) {
            return new CommonResult<String>(false, I18nUtil.getMessage("option.fail", LocaleContextHolder.getLocale()), null);
        }

        return new CommonResult<String>(true, I18nUtil.getMessage("option.success", LocaleContextHolder.getLocale()), t.getId());
    }

    @Override
    @PutMapping("/")
    @ApiOperation("更新实体")
    public CommonResult<String> updateById(@ApiParam(name = "model", value = "实体信息") @RequestBody @Validated({UpdateGroup.class}) QfFinanceSasacCompanyM t) {
        boolean result = baseService.updateInfo(t);
        if (!result) {
            return new CommonResult<String>(false, I18nUtil.getMessage("option.fail", LocaleContextHolder.getLocale()), null);
        }
        return new CommonResult<String>(true, I18nUtil.getMessage("option.success", LocaleContextHolder.getLocale()), null);
    }

    @PutMapping("/updateStatus")
    @ApiOperation("生效")
    public CommonResult<String> updateStatus(@ApiParam(name = "model", value = "实体信息") @RequestBody QfFinanceSasacCompanyM t) {
        boolean result = baseService.updateStatus(t);
        if (!result) {
            return new CommonResult<String>(false, I18nUtil.getMessage("option.fail", LocaleContextHolder.getLocale()), null);
        }
        return new CommonResult<String>(true, I18nUtil.getMessage("option.success", LocaleContextHolder.getLocale()), null);
    }


    @Override
    @DeleteMapping("/{id}")
    @ApiOperation("根据id删除")
    public CommonResult<String> deleteById(@ApiParam(name = "id", value = "实体id") @PathVariable String id) {
        boolean result = baseService.removeById(id);
        if (!result) {
            return new CommonResult<String>(false, I18nUtil.getMessage("delete.fail", LocaleContextHolder.getLocale()), null);
        }
        return new CommonResult<String>(true, I18nUtil.getMessage("option.success", LocaleContextHolder.getLocale()), null);
    }

    @Override
    @DeleteMapping("/")
    @ApiOperation("根据id集合批量删除")
    public CommonResult<String> deleteByIds(@ApiParam(name = "ids", value = "实体集合") @RequestParam String... ids) {
        boolean result = baseService.removeByIds(Arrays.asList(ids));
        if (!result) {
            return new CommonResult<String>(false, I18nUtil.getMessage("delete.fail", LocaleContextHolder.getLocale()), null);
        }
        return new CommonResult<String>(true, I18nUtil.getMessage("option.success", LocaleContextHolder.getLocale()), null);
    }

    /**
     * @param mainId
     * @return
     */
    @ApiOperation(value = "上报国资委报表转换")
    @GetMapping("/conversion")
    public CommonResult<String> conversion(@RequestParam(value = "mainId", required = false) String mainId) {
        try {
            boolean result = baseService.conversion(mainId);
            if (!result) {
                throw new IllegalArgumentException(I18nUtil.getMessage("option.fail", LocaleContextHolder.getLocale()));
            }
            return new CommonResult<>();
        } catch (Exception e) {
            e.printStackTrace();
            throw new IllegalArgumentException(I18nUtil.getMessage("option.fail", LocaleContextHolder.getLocale()) + "," + e.getMessage());
        }
    }


    @ApiOperation(value = "报表导出")
    @PostMapping("/export")
    public void export(HttpServletRequest request, HttpServletResponse response, @ApiParam(name = "queryFilter", value = "分页查询信息") @RequestBody QueryFilter<QfFinanceSasacCompanyD> queryFilter) throws Exception {
        PageList<QfFinanceSasacCompanyD> page = financeSasacCompanyDManager.query(queryFilter);
        financeSasacCompanyDManager.export(request, response, page.getRows(), "境外子企业基本情况");
    }

    @PostMapping(value = "/detailQuery", produces = {"application/json; charset=utf-8"})
    @ApiOperation("获取详情数据")
    public CommonResult detailQuery(@ApiParam(name = "queryFilter", value = "分页查询信息") @RequestBody QueryFilter<QfFinanceSasacCompanyD> queryFilter) {
        PageList<QfFinanceSasacCompanyD> pageList = financeSasacCompanyDManager.query(queryFilter);
        List<JSONObject> list = baseService.detailQuery(pageList.getRows());
        return CommonResult.success(list, null);
    }

}
