package com.artfess.manage.material.controller;


import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.controller.BaseController;
import com.artfess.base.enums.ResponseErrorEnums;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.query.QueryOP;
import com.artfess.base.valid.AddGroup;
import com.artfess.base.valid.UpdateGroup;
import com.artfess.manage.material.manager.CmgtMaterialPurchasingDetailManager;
import com.artfess.manage.material.manager.CmgtMaterialPurchasingManager;
import com.artfess.manage.material.manager.CmgtMaterialFlowLogManager;
import com.artfess.manage.material.model.CmgtMaterialPurchasing;
import com.artfess.manage.material.model.CmgtMaterialPurchasingDetail;
import com.artfess.manage.material.vo.RecipientsVo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import net.bytebuddy.asm.Advice;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.Assert;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import java.time.LocalDate;
import java.util.Arrays;
import java.util.List;

/**
 * 物资采购申请 前端控制器
 *
 * @author wujl
 * @company 阿特菲斯信息技术有限公司
 * @since 2022-07-23
 */
@Slf4j
@RestController
@Api(tags = "物资管理-采购领用申请单")
@RequestMapping("/manager/material/purchasing/")
@ApiGroup(group = {ApiGroupConsts.MANAGER_BIZ})
public class CmgtMaterialPurchasingController extends BaseController<CmgtMaterialPurchasingManager, CmgtMaterialPurchasing> {

    @Autowired
    private CmgtMaterialPurchasingDetailManager materialPurchasingDetailManager;

    @Autowired
    private CmgtMaterialPurchasingManager cmgtMaterialPurchasingManager;

    @Autowired
    private CmgtMaterialFlowLogManager flowLogManager;

    @Override
    @PostMapping("/add")
    @ApiOperation("添加物资领用或者采购申请的接口")
    public CommonResult<String> create(@ApiParam(name = "model", value = "实体信息") @RequestBody @Validated({AddGroup.class}) CmgtMaterialPurchasing t) {
        cmgtMaterialPurchasingManager.createInfo(t);
        return new CommonResult<>();
    }

    @Override
    @PutMapping("/update")
    @ApiOperation("更新物资领用或者采购申请")
    public CommonResult<String> updateById(@ApiParam(name = "model", value = "实体信息") @RequestBody @Validated({UpdateGroup.class}) CmgtMaterialPurchasing t) {
        cmgtMaterialPurchasingManager.updateInfo(t);
        return new CommonResult<>();
    }


    @PostMapping("/getDetailList")
    @ApiOperation("获取申请单下所有物资明细")
    public CommonResult<String> getDetailList(@ApiParam(name = "pid", value = "申请单id") @RequestBody String pid) {
        List<CmgtMaterialPurchasingDetail> list = materialPurchasingDetailManager.findAllByPurchasingId(pid);
        return CommonResult.success(list, null);
    }


    @Override
    @DeleteMapping("/delete")
    @ApiOperation("根据id集合批量删除")
    public CommonResult<String> deleteByIds(@ApiParam(name = "ids", value = "实体集合") @RequestParam String... ids) {
        boolean result = true;
        //删除明细
        for (String id : ids) {
            materialPurchasingDetailManager.deleteByMaterialInfoId(id);
            result = cmgtMaterialPurchasingManager.deleteById(id);
        }
        if (!result) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, "删除实体失败");
        }
        return new CommonResult<>();
    }


    @DeleteMapping("/findByIds")
    @ApiOperation("根据选中的采购申请单ids获取合并后的物资信息")
    public CommonResult<String> findByIds(@ApiParam(name = "ids", value = "实体集合") @RequestParam String... ids) {
        Assert.notEmpty(ids, "请选择要合并的申请单");
        List<CmgtMaterialPurchasingDetail> list = this.baseService.detailList(Arrays.asList(ids));
        return CommonResult.success(list, null);
    }

    @PostMapping("/updateStatus")
    @ApiOperation("申请单确认")
    public CommonResult<String> updateStatus(@ApiParam(name = "model", value = "实体信息") @RequestBody CmgtMaterialPurchasing t) {
        boolean result = baseService.updateStatus(t);
        if (!result) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, "确认失败");
        }
        return new CommonResult<>();
    }

    @Override
    @PostMapping(value = "/query", produces = {"application/json; charset=utf-8"})
    @ApiOperation("获取物资采购申请单列表")
    public PageList<CmgtMaterialPurchasing> query(@ApiParam(name = "queryFilter", value = "分页查询信息") @RequestBody QueryFilter<CmgtMaterialPurchasing> queryFilter) {
        queryFilter.addFilter("applyType", "1", QueryOP.EQUAL);
//        queryFilter.addFilter("APPLICATION_DATE_" , LocalDate.now(), QueryOP.EQUAL);
        return baseService.query(queryFilter);
    }

    @PostMapping(value = "/count", produces = {"application/json; charset=utf-8"})
    @ApiOperation("获取当月物资采购申请单数量")
    public CommonResult count(@ApiParam(name = "queryFilter", value = "分页查询信息") @RequestBody QueryFilter<CmgtMaterialPurchasing> queryFilter) {
        queryFilter.addFilter("applyType", "1", QueryOP.EQUAL);
        queryFilter.addFilter("APPLICATION_DATE_" , LocalDate.now(), QueryOP.EQUAL);
        queryFilter.getPageBean().setPageSize(-1);
        PageList<CmgtMaterialPurchasing> query = baseService.query(queryFilter);
        return CommonResult.success(query.getRows().size(), null);
    }

    @PostMapping(value = "/recipientsQuery", produces = {"application/json; charset=utf-8"})
    @ApiOperation("获取物资领用申请单列表")
    public PageList<CmgtMaterialPurchasing> recipientsQuery(@ApiParam(name = "queryFilter", value = "分页查询信息") @RequestBody QueryFilter<CmgtMaterialPurchasing> queryFilter) {
        queryFilter.addFilter("applyType", "2", QueryOP.EQUAL);
        return baseService.query(queryFilter);
    }

    @PostMapping(value = "/recipientsPage", produces = {"application/json; charset=utf-8"})
    @ApiOperation("获取物资领用分页查询接口")
    public PageList<CmgtMaterialPurchasing> recipientsPage(@ApiParam(name = "queryFilter", value = "分页查询信息") @RequestBody QueryFilter<CmgtMaterialPurchasing> queryFilter) {
        queryFilter.addFilter("applyType", "2", QueryOP.EQUAL);
        queryFilter.addFilter("STATUS_", "1", QueryOP.NOT_EQUAL);
        PageList<CmgtMaterialPurchasing> query = baseService.query(queryFilter);
        query.getRows().forEach(cmgtMaterialPurchasing -> {
            if("2".equals(cmgtMaterialPurchasing.getStatus())){
                cmgtMaterialPurchasing.setStatus("未领用");
            }
            if("3".equals(cmgtMaterialPurchasing.getStatus())){
                cmgtMaterialPurchasing.setStatus("已领用");
            }
        });
        return baseService.query(queryFilter);
    }


    @PostMapping("/recipients")
    @ApiOperation("领用物资")
    public CommonResult<String> create(@ApiParam(name = "model", value = "实体信息") @RequestBody @Validated({AddGroup.class}) RecipientsVo t) {
        boolean save = flowLogManager.recipients(t);
        if (!save) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, null);
        }
        return new CommonResult<>();
    }

    @DeleteMapping("/recipientsMaterialList")
    @ApiOperation("根据选中的领用申请单ids获取领用物资信息")
    public CommonResult<String> recipientsMaterialList(@ApiParam(name = "ids", value = "实体集合") @RequestParam String... ids) {
        Assert.notEmpty(ids, "请选择要领用的申请单");
        List<CmgtMaterialPurchasingDetail> list = this.baseService.recipientsMaterialList(Arrays.asList(ids));
        return CommonResult.success(list, null);
    }
}
