package com.artfess.cqlt.task.controller;

import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.controller.BaseController;
import com.artfess.base.enums.ResponseErrorEnums;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.util.BeanUtils;
import com.artfess.cqlt.task.manager.SchedulerTaskLogManager;
import com.artfess.cqlt.task.manager.SchedulerTaskManager;
import com.artfess.cqlt.task.model.SchedulerTask;
import com.artfess.cqlt.task.model.SchedulerTaskLog;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.quartz.SchedulerException;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;

/**
 * 数据抽取任务信息 前端控制器
 *
 * @company 阿特菲斯信息技术有限公司
 * @author chens
 * @since 2021-12-16
 */
@RestController
@RequestMapping("/biz/task/schedulerTask/v1/")
@Api(tags="数据抽取任务")
@ApiGroup(group= {ApiGroupConsts.GROUP_BIZ})
public class SchedulerTaskController extends BaseController<SchedulerTaskManager, SchedulerTask> {

    @Resource
    SchedulerTaskLogManager schedulerTaskLogManager;

    @PostMapping(value="/sliceQuery", produces={"application/json; charset=utf-8" })
    @ApiOperation("分页数据抽取任务")
    public PageList<SchedulerTask> sliceQuery(@ApiParam(name="queryFilter", value="分页查询信息") @RequestBody QueryFilter<SchedulerTask> queryFilter) {
        return baseService.query(queryFilter);
    }

    @Override
    @PostMapping("/")
    @ApiOperation("添加数据抽取任务")
    public CommonResult<String> create(@ApiParam(name="model", value="数据抽取任务") @RequestBody SchedulerTask schedulerTask) {
        boolean rtn = BeanUtils.validClass(schedulerTask.getTaskClass());
        if (rtn) {
            boolean result = baseService.insertSchedulerTask(schedulerTask);
            if(!result) {
                return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, "新增数据抽取任务失败!");
            }
        } else {
            return new CommonResult(false, "验证类失败!", null);
        }

        return new CommonResult<>();
    }

    @Override
    @PutMapping("/")
    @ApiOperation("更新数据抽取任务")
    public CommonResult<String> updateById(@ApiParam(name="model", value="数据抽取任务") @RequestBody SchedulerTask schedulerTask) {
        try {
            boolean result = baseService.updateSchedulerTask(schedulerTask);
            if(!result) {
                return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, "更新数据抽取任务失败!");
            }
            return new CommonResult<>("修改成功！");
        }catch (Exception e){
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, e.getMessage());
        }
    }

    @Override
    @DeleteMapping("/{id}")
    @ApiOperation("根据id删除")
    public CommonResult<String> deleteById(@ApiParam(name="id", value="实体id") @PathVariable String id) {
        try {
            boolean result = false;
            result = baseService.deleteSchedulerTaskById(id);
            if(!result) {
                return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, "删除实体失败");
            }
            return new CommonResult<>("删除成功！");
        } catch (SchedulerException e) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION,  e.getMessage());
        }
    }

    @PostMapping(value = "/startUpJob")
    @ApiOperation(value = "启动数据抽取任务", httpMethod = "POST", notes = "启动数据抽取任务")
    public CommonResult<String> startUpJob(@ApiParam(name = "taskId", value = "任务id", required = false) @RequestParam String taskId) throws SchedulerException {
         boolean result = baseService.startUpJob(taskId);
         if(result == true){
             return new CommonResult<>("启动任务成功！");
         }else{
             return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, "启动任务失败！");
         }
    }

    @PostMapping(value = "/stopJob")
    @ApiOperation(value = "停止数据抽取任务", httpMethod = "POST", notes = "停止数据抽取任务")
    public CommonResult<String> stopJob(@ApiParam(name = "taskId", value = "任务id", required = true) @RequestParam String taskId) throws SchedulerException {
        boolean result = baseService.stopJob(taskId);
        if(result == true){
            return new CommonResult<>("停止任务成功！");
        }else{
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, "停止任务失败！");
        }
    }

    @PostMapping(value = "/delJob")
    @ApiOperation("根据任务id删除任务")
    public CommonResult<String> delJob(@ApiParam(name = "taskId", value = "任务id", required = true) @RequestParam String taskId) throws SchedulerException {
        boolean result = baseService.delJob(taskId);
        if(result == true){
            return new CommonResult<>("删除任务成功！");
        }else{
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, "删除任务失败！");
        }
    }

    @PostMapping(value="/sliceQueryLog", produces={"application/json; charset=utf-8" })
    @ApiOperation("分页数据抽取任务日志")
    public PageList<SchedulerTaskLog> sliceQueryLog(@ApiParam(name="queryFilter", value="分页查询信息") @RequestBody QueryFilter<SchedulerTaskLog> queryFilter) {
        return schedulerTaskLogManager.query(queryFilter);
    }

}
