package com.artfess.device.base.controller;

import com.alibaba.fastjson.JSON;
import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.annotation.PowerLogInfo;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.enums.LogType;
import com.artfess.base.enums.OperationType;
import com.artfess.base.enums.ResponseErrorEnums;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.valid.UpdateGroup;
import com.artfess.device.base.manager.DeviceVideoBaseManager;
import com.artfess.device.base.manager.DeviceVideoPointManager;
import com.artfess.device.base.model.DeviceVideoBase;
import com.artfess.device.base.model.DeviceVideoPoint;
import com.artfess.device.base.model.DeviceVideoPointExtend;
import com.artfess.device.base.model.DeviceVideoZone;
import com.artfess.device.base.vo.CollectVo;
import com.artfess.device.base.vo.HaiKangRequest;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;

/**
 * 视频目录基础信息表（DEVICE_VIDEO_BASE） 前端控制器
 *
 * @company 阿特菲斯信息技术有限公司
 * @author min.wu
 * @since 2022-08-19
 */
@Slf4j
@RestController
@RequestMapping("/biz/catalog/base/")
@Api(tags = "设施设备-视频设施")
@ApiGroup(group = {ApiGroupConsts.DEVICE_BIZ})
public class DeviceVideoBaseController {

    @Autowired
    private DeviceVideoBaseManager baseService;

    @Autowired
    private DeviceVideoPointManager videoPointManager;

    @PostMapping("/getCataLogBase")
    @ApiOperation("获取视频目录")
    @PowerLogInfo(logType = LogType.BIZ, operaionType = OperationType.QUERY, description = "获取视频目录")
    public CommonResult<String> getCataLogBase() {
        List<DeviceVideoBase> list = baseService.getCataLogBase();
        return CommonResult.success(list, null);
    }

    @PostMapping("/getTree")
    @ApiOperation("获取目录区域信息树")
    @PowerLogInfo(logType = LogType.BIZ, operaionType = OperationType.QUERY, description = "获取目录区域信息树")
    public CommonResult<String> getTree(@ApiParam(name="model", value="实体信息") @RequestBody CollectVo t) {
        List<DeviceVideoZone> list = baseService.getTree(t);
        return CommonResult.success(list, null);
    }

    @PostMapping("/asycCataLog")
    @ApiOperation("同步海康目录信息")
    @PowerLogInfo(logType = LogType.BIZ, operaionType = OperationType.QUERY, description = "同步海康目录信息")
    public CommonResult<String> asycCataLog() {
        baseService.asycCataLog();
        return new CommonResult<>();
    }

    @PostMapping("/asycCataLogZone")
    @ApiOperation("根据目录信息同步海康区域信息，使用treeCode（视频目录code）作为入参")
    @PowerLogInfo(logType = LogType.BIZ, operaionType = OperationType.QUERY, description = "根据目录信息同步海康区域信息")
    public CommonResult<String> asycCataLogZone(@RequestBody HaiKangRequest haiKangRequest) {
        baseService.asycCataLogZone(haiKangRequest);
        return new CommonResult<>();
    }

    @PostMapping("/asycZoneRes")
    @ApiOperation("根据区域编号同步海康监控点信息，使用regionIndexCode（区域下拉code）和treeCode（视频目录code）作为入参")
    @PowerLogInfo(logType = LogType.BIZ, operaionType = OperationType.QUERY, description = "根据区域编号同步海康监控点信息")
    public CommonResult<String> asycZoneRes(@RequestBody HaiKangRequest haiKangRequest) {
        HaiKangRequest asycRequest =new HaiKangRequest();
        //asycRequest.setRegionIndexCode(haiKangRequest.getRegionIndexCode());
        baseService.asycZoneRes(asycRequest);
        return new CommonResult<>();
    }

    @PostMapping(value="/query", produces={"application/json; charset=utf-8" })
    @ApiOperation("分页查询监控点")
    @PowerLogInfo(logType = LogType.BIZ, operaionType = OperationType.QUERY, description = "分页查询监测点")
    public CommonResult<String> query(@ApiParam(name="queryFilter", value="分页查询信息") @RequestBody QueryFilter<DeviceVideoBase> queryFilter) {
        log.info("分页查询监测点请求参数:{}", JSON.toJSONString(queryFilter));
        PageList<DeviceVideoPoint> page = baseService.findByPage(queryFilter);
        log.info("分页查询监测点响应数据:{}", JSON.toJSONString(page));
        return CommonResult.success(page, "查询成功");
    }

    @PostMapping("/hkPreviewURLs")
    @ApiOperation("根据海康监控点获取视频流接口")
    @PowerLogInfo(logType = LogType.BIZ, operaionType = OperationType.ADD, description = "根据监控点获取视频流接口")
    public CommonResult<String> hkPreviewURLs(@ApiParam(name="code",value="视频点位编码",required=true)  @RequestParam String code,
                                              @ApiParam(name="module",value="视频流协议（rtsp,rtmp,hls,ws,wss）",required=true)  @RequestParam String module) {
        String result = baseService.hkPreviewURLs(code,module);
        return CommonResult.success(result, "获取成功");
    }

    @RequestMapping(value="/zgPreviewURLs",method= RequestMethod.POST, produces = {"application/json; charset=utf-8" })
    @ApiOperation(value = "根据紫光华智监控点获取视频流接口", httpMethod = "POST", notes = "根据紫光华智监控点获取视频流接口")
    public CommonResult<String> zgPreviewURLs(@ApiParam(name="code",value="视频点位编码",required=true)  @RequestParam String code,
                                              @ApiParam(name="module",value="视频流协议（1:RTSP,2:RTMP,3:HLS,4:HLS_SSL,5:HTTP FLV,6:HTTPS FLV,7:WS FLV,8:WSS FLV）",required=true)  @RequestParam Integer module) {
        String url = videoPointManager.getZGVideoUri(code,module,false);
        return new CommonResult<String>(true, "",url);
    }

    @PostMapping(value="/findByZoneCode", produces={"application/json; charset=utf-8" })
    @ApiOperation(value = "根据区域信息获取监控点列表", response = DeviceVideoPoint.class)
    @PowerLogInfo(logType = LogType.BIZ, operaionType = OperationType.QUERY, description = "根据区域信息获取监控点列表")
    public CommonResult<String> findByZoneCode(@ApiParam(name="model", value="分页查询信息") @RequestBody QueryFilter<DeviceVideoBase> queryFilter) {
        log.info("根据区域信息获取监控点列表请求参数:{}", JSON.toJSONString(queryFilter));
        List<DeviceVideoPoint> list = baseService.findByZoneCode(queryFilter);
        log.info("根据区域信息获取监控点列表响应数据:{}", JSON.toJSONString(list));
        return CommonResult.success(list, "查询成功");
    }


    @PostMapping("/getAllTree")
    @ApiOperation("获取目录区域信息和监控点信息树")
    @PowerLogInfo(logType = LogType.BIZ, operaionType = OperationType.QUERY, description = "获取目录区域信息和监控点信息树")
    public CommonResult<String> getAllTree(@ApiParam(name="model", value="实体信息") @RequestBody CollectVo t) {
        List<DeviceVideoZone> list = baseService.getAllTree(t);
        return CommonResult.success(list, null);
    }

    @PutMapping("/updateById")
    @ApiOperation("修改点位扩展信息")
    public CommonResult<String> updateById(@ApiParam(name="model", value="实体信息") @RequestBody @Validated({UpdateGroup.class}) DeviceVideoPointExtend t) {
        String id = videoPointManager.updateInfo(t);
        if(!StringUtils.isNotBlank(id)) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, "更新实体失败");
        }
        return new CommonResult<>();
    }


    @GetMapping("/{code}")
    @ApiOperation("获取视频点位信息")
    public DeviceVideoPoint getById(@ApiParam(name="code", value="实体id") @PathVariable String code) {
        return videoPointManager.findByCode(code);
    }

}
