package com.artfess.cgpt.supplier.controller;

import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.controller.BaseController;
import com.artfess.base.feign.UCFeignService;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.*;
import com.artfess.base.util.BeanUtils;
import com.artfess.cgpt.expert.manager.BizExpertDatabaseManager;
import com.artfess.cgpt.expert.model.BizExpertDatabase;
import com.artfess.cgpt.supplier.manager.EnterpriseManager;
import com.artfess.cgpt.supplier.model.BizSupplierEnterprise;
import com.artfess.cgpt.supplier.model.Enterprise;
import com.artfess.cgpt.supplier.vo.BizEnterpriseVo;
import com.artfess.cgpt.utils.BizUtils;
import com.artfess.uc.manager.UserManager;
import com.artfess.uc.model.Org;
import com.artfess.uc.model.User;
import com.artfess.uc.util.ContextUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.fasterxml.jackson.databind.node.ArrayNode;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;
import java.util.stream.Collectors;

/**
 * @Description:企业管理
 * @Author: Rong Tao
 * @Date: 2024/7/6 10:07
 */
@RestController
@RequestMapping("/enterprise/v1/")
@Api(tags="企业管理")
@ApiGroup(group = {ApiGroupConsts.GROUP_BIZ})
public class EnterpriseController extends BaseController<EnterpriseManager, Enterprise> {

    @Autowired
    UserManager userService;
    @Autowired
    UCFeignService ucFeignService;
    @Autowired
    private BizExpertDatabaseManager expertDatabaseManager;

    @PostMapping(value="/queryOrgNoOccupy")
    @ApiOperation("M-分页查询未添加战略供应商的企业")
    public PageList<Org> queryOrgNoOccupy(@RequestBody QueryFilter<Enterprise> queryFilter) {
        return baseService.queryOrgNoOccupy(queryFilter);
    }

    @PostMapping(value="/queryAllOrg")
    @ApiOperation("M-分页查询所有组织（企业表联查）")
    public PageList<Org> queryAllOrg(@RequestBody QueryFilter<Enterprise> queryFilter) {
        return baseService.queryOrgNoOccupy(queryFilter);
    }

    @PostMapping(value="/queryAll")
    @ApiOperation("M-分页查询所有数据")
    public PageList<Enterprise> queryAll(@RequestBody QueryFilter<Enterprise> queryFilter) {
        return baseService.queryByPage(queryFilter);
    }

    @PostMapping("/saveOrUpdateEntity")
    @ApiOperation("保存或更新数据")
    public CommonResult saveOrUpdateEntity(@RequestBody Enterprise entity) {
        baseService.saveOrUpdateEntity(entity);
        return new CommonResult(true,"操作成功");
    }

    /**
     * 根据公司获取用户列表
     * @param vo 查询条件vo
     * @return
     * @throws Exception
     */
    @PostMapping("/getNoExpertUserByOrg")
    @ApiOperation(value = "获取当前公司不在专家库的用户列表", httpMethod = "POST", notes = "获取用户列表")
    public PageList<User> getNoExpertUserByOrg(@ApiParam(name="vo",value="查询条件vo") @RequestBody BizEnterpriseVo vo) throws Exception{
        QueryFilter<User> queryFilter = QueryFilter.build();
        queryFilter.setPageBean(new PageBean(vo.getIndex(),vo.getSize()));

        if(BeanUtils.isEmpty(ContextUtil.getCurrentOrgId())) {
            return new PageList<>();
        }
        //查询当前公司的用户
        queryFilter.addFilter("ORG_ID_", ContextUtil.getCurrentOrgId(), QueryOP.EQUAL, FieldRelation.AND,"orgTree");

        //过滤专家库已存在的
        LambdaQueryWrapper<BizExpertDatabase> expertQW = new LambdaQueryWrapper<>();
        expertQW.select(BizExpertDatabase::getAccount)
                .eq(BizExpertDatabase::getOrgId,ContextUtil.getCurrentOrgId())
                .eq(BizExpertDatabase::getIsDele,"0");
        List<Object> objs = expertDatabaseManager.listObjs(expertQW);
        if(BeanUtils.isNotEmpty(objs) && objs.size()>0){
            List<String> accounts = objs.stream().map(s -> String.valueOf(s)).collect(Collectors.toList());
            queryFilter.addFilter("account_", accounts, QueryOP.NOT_IN, FieldRelation.AND,"acc");
        }

        if(BeanUtils.isNotEmpty(vo.getFullName())){
            queryFilter.addFilter("fullname",vo.getFullName(), QueryOP.LIKE, FieldRelation.OR,"quick");
        }

        IPage<User> list = userService.getOrgUserQuery(queryFilter);
        return new PageList<>(list);
    }
    /**
     * 根据公司获取用户列表
     * @param vo 查询条件vo
     * @return
     * @throws Exception
     */
    @PostMapping("/getNoExpertUserByOrgPlat")
    @ApiOperation(value = "获取不在专家库的用户列表", httpMethod = "POST", notes = "获取用户列表")
    public PageList<User> getNoExpertUserByOrgPlat(@ApiParam(name="vo",value="查询条件vo") @RequestBody BizEnterpriseVo vo) throws Exception{
        QueryFilter<User> queryFilter = QueryFilter.build();
        queryFilter.setPageBean(new PageBean(vo.getIndex(),vo.getSize()));
        //如果组织编号不为空，查询该组织下的用户
        if(BeanUtils.isNotEmpty(vo.getCompanyCode())){
            ArrayNode orgListByUserId = ucFeignService.getOrgsByCodes(vo.getCompanyCode());
            if(BeanUtils.isNotEmpty(orgListByUserId)){
                String id = orgListByUserId.get(0).get("id").asText();
                queryFilter.addFilter("ORG_ID_",id, QueryOP.IN, FieldRelation.AND,"orgTree");

                //过滤专家库已存在的
                LambdaQueryWrapper<BizExpertDatabase> expertQW = new LambdaQueryWrapper<>();
                expertQW.select(BizExpertDatabase::getAccount)
                        .in(BizExpertDatabase::getOrgId,id)
                        .eq(BizExpertDatabase::getIsDele,"0");
                List<Object> objs = expertDatabaseManager.listObjs(expertQW);
                if(BeanUtils.isNotEmpty(objs) && objs.size()>0){
                    List<String> accounts = objs.stream().map(s -> String.valueOf(s)).collect(Collectors.toList());
                    queryFilter.addFilter("account_", accounts, QueryOP.NOT_IN, FieldRelation.AND,"acc");
                }
            }
        }else {
            //如果组织编号为空
            //1.判断当前用户是否为管理员，如果不是-》查询当前用户公司的用户
            if(BizUtils.isAdmin(ucFeignService)){
                queryFilter.addFilter("ORG_ID_", ContextUtil.getCurrentOrgId(), QueryOP.IN, FieldRelation.AND,"orgTree");

                //过滤专家库已存在的
                LambdaQueryWrapper<BizExpertDatabase> expertQW = new LambdaQueryWrapper<>();
                expertQW.select(BizExpertDatabase::getAccount)
                        .eq(BizExpertDatabase::getOrgId,ContextUtil.getCurrentOrgId())
                        .eq(BizExpertDatabase::getIsDele,"0");
                List<Object> objs = expertDatabaseManager.listObjs(expertQW);
                if(BeanUtils.isNotEmpty(objs) && objs.size()>0){
                    List<String> accounts = objs.stream().map(s -> String.valueOf(s)).collect(Collectors.toList());
                    queryFilter.addFilter("account_", accounts, QueryOP.NOT_IN, FieldRelation.AND,"acc");
                }
            }

        }

        if(BeanUtils.isNotEmpty(vo.getFullName())){
            queryFilter.addFilter("fullname",vo.getFullName(), QueryOP.LIKE, FieldRelation.OR,"quick");
        }

        IPage<User> list = userService.getOrgUserQuery(queryFilter);
        return new PageList<>(list);
    }

    /**
     * 根据公司获取用户列表
     * @param vo 查询条件vo
     * @return
     * @throws Exception
     */
    @PostMapping("/getUserByOrgPlat")
    @ApiOperation(value = "获取用户列表-平台管理", httpMethod = "POST", notes = "获取用户列表")
    public PageList<User> getUserByOrgPlat(@ApiParam(name="vo",value="查询条件vo") @RequestBody BizEnterpriseVo vo) throws Exception{
        QueryFilter<User> queryFilter = QueryFilter.build();
        queryFilter.setPageBean(new PageBean(vo.getIndex(),vo.getSize()));
        //如果组织编号不为空，查询该组织下的用户
        if(BeanUtils.isNotEmpty(vo.getCompanyCode())){
            ArrayNode orgListByUserId = ucFeignService.getOrgsByCodes(vo.getCompanyCode());
            if(BeanUtils.isNotEmpty(orgListByUserId)){
                String id = orgListByUserId.get(0).get("id").asText();
                queryFilter.addFilter("ORG_ID_",id, QueryOP.IN, FieldRelation.AND,"orgTree");
            }
        }else {
            //如果组织编号为空
            //1.判断当前用户是否为管理员，如果不是-》查询当前用户公司的用户
            if(BizUtils.isAdmin(ucFeignService)){
                queryFilter.addFilter("ORG_ID_", ContextUtil.getCurrentOrgId(), QueryOP.IN, FieldRelation.AND,"orgTree");
            }

        }

        if(BeanUtils.isNotEmpty(vo.getFullName())){
            queryFilter.addFilter("fullname",vo.getFullName(), QueryOP.LIKE, FieldRelation.OR,"quick");
        }

        IPage<User> list = userService.getOrgUserQuery(queryFilter);
        return new PageList<>(list);
    }

    /**
     * 根据公司获取用户列表
     * @param vo 查询条件vo
     * @return
     * @throws Exception
     */
    @PostMapping("/getUserByOrg")
    @ApiOperation(value = "获取用户列表", httpMethod = "POST", notes = "获取用户列表")
    public PageList<User> getUserByOrg(@ApiParam(name="vo",value="查询条件vo") @RequestBody BizEnterpriseVo vo) throws Exception{
        QueryFilter<User> queryFilter = QueryFilter.build();
        queryFilter.setPageBean(new PageBean(vo.getIndex(),vo.getSize()));

        if(BeanUtils.isEmpty(ContextUtil.getCurrentOrgId())) {
            return new PageList<>();
        }
        //查询当前公司的用户
        queryFilter.addFilter("ORG_ID_", ContextUtil.getCurrentOrgId(), QueryOP.EQUAL, FieldRelation.AND,"orgTree");


        if(BeanUtils.isNotEmpty(vo.getFullName())){
            queryFilter.addFilter("fullname",vo.getFullName(), QueryOP.LIKE, FieldRelation.OR,"quick");
        }

        IPage<User> list = userService.getOrgUserQuery(queryFilter);
        return new PageList<>(list);
    }

    /**
     * 获取用户列表（含组织信息）
     * @param queryFilter 通用查询
     * @return
     * @throws Exception
     */
    @PostMapping("/getUserList")
    @ApiOperation(value = "获取用户列表（含组织信息）", httpMethod = "POST", notes = "获取用户列表（含组织信息）")
    public CommonResult<PageList<User>> getUserList(@RequestBody QueryFilter<User> queryFilter) throws Exception {
        PageList<User> result = userService.queryOrgUserPage(queryFilter);
        return new CommonResult<>(true,"查询成功",result);
    }

}
