package com.artfess.dataShare.dataApi.manager.impl;

import com.artfess.base.enums.ResponseErrorEnums;
import com.artfess.base.exception.ApplicationException;
import com.artfess.base.model.CommonResult;
import com.artfess.base.util.BeanUtils;
import com.artfess.dataShare.dataApi.manager.BizServiceApiManager;
import com.artfess.dataShare.dataApi.model.BizServiceApi;
import com.artfess.dataShare.dataApi.model.BizServiceCatalogs;
import com.artfess.dataShare.dataApi.dao.BizServiceCatalogsDao;
import com.artfess.dataShare.dataApi.manager.BizServiceCatalogsManager;
import com.artfess.base.manager.impl.BaseManagerImpl;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 数据资产-数据服务目录 服务实现类
 *
 * @company 阿特菲斯信息技术有限公司
 * @author wubin
 * @since 2024-11-21
 */
@Service
public class BizServiceCatalogsManagerImpl extends BaseManagerImpl<BizServiceCatalogsDao, BizServiceCatalogs> implements BizServiceCatalogsManager {


    @Resource
    private BizServiceApiManager serviceApiManager;

    @Override
    @Transactional
    public String upTree(BizServiceCatalogs serviceCatalogs){
        if (!validateCatalog(serviceCatalogs)){
            throw new ApplicationException("目录名字【" + serviceCatalogs.getName() + "】或目录编号【" + serviceCatalogs.getCode() + "】重复！");
        }
        BizServiceCatalogs old = this.get(serviceCatalogs.getId());
        serviceCatalogs.setHasChildren(old.getHasChildren());
        serviceCatalogs.setIsDele(old.getIsDele());
        return this.updateTree(serviceCatalogs,old.getName());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean deleteById(String id) {
        BizServiceCatalogs bizServiceCatalogs = this.get(id);
        Integer count = this.getHasChildren(bizServiceCatalogs);
        Assert.state(count == 0,"该目录下有子目录，删除失败");
        //判断该目录下有没有管理接口
        LambdaQueryWrapper<BizServiceApi> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(BizServiceApi::getCatalogId,id);
        int apiCount = serviceApiManager.count(queryWrapper);
        if (apiCount > 0){
            throw new ApplicationException("该目录下有接口管理，删除失败！");
        }
        boolean result = this.removeById(id);
        //更新父目录的子节点
        if(result){
            refreshHasChildren(bizServiceCatalogs);
        }
        return result;
    }

    @Override
    public String saveTree(BizServiceCatalogs serviceCatalogs) {
        if (!validateCatalog(serviceCatalogs)){
            throw new ApplicationException("目录名字【" + serviceCatalogs.getName() + "】或目录编号【" + serviceCatalogs.getCode() + "】重复！");
        }
        return this.insertTree(serviceCatalogs);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean removeTree(String id, String parentId) {
        boolean result = false;
        BizServiceCatalogs oldCatalog = get(id);
        BizServiceCatalogs oldParent = get(oldCatalog.getParentId());
        BizServiceCatalogs newCatalog = get(id);
        newCatalog.setParentId(parentId);
        if (!validateCatalog(newCatalog)){
            throw new ApplicationException("目录名字【" + newCatalog.getName() + "】或目录编号【" + newCatalog.getCode() + "】重复！");
        }

        BizServiceCatalogs newParent = get(parentId);
        newCatalog.buildFullIdAndName(newParent);
        List<BizServiceCatalogs> children = baseMapper.selectList(new LambdaQueryWrapper<BizServiceCatalogs>()
                .eq(BizServiceCatalogs::getIsDele, "0")
                .eq(BizServiceCatalogs::getParentId, newCatalog.getId()));
        for (BizServiceCatalogs child : children) {
            child.buildFullIdAndName(newCatalog);
            updateById(child);
        }
        if (this.updateTree(newCatalog,oldCatalog.getName()) != null){
            oldParent.setHasChildren(oldParent.getHasChildren() - 1);
            result = updateById(oldParent);;
        }
        return result;
    }

    /**
     * 验证重复目录
     * @param serviceCatalogs
     * @return
     */
    private boolean validateCatalog(BizServiceCatalogs serviceCatalogs){
        LambdaQueryWrapper<BizServiceCatalogs> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(BizServiceCatalogs::getParentId,serviceCatalogs.getParentId());
        queryWrapper.and(w -> {
            w.eq(BizServiceCatalogs::getName,serviceCatalogs.getName())
                    .or()
                    .eq(BizServiceCatalogs::getCode,serviceCatalogs.getCode());
        });
        if (BeanUtils.isNotEmpty(serviceCatalogs.getId())){
            queryWrapper.ne(BizServiceCatalogs::getId,serviceCatalogs.getId());
        }
        int count = count(queryWrapper);
        return count > 0 ? false : true;
    }

    /**
     * 更新父目录的子节点
     * @param bizServiceCatalogs 当前目录
     */
    @Transactional(rollbackFor = Exception.class)
    public void refreshHasChildren(BizServiceCatalogs bizServiceCatalogs){
        //父目录
        BizServiceCatalogs parent = this.get(bizServiceCatalogs.getParentId());
        if (parent == null){
            return;
        }
        //获取子目录数量
        Integer hasChildren = getHasChildren(parent);
        parent.setHasChildren(hasChildren);
        //更新子目录
        this.update(parent);
        refreshHasChildren(parent);
    }

    /**
     * 获取子目录的数量
     * @param bizServiceCatalogs
     * @return
     */
    private Integer getHasChildren(BizServiceCatalogs bizServiceCatalogs){
        LambdaQueryWrapper<BizServiceCatalogs> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(BizServiceCatalogs::getParentId,bizServiceCatalogs.getId());
        Integer count = this.baseMapper.selectCount(queryWrapper);
        return count;
    }
}
