package com.artfess.dataShare.dataApi.manager.impl;

import com.alibaba.fastjson.JSONArray;
import com.artfess.base.enums.ResponseErrorEnums;
import com.artfess.base.exception.ApplicationException;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.util.BeanUtils;
import com.artfess.base.util.StringUtil;
import com.artfess.dataShare.dataApi.model.BizServiceApi;
import com.artfess.dataShare.dataApi.dao.BizServiceApiDao;
import com.artfess.dataShare.dataApi.manager.BizServiceApiManager;
import com.artfess.base.manager.impl.BaseManagerImpl;
import com.artfess.dataShare.dataApi.vo.RespParamVo;
import com.artfess.dataShare.dataShare.model.BizShareApi;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.web.client.RestTemplate;

import javax.annotation.Resource;
import java.io.IOException;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.Map;

/**
 * 数据资产-数据服务接口信息 服务实现类
 *
 * @company 阿特菲斯信息技术有限公司
 * @author wubin
 * @since 2024-11-22
 */
@Slf4j
@Service
public class BizServiceApiManagerImpl extends BaseManagerImpl<BizServiceApiDao, BizServiceApi> implements BizServiceApiManager {

    @Resource
    private RestTemplate restTemplate;

    @Value("${dataShare-api.url.serviceApi.register}")
    private String API_REGISTER_URL;

    @Value("${dataShare-api.url.serviceApi.checkSqlValidity}")
    private String API_CHECKSQLVALIDITY_URL;

    @Value("${dataShare-api.url.serviceApi.unregister}")
    private String API_UNREGISTER_URL;

    @Value("${dataShare-api.url.serviceApi.updateServiceApi}")
    private String API_UPDATESERVICEAPI_URL;

    @Value("${dataShare-api.url.serviceApi.query}")
    private String API_QUERY_URL;

    @Value("${dataShare-api.url.serviceApi.getInfo}")
    private String API_GETINFO_URL;

    @Value("${dataShare-api.url.serviceApi.checkApi}")
    private String API_CHECKAPI_URL;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean register(BizServiceApi serviceApi) {
        serviceApi.setApiUrl("/api"+serviceApi.getApiUrl());
        serviceApi.setStatus("0");
        serviceApi.setIsDele("0");
        boolean status = this.registerToDB(serviceApi);
        return status;
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public boolean publish(String id) {
        BizServiceApi serviceApi = this.get(id);
        int count = 0;
        if (BeanUtils.isNotEmpty(serviceApi)){
            serviceApi.setStatus("1");
            count = baseMapper.updateById(serviceApi);
        }
        CommonResult commonResult = new CommonResult<>(ResponseErrorEnums.FAIL_OPTION);
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        HttpEntity<BizServiceApi> request = new HttpEntity<>(serviceApi,headers);
        //在容器中注册接口
        commonResult = restTemplate.postForObject(API_REGISTER_URL, request, CommonResult.class);
        return commonResult.getState();
    }

    @Override
    public boolean checkApiValidity(BizServiceApi serviceApi) {
        return checkApiInDB(serviceApi) && checkApiInSpring(serviceApi);
    }

    private boolean checkApiInSpring(BizServiceApi serviceApi) {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        HttpEntity<BizServiceApi> request = new HttpEntity<>(serviceApi,headers);
        CommonResult<String> commonResult = restTemplate.postForObject(API_CHECKAPI_URL, request, CommonResult.class);
        return commonResult.getState();
    }

    public boolean checkApiInDB(BizServiceApi serviceApi){
        LambdaQueryWrapper<BizServiceApi> queryWrapper = new LambdaQueryWrapper<BizServiceApi>();
        queryWrapper.eq(BizServiceApi::getApiUrl,serviceApi.getApiUrl());
        queryWrapper.eq(BizServiceApi::getApiMode,serviceApi.getApiMode());
        return this.count(queryWrapper) == 0 ? true : false;
    }

    @Override
    public boolean checkSqlValidity(BizServiceApi serviceApi) throws SQLException {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        HttpEntity<BizServiceApi> request = new HttpEntity<>(serviceApi,headers);
        CommonResult commonResult = restTemplate.postForObject(API_CHECKSQLVALIDITY_URL, request, CommonResult.class);
        return commonResult.getState();
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public boolean unregister(String id) {
        BizServiceApi serviceApi = this.get(id);
        boolean status = this.unregisterFromDB(serviceApi.getId());
        CommonResult commonResult = new CommonResult<>(ResponseErrorEnums.FAIL_OPTION);

        if (status){
            HttpHeaders headers = new HttpHeaders();
            headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
            HttpEntity<BizServiceApi> request = new HttpEntity<>(serviceApi,headers);
            commonResult = restTemplate.postForObject(API_UNREGISTER_URL, request, CommonResult.class);
        }

        if(!commonResult.getState()){
            status = false;
            throw new ApplicationException(commonResult.getMessage());
        }
        status = commonResult.getState();
        return status;
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public boolean soldOut(String id) {
        boolean status = false;
        BizServiceApi serviceApi = this.get(id);
        serviceApi.setStatus("0");
        int count = 0;
        if (BeanUtils.isNotEmpty(serviceApi)){
            count = baseMapper.updateById(serviceApi);
        }
        CommonResult commonResult = new CommonResult<>(ResponseErrorEnums.FAIL_OPTION);
        if (count > 0){
            HttpHeaders headers = new HttpHeaders();
            headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
            HttpEntity<BizServiceApi> request = new HttpEntity<>(serviceApi,headers);
            commonResult = restTemplate.postForObject(API_UNREGISTER_URL, request, CommonResult.class);
        }
        if(!commonResult.getState()){
            status = false;
            throw new ApplicationException(commonResult.getMessage());
        }
        status = commonResult.getState();
        return status;
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public boolean updateServiceApi(BizServiceApi serviceApi) {
        BizServiceApi oldServiceApi = this.get(serviceApi.getId());
        if(!oldServiceApi.getApiMode().name().equals(serviceApi.getApiMode().name())
                || !oldServiceApi.getApiUrl().equals(serviceApi.getApiUrl())){
            if (!checkApiValidity(serviceApi)){
                throw new ApplicationException("接口已被注册！");
            }
        }
        if(!checkApiCode(serviceApi)){
            throw new ApplicationException("接口编码重复！");
        }
        int count = baseMapper.updateById(serviceApi);
        return count > 0 ? true : false;
    }

    @Override
    public PageList<BizServiceApi> queryApi(QueryFilter<BizServiceApi> queryFilter) {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);
        HttpEntity<QueryFilter<BizServiceApi>> request = new HttpEntity<>(queryFilter,headers);
        PageList pageList = restTemplate.postForObject(API_QUERY_URL, request, PageList.class);
        return pageList;
    }

    @Override
    public BizServiceApi getInfo(String id) {
        Assert.hasText(id,"ID不能为空！");
        BizServiceApi serviceApi =this.baseMapper.selectById(id);
        /*Map<String,String> param = new HashMap<>();
        param.put("id",id);
        BizServiceApi serviceApi = restTemplate.getForObject(API_GETINFO_URL, BizServiceApi.class, param);*/
        return serviceApi;
    }

    @Override
    public BizServiceApi viewApiDocument(String id) throws IOException {
        Assert.hasText(id,"ID不能为空！");
        BizServiceApi serviceApi =this.baseMapper.selectById(id);
/*        Map<String,String> param = new HashMap<>();
        param.put("id",id);
        BizServiceApi serviceApi = restTemplate.getForObject(API_GETINFO_URL, BizServiceApi.class, param);*/
        String responseExample = serviceApi.getResponseExample();
        if (StringUtil.isNotEmpty(responseExample)){
            JSONArray jsonArray = JSONArray.parseArray(responseExample);
            Integer size = jsonArray.size();

            RespParamVo respParamVo = new RespParamVo();
            respParamVo.setResultCode("200");
            respParamVo.setMessage("成功");
            respParamVo.setTotalCount(size.toString());
            respParamVo.setCount(size.toString());
            respParamVo.setRows(jsonArray.toJavaList(Object.class));
            serviceApi.setRespParamVo(respParamVo);
        }
        return serviceApi;
    }

    @Override
    public boolean checkApiCode(BizServiceApi serviceApi) {
        LambdaQueryWrapper<BizServiceApi> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(BizServiceApi::getApiCode,serviceApi.getApiCode());
        queryWrapper.and(w -> {
            w.eq(BizServiceApi::getApiCode,serviceApi.getApiCode());
        });
        if (BeanUtils.isNotEmpty(serviceApi.getId())){
            queryWrapper.ne(BizServiceApi::getId,serviceApi.getId());
        }
        int count = count(queryWrapper);
        return count > 0 ? false : true;
    }

    /**
     * 从数据库中删除接口信息
     * @param id
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    public boolean unregisterFromDB(String id){
        return this.removeById(id);
    }

    /**
     * 持久化接口信息
     * @param serviceApi
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    public boolean registerToDB(BizServiceApi serviceApi) {
        if(!checkApiValidity(serviceApi)){
            throw new ApplicationException("接口已被注册！");
        }
        if(!checkApiCode(serviceApi)){
            throw new ApplicationException("接口编码重复！");
        }
        return this.save(serviceApi);
    }
}
