package com.artfess.form.persistence.manager.impl;

import com.artfess.base.datasource.DatabaseContext;
import com.artfess.base.datasource.DatabaseSwitchResult;
import com.artfess.base.exception.BaseException;
import com.artfess.base.manager.CommonManager;
import com.artfess.base.manager.impl.BaseManagerImpl;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.query.QueryOP;
import com.artfess.base.util.AppUtil;
import com.artfess.base.util.BeanUtils;
import com.artfess.base.util.FileUtil;
import com.artfess.base.util.JAXBUtil;
import com.artfess.base.util.SQLUtil;
import com.artfess.base.util.StringUtil;
import com.artfess.base.util.ThreadMsgUtil;
import com.artfess.base.util.UniqueIdUtil;
import com.artfess.base.util.ZipUtil;
import com.artfess.form.model.CustomChart;
import com.artfess.form.model.CustomChartList;
import com.artfess.form.persistence.dao.CustomChartDao;
import com.artfess.form.persistence.manager.CustomChartManager;
import com.artfess.table.meta.impl.BaseTableMeta;
import com.artfess.table.util.MetaDataUtil;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import java.io.File;
import java.io.IOException;
import java.io.Serializable;
import java.nio.file.Paths;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 自定义对图表
 *
 * @author heyifan
 * @company 广州宏天软件股份有限公司
 * @email heyf@jee-soft.cn
 * @date 2020年4月14日
 */
@Service("customChartManager")
public class CustomChartManagerImpl extends BaseManagerImpl<CustomChartDao, CustomChart> implements CustomChartManager {
    @Resource
    CommonManager commonManager;
    @Resource
    DatabaseContext databaseContext;

    @Override
    public CustomChart get(Serializable id) {
        CustomChart customChart = super.get(id);
        if (BeanUtils.isEmpty(customChart)) {
            QueryFilter<CustomChart> queryFilter = QueryFilter.<CustomChart>build();
            queryFilter.addFilter("ALIAS_", id, QueryOP.EQUAL);
            PageList<CustomChart> query = this.query(queryFilter);
            if (query.getRows().size() > 0) {
                customChart = query.getRows().get(0);
            }
        }
        return customChart;
    }

    @SuppressWarnings({"rawtypes", "unchecked"})
    @Override
    public Object getListData(CustomChart customDialog, QueryFilter<?> filter, String dbType) throws IOException {
        String sql = "";
        if (customDialog.getIsTable() == 2) {
            sql = customDialog.getDiySql();
        } else {
            sql = "select * from " + customDialog.getObjName();
        }
        PageList query = commonManager.query(sql, filter);
        //大小写兼容处理
        for (int i = 0; i < query.getRows().size(); i++) {
            Map<String, Object> m = (Map<String, Object>) query.getRows().get(i);
            Map<String, Object> tm = new HashMap<String, Object>();
            for (String k : m.keySet()) {
                tm.put(k.toUpperCase(), m.get(k));
            }
            query.getRows().set(i, tm);
        }
        return query;
    }

    @Override
    public CustomChart getChartByAlias(String alias) {
        return baseMapper.getChartByAlias(alias);
    }

    @Override
    public boolean listChartByAlias(String alias) {
        List<CustomChart> customCharts = baseMapper.listChartByAlias(alias);
        if (customCharts.size() > 1) {
            return false;
        }
        return true;
    }

    @Override
    public Map<String, String> exportData(List<String> ids) throws Exception {
        Map<String, String> map = new HashMap<>();
        CustomChartList customChartList = new CustomChartList();
        for (String id : ids) {
            CustomChart customChart = this.get(id);
            customChartList.addCustomChart(customChart);
        }
        try {
            String xml = JAXBUtil.marshall(customChartList, CustomChartList.class);
            map.put("customCharts.form.xml", xml);
        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException("导出失败！" + e.getMessage(), e);
        }
        return map;
    }

    @Override
    @Transactional
    public void importData(MultipartFile file) {
        String unZipFilePath = null;
        try {
            String rootRealPath = Paths.get(FileUtil.getIoTmpdir(), "attachFiles/unZip").toString();
            FileUtil.createFolder(rootRealPath, true);
            ZipUtil.unZipFile(file, rootRealPath);
            String name = file.getOriginalFilename();
            String fileDir = StringUtil.substringBeforeLast(name, ".");
            unZipFilePath = Paths.get(rootRealPath, fileDir).toString();
            String xml = FileUtil.readFile(Paths.get(unZipFilePath, "customCharts.form.xml").toString());
            if (StringUtil.isEmpty(xml)) {
                throw new BaseException("导入的xml文件没有内容！");
            }
            CustomChartList list = (CustomChartList) JAXBUtil.unmarshall(xml, CustomChartList.class);
            List<CustomChart> customChartList = list.getCustomChartList();
            for (CustomChart customChart : customChartList) {
                importData(customChart);
            }
        } catch (Exception e) {
            throw new BaseException(e.getMessage(), e);
        } finally {
            if (StringUtil.isNotEmpty(unZipFilePath)) {
                File formDir = new File(unZipFilePath);
                if (formDir.exists()) {
                    FileUtil.deleteDir(formDir);
                }
            }
        }
    }

    private void importData(CustomChart customChart) {
        QueryWrapper wrapper = new QueryWrapper();
        wrapper.eq("alias_", customChart.getAlias());
        List<CustomChart> customChartList = baseMapper.selectList(wrapper);
        if (customChartList != null && !customChartList.isEmpty()) {
            // 数据图表 已存在故跳过！
            ThreadMsgUtil.addMapMsg2("customChartsSkipped", String.format("<div style='margin-left:12px; margin-top:6px;'>%s [%s]</div>", customChart.getName() + "", customChart.getAlias() + ""));
            return;
        }
        try (DatabaseSwitchResult dResult = databaseContext.setDataSource(customChart.getDsalias())) {
            // isTable 1:使用数据源中的数据表 2.自定义sql
            short isTable = customChart.getIsTable() != null ? customChart.getIsTable() : 1;
            if (isTable == 1) {
                try {
                    BaseTableMeta baseTableMeta = MetaDataUtil.getBaseTableMetaAfterSetDT(dResult.getDbType());
                    baseTableMeta.getTableByName(customChart.getObjName());
                } catch (Exception e) {
                    throw new BaseException(String.format("数据源: %s 中的数据表: %s 不存在!", customChart.getDsalias() + "", customChart.getObjName() + ""));
                }
            } else {
                checkSql(customChart.getDiySql());
            }
        } catch (Exception e) {
            throw new BaseException(e.getMessage(), e);
        }
        String id = UniqueIdUtil.getSuid();
        customChart.setId(id);
        this.create(customChart);
        // 数据图表 导入成功！
        ThreadMsgUtil.addMapMsg2("customChartsSaved", String.format("<div style='margin-left:12px; margin-top:6px;'>%s [%s]</div>", customChart.getName() + "", customChart.getAlias() + ""));
    }

    private void checkSql(String sql) {
        if (SQLUtil.containsSqlInjection(sql)) {
            throw new BaseException("SQL语句含有非法注入！");
        }
        JdbcTemplate jdbcTemplate = null;
        try {
            jdbcTemplate = AppUtil.getBean(JdbcTemplate.class);
            jdbcTemplate.execute(sql);
        } catch (Exception e) {
            String msg = e.getMessage();
            if (msg != null) {
                if (msg.contains("bad SQL grammar")) {
                    throw new BaseException("SQL语句存在语法错误！");
                }
            }
            throw new BaseException("SQL验证失败: " + e.getMessage(), e);
        }
    }
}
