package com.artfess.workflow.runtime.script;

import java.io.IOException;
import java.util.*;
import java.util.Map.Entry;

import javax.annotation.Resource;

import com.artfess.base.manager.CommonManager;
import com.artfess.base.util.AppUtil;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Component;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.artfess.base.feign.UCFeignService;
import com.artfess.base.groovy.IUserScript;
import com.artfess.base.util.BeanUtils;
import com.artfess.base.util.JsonUtil;
import com.artfess.base.util.StringUtil;
import com.artfess.bpm.api.cmd.ActionCmd;
import com.artfess.bpm.api.constant.BpmConstants;
import com.artfess.bpm.api.context.ContextThreadUtil;
import com.artfess.bpm.api.model.identity.BpmIdentity;
import com.artfess.bpm.model.identity.DefaultBpmIdentity;
import com.artfess.uc.api.impl.model.UserFacade;
import com.artfess.uc.api.impl.util.ContextUtil;
import com.artfess.uc.api.model.IUser;
import com.artfess.uc.api.service.IUserService;


/**
 * 根据用户关系或组织关系获取人员
 * @author zhangxw
 *
 */

@Component
public class UserRelScript implements IUserScript {
	
	@Resource
	IUserService userService;
	@Resource
	UCFeignService uCFeignService;
	
	/**
	 * 将用户列表转换成BpmIdentity列表
	 * @param list
	 * @return
	 */
	private Set<BpmIdentity> convertUserList(List<IUser> list){
		Set<BpmIdentity> identitys = new LinkedHashSet<BpmIdentity>();
		for (IUser iUser : list) {
			if(BeanUtils.isNotEmpty(iUser) && (BeanUtils.isEmpty(iUser.getStatus()) || iUser.getStatus()==1)){
				DefaultBpmIdentity bpmIdentity = new DefaultBpmIdentity();
				bpmIdentity.setId(iUser.getUserId());
				bpmIdentity.setName(iUser.getFullname());
				bpmIdentity.setType(BpmIdentity.TYPE_USER);
				identitys.add(bpmIdentity);
			}
		}
		return identitys;
	}
	
	/**
	 * 根据用户关系获取人员列表
	 * @param userId 用户ID
	 * @param typeCode 关系类型code
	 * @return
	 */
	private List<IUser> getSuperUserByRel(String userId, String typeCode){
		List<IUser> list = new ArrayList<IUser>();
		ObjectNode obj=JsonUtil.getMapper().createObjectNode();
		obj.put("userId", userId);
		obj.put("typeCode", typeCode);
		List<ObjectNode> sysUserRels = uCFeignService.getSuperUser(obj);
		if(BeanUtils.isNotEmpty(sysUserRels)){
			for (ObjectNode sysUserRel : sysUserRels) {
				IUser user = new UserFacade();
				user.setUserId(sysUserRel.get("id").asText());
				user.setFullname(sysUserRel.get("fullname").asText());
				list.add(user);
			}
		}
		return list;
	}
	
	/**
	 * json转iuser
	 * @param objs
	 * @return
	 */
	private List<IUser> usersObjConvertToIusers(List<ObjectNode> objs){
		List<IUser> list = new ArrayList<IUser>();
		if(BeanUtils.isNotEmpty(objs)){
			for (ObjectNode obj : objs) {
				UserFacade user = new UserFacade();
				user.setId(obj.get("id").asText());
				user.setAccount(obj.get("account").asText());
				user.setFullname(obj.get("fullname").asText());
				user.setUserId(obj.get("id").asText());
				user.setStatus(obj.get("status").asInt());
				list.add(user);
			}
		}
		return list;
	}
	
	/**
	 * 获取组织负责人(包含主负责人)或主负责人
	 * @param orgId  组织ID
	 * @param isMain 是否主负责人
	 * @return
	 */
	public Set<BpmIdentity> getChargesByOrgId(Object orgId,Boolean isMain){
		Set<BpmIdentity> identitys = new LinkedHashSet<BpmIdentity>();
		if(BeanUtils.isNotEmpty(orgId)){
			String obj=orgId+"";
			Object[] orgIds = obj.split(",");
			List<IUser> list = new ArrayList<IUser>();
			for (Object id : orgIds) {
				List<ObjectNode> users = uCFeignService.getChargesByOrgId(id.toString(),isMain);
				if(BeanUtils.isNotEmpty(users)) {
					list.addAll(usersObjConvertToIusers(users));
				}
			}
			if(BeanUtils.isNotEmpty(list)){
				identitys = convertUserList(list);
			}
		}
		return identitys;
	}
	
	/**
	 * 获取上级组织负责人(包含主负责人)或主负责人
	 * @param orgId  组织ID
	 * @param isMain 是否主负责人
	 * @return
	 */
	public Set<BpmIdentity> getUpChargesByOrgId(Object orgId,Boolean isMain){
		Set<BpmIdentity> identitys = new LinkedHashSet<BpmIdentity>();
		if(BeanUtils.isNotEmpty(orgId)){
			ObjectNode org =  uCFeignService.getOrgByIdOrCode(orgId.toString());
			if(BeanUtils.isNotEmpty(org)){
				identitys = getChargesByOrgId(org.get("parentId").asText(),isMain);
			}
		}
		return identitys;
	}
	
	
	/**
	 * 获取上一节点执行人的组织负责人
	 * @param isMain 是否主负责人
     * @param demCode 传维度id或编码都可以
	 * @return
	 */
	public Set<BpmIdentity> getChargesByPreNode(Boolean isMain,String demCode){
		Set<BpmIdentity> identitys = new LinkedHashSet<BpmIdentity>();
		ObjectNode orgUser = uCFeignService.getOrgUserMaster(ContextUtil.getCurrentUserId(), demCode);
		if(BeanUtils.isNotEmpty(orgUser)){
			identitys = getChargesByOrgId(orgUser.get("orgId").asText(),isMain);
		}
		return identitys;
	}
	
	/**
	 * 通过上一节点执行人获取汇报线上级 人员列表(参数选填)
	 * @param level 级别
	 * @param typeCode 类型code
	 * @return
	 */
	public Set<BpmIdentity> getByRelPreNode(String typeCode){
		if(StringUtil.isEmpty(typeCode)){
			typeCode = null;
		}
		Set<BpmIdentity> identitys = new LinkedHashSet<BpmIdentity>();
		List<IUser> list = getSuperUserByRel(ContextUtil.getCurrentUser().getUserId(), typeCode);
		identitys = convertUserList(list);
		return identitys;
	}
	
	/**
	 * 通过发起人获取汇报线上级 人员列表(参数选填)
	 * @param level 级别
	 * @param typeCode 类型ID
	 * @return
	 */
	public Set<BpmIdentity> getByRelStartUser(String typeCode){
		Set<BpmIdentity> identitys = new LinkedHashSet<BpmIdentity>();
		ActionCmd taskCmd = ContextThreadUtil.getActionCmd();
		if(BeanUtils.isNotEmpty(taskCmd)){
			String userId =  (String) taskCmd.getVariables().get(BpmConstants.START_USER);
			List<IUser> list = getSuperUserByRel(userId, typeCode);
			identitys = convertUserList(list);
		}
		return identitys;
	}
	
		
	/**
	 * 通过发起人获取汇报线上级 人员列表
	 * @param level 级别
	 * @param typeId 类型ID
	 * @return
	 */
	public Set<BpmIdentity> getByRelStartUser(){
		return getByRelStartUser(null);
	}
	
	/**
	 * 获取子表字段(人员选择器)作为节点审批人员。
	 * @param tableName	表明
	 * @param field		字段名
	 * @return
	 * @throws IOException 
	 */
	public Set<BpmIdentity> getSubFieldUser(String tableName,String field) throws IOException{
		Set<BpmIdentity> identitys = new LinkedHashSet<BpmIdentity>();
		ActionCmd cmd = ContextThreadUtil.getActionCmd();
		String json= cmd.getBusData();
		ObjectNode jsonObj=(ObjectNode) JsonUtil.toJsonNode(json);
		Iterator<Entry<String, JsonNode>> it = jsonObj.fields();
		List<IUser> userList = new ArrayList<IUser>();
		while(it.hasNext()){
			ObjectNode mainTable = (ObjectNode) it.next();
			if(BeanUtils.isNotEmpty(mainTable)){
				ArrayNode subTable =  (ArrayNode) mainTable.get("sub_"+tableName);
				for (JsonNode object : subTable) {
					ObjectNode subData = (ObjectNode) object;
					String fieldValue = subData.get(field).asText();
					if(StringUtil.isNotEmpty(fieldValue)){
						String[] ids = fieldValue.split(",");
						for (String id : ids) {
							IUser iUser = userService.getUserById(id);
							if(BeanUtils.isNotEmpty(iUser)){
								userList.add(iUser);
							}
						}
					}
				}
			}
		}
		if(BeanUtils.isNotEmpty(userList)){
			identitys.addAll(convertUserList(userList));
		}
		return identitys;
	}
	/**
	 * 获取子表字段(部门选择器，组织负责人)作为节点审批人员。
	 * @param tableName	表明
	 * @param field		字段名
	 * @return
	 * @throws IOException 
	 */
	public Set<BpmIdentity> getSubFieldOrg(String tableName,String field,boolean isMain) throws IOException{
		Set<BpmIdentity> identitys = new LinkedHashSet<BpmIdentity>();
		ActionCmd cmd = ContextThreadUtil.getActionCmd();
		String json= cmd.getBusData();
		ObjectNode jsonObj=(ObjectNode) JsonUtil.toJsonNode(json);
		Iterator<Entry<String, JsonNode>> it = jsonObj.fields();
		while(it.hasNext()){
			ObjectNode mainTable = (ObjectNode) it.next();
			if(BeanUtils.isNotEmpty(mainTable)){
				ArrayNode subTable =  (ArrayNode) mainTable.get("sub_"+tableName);
				for (Object object : subTable) {
					ObjectNode subData = (ObjectNode) object;
					String fieldValue = subData.get(field).asText();
					if(StringUtil.isNotEmpty(fieldValue)){
						String[] orgIds = fieldValue.split(",");
						for (String orgId : orgIds) {
							Set<BpmIdentity> orgIdentitys = getChargesByOrgId(orgId, isMain);
							identitys.addAll(orgIdentitys);
						}
					}
				}
			}
		}
		return identitys;
	}
	/*
	 * 自定义业务数据SQL获取人员ID，返回人员列表(会拼接busDataId)
	 * @param sql SQL语句
	 * */
		public Set<BpmIdentity> getUserFromBusSql(String sql) throws Exception{
		JdbcTemplate jdbcTemplate = AppUtil.getBean(JdbcTemplate.class);
		Set<BpmIdentity> identitys = new LinkedHashSet<BpmIdentity>();
		sql = sql.toUpperCase();
		if(sql.contains("<#ID#>")){
			ActionCmd cmd = ContextThreadUtil.getActionCmd();
			//String json= cmd.getBusData();
			HashMap boInst = (HashMap)cmd.getTransitVars().get("bo_inst_");
			//ObjectNode jsonObj=(ObjectNode) JsonUtil.toJsonNode(json);
			String boKey = (String)boInst.keySet().iterator().next();
			ObjectNode boData = (ObjectNode)boInst.get(boKey);
			String id = boData.get("id_").asText();
			//String id = jsonObj.get("id_").asText();
			sql = sql.replace("<#ID#>",id);
		}
		List<Map<String, Object>> query = jdbcTemplate.queryForList(sql);
		if(BeanUtils.isEmpty(query)){
			return identitys;
		}
		if(query.get(0).size() != 1){
			throw new RuntimeException("SQL查询列数不能超过1");
		}else{
			List<IUser> users = new ArrayList<>();
			for (Map<String, Object> stringObjectMap : query) {
				String key = stringObjectMap.keySet().iterator().next();
				String id = (String) stringObjectMap.get(key);
				if(isContainsUserId(users,id)){
					continue;
				}
				IUser user = userService.getUserById(id);
				if(BeanUtils.isNotEmpty(user)){
					users.add(user);
				}
			}
			if(BeanUtils.isNotEmpty(users)){
				identitys.addAll(convertUserList(users));
			}
			return identitys;
		}
	}

	//判断用户列表中是否含有某个用户ID
	private boolean isContainsUserId(List<IUser> users,String id){
		if(BeanUtils.isNotEmpty(users)){
			return false;
		}
		for (IUser user : users) {
			if(user.getUserId() == id){
				return true;
			}
		}
		return false;

	}
}
