package com.artfess.cqxy.contract.controller;

import com.artfess.base.annotation.ApiGroup;
import com.artfess.base.constants.ApiGroupConsts;
import com.artfess.base.controller.BaseController;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.util.BeanUtils;
import com.artfess.cqxy.completionAcceptance.model.CompletionAcceptance;
import com.artfess.cqxy.contract.manager.ContractManager;
import com.artfess.cqxy.contract.manager.FundPlanManager;
import com.artfess.cqxy.contract.model.Contract;
import com.artfess.cqxy.contract.model.FundPlan;
import com.artfess.cqxy.projectManagement.manager.ProjectManagementManager;
import com.artfess.cqxy.projectManagement.model.ProjectManagement;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.Assert;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.Arrays;

/**
 * @Description:
 * @Author: Rong Tao
 * @Date: 2022/7/7 14:48
 */

@Slf4j
@RestController
@Api(tags = "合同管理-资金计划接口")
@ApiGroup(group = {ApiGroupConsts.GROUP_BIZ})
@RequestMapping("/biz/contract/fundPlan/v1")
public class FundPlanController extends BaseController<FundPlanManager, FundPlan> {
    @Autowired
    private ProjectManagementManager pm;

    @Autowired
    private ContractManager cm;

    /**
     * 保存或修改数据
     * @param fundPlan 实体对象
     * @return 新增或更新结果
     */
    @PostMapping("/save")
    @ApiOperation(value = "S-保存或修改数据",httpMethod = "POST")
    public CommonResult insert(@RequestBody FundPlan fundPlan) {
        Assert.notNull(fundPlan.getProjectId(),"项目ID不能为空");
        boolean add = StringUtils.isEmpty(fundPlan.getId());
        if(StringUtils.isNotBlank(fundPlan.getContractId())){
            Contract contract = cm.getById(fundPlan.getContractId());
            if(null != contract){
                fundPlan.setContractName(contract.getName());
            }
        }
        baseService.saveOrUpdate(fundPlan);
        String msg = (add ? "添加" : "修改" + "成功！");
        return new CommonResult(true, msg);
    }


    /**
     * 批量导入Excel数据
     * @param file Excel文件
     */
    @PostMapping("/importExcelData")
    @ApiOperation(value = "S-批量导入Excel数据")
    public CommonResult importExcelData (@ApiParam("上传文件") @RequestParam("file") MultipartFile file, String projectId,String contractId){
        log.debug("批量导入Excel数据接口请求参数：{}",file, projectId,contractId);
        baseService.importExcelData(file, projectId,contractId);
        return new CommonResult<>("导入成功");
    }

    /**
     * 导出数据到Excel
     * @param queryFilter 通用查询器
     * @param response 响应对象
     * @return 通用结果
     * @throws IOException IO异常
     */
    @PostMapping("/exportDataToExcel")
    @ApiOperation(value = "S-导出数据到Excel",notes = "参数说明：查询器只使到了query组传参，其他都可以不要，查询条件要带上表别名：主表 bra、副表 pm；传入pm.ID_导出该项目下的数据，否则将导出所有数据")
    public CommonResult exportDataToExcel (@ApiParam(name = "queryFilter",value = "通用查询器")@RequestBody QueryFilter<FundPlan> queryFilter, HttpServletResponse response) throws IOException {
        log.debug("导出数据到Excel接口请求参数：{}",queryFilter,response);
        baseService.exportDatatoExcel(queryFilter,response);
        return new CommonResult<>("导出成功");
    }

    /**
     * 删除数据
     * @param ids 主键集合
     * @return 删除结果
     */
    @PostMapping("/delete")
    @ApiOperation(value = "S-删除数据",httpMethod = "POST")
    public CommonResult delete(@ApiParam(name = "ids",value = "ID集合") @RequestParam("ids") String ids) {
        Assert.notNull(ids,"主键集合不能为空");
        boolean state = baseService.deleteByIds(Arrays.asList(ids.split(",")));
        String msg = state ? "删除成功" : "删除失败";
        log.debug("合同管理-资金计划-删除数据接口响应结果：{}", msg);
        return new CommonResult<String>(state, msg);
    }

    /**
     * 通过主键查询单条数据
     * @param id 主键
     * @return 单条数据
     */
    @PostMapping("/detail/{id}")
    @ApiOperation(value = "S-根据ID查询详情",httpMethod = "POST")
    public CommonResult<FundPlan> selectOne(@ApiParam(name = "id",value = "主键ID") @PathVariable String id) {
        Assert.notNull(id,"主键不能为空");
        FundPlan result = baseService.getById(id);
        boolean state = !BeanUtils.isEmpty(result);
        log.debug("合同管理-资金计划-根据ID查询详情 接口响应结果：{}", result);
        return new CommonResult<FundPlan>(state, state?"操作成功":"该ID没有对应数据", result);
    }

    /**
     * 分页查询所有数据
     * @param queryFilter 通用查询器
     * @return 分页数据
     */
    @PostMapping("/page")
    @ApiOperation(value = "S-分页查询所有数据",notes = "参数说明：在通用查询器中的query对象组传入查询参数，传入 PROJECT_ID_ 项目ID 以切换项目，所有参数都需要带上表别名——主表为bra、副表为pm",httpMethod = "POST")
    public CommonResult<PageList<FundPlan>> selectAll(@ApiParam(name = "queryFilter",value = "通用查询器")@RequestBody QueryFilter<FundPlan> queryFilter) {
        PageList<FundPlan> result = baseService.queryAllByPage(queryFilter);
        log.debug("合同管理-资金计划-分页查询所有数据接口响应结果：{}", result);
        return new CommonResult<>(true,"操作成功",result);
    }

    /**
     * 导出本月数据到Excel
     * @param queryFilter 通用查询器
     * @param response 响应对象
     * @return 通用结果
     * @throws IOException IO异常
     */
    @PostMapping("/exportDataToExcel2")
    @ApiOperation(value = "S-导出本月数据到Excel",notes = "参数说明：查询器只使到了query组传参，其他都可以不要，查询条件要带上表别名：主表 bra、副表 pm；传入pm.ID_导出该项目下的数据，否则将导出所有数据")
    public CommonResult exportDataToExcel2 (@ApiParam(name = "queryFilter",value = "通用查询器")@RequestBody QueryFilter<FundPlan> queryFilter, HttpServletResponse response) throws IOException {
        log.debug("导出本月数据到Excel接口请求参数：{}",queryFilter,response);
        baseService.exportDatatoExcel2(queryFilter,response);
        return new CommonResult<>("导出成功");
    }
}
