package com.artfess.application.persistence.manager.impl;

import com.artfess.base.exception.BaseException;
import com.artfess.base.feign.UCFeignService;
import com.artfess.base.manager.impl.BaseManagerImpl;
import com.artfess.base.model.CommonResult;
import com.artfess.base.util.BeanUtils;
import com.artfess.base.util.FileUtil;
import com.artfess.base.util.JAXBUtil;
import com.artfess.base.util.JsonUtil;
import com.artfess.base.util.StringUtil;
import com.artfess.base.util.ThreadMsgUtil;
import com.artfess.base.util.UniqueIdUtil;
import com.artfess.base.util.ZipUtil;
import com.artfess.application.model.AppModel;
import com.artfess.application.model.AppModelList;
import com.artfess.application.persistence.dao.AppModelDao;
import com.artfess.application.persistence.manager.AppModelManager;
import com.artfess.sysConfig.persistence.dao.SysAuthUserDao;
import com.artfess.sysConfig.persistence.manager.SysAuthUserManager;
import com.artfess.sysConfig.persistence.manager.SysTypeManager;
import com.artfess.sysConfig.persistence.model.SysAuthUser;
import com.artfess.sysConfig.persistence.model.SysType;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import java.io.File;
import java.nio.file.Paths;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 
 * <pre> 
 * 描述：移动端应用 处理实现类
 * 构建组：x7
 * 作者:pangq
 * 邮箱:pangq@jee-soft.cn
 * 日期:2020-07-06 15:23:30
 * 版权：广州宏天软件股份有限公司
 * </pre>
 */
@Service("appModelManager")
public class AppModelManagerImpl extends BaseManagerImpl<AppModelDao, AppModel> implements AppModelManager{
	@Resource
	SysTypeManager sysTypeManager;
	@Resource
	SysAuthUserManager sysAuthUserManager;
	@Resource
	SysAuthUserDao sysAuthUserDao;
	@Resource
	UCFeignService ucFeignService;

	@Override
	public List<AppModel> getByTypeId(String typeId) {
		QueryWrapper<AppModel> queryWrapper = new QueryWrapper<AppModel>();
		queryWrapper.and(i -> i.eq("TYPE_ID_", typeId).eq("ENABLE_", 1));
		queryWrapper.orderByAsc("SN_");
		queryWrapper.orderByDesc("CREATE_TIME_");
		return this.list(queryWrapper);
	}

	@Override
	public Map<String, String> exportData(List<String> ids) throws Exception {
		Map<String, String> map = new HashMap<>();
		AppModelList appModelList = new AppModelList();
		for (String id : ids) {
			AppModel appModel = this.get(id);
			SysType sysType = sysTypeManager.get(appModel.getTypeId());
			if (sysType == null){
				sysType = sysTypeManager.getByTypeKeyAndGroupKey("appType","mrfl");
				if (sysType == null){
					sysType = sysTypeManager.getByTypeKeyAndGroupKey("appType","default");
				}
			}
			appModel.setTypeKey(sysType.getTypeKey());
			Map<String,Object> condition = new HashMap<>();
			condition.put("authorize_id_", id);
			List<SysAuthUser> sysAuthUserList = sysAuthUserDao.selectByMap(condition);
			for (SysAuthUser sysAuthUser : sysAuthUserList){
				String rightType = sysAuthUser.getRightType();
				String ownerId = sysAuthUser.getOwnerId();
				if ("user".equals(rightType)){
					CommonResult<JsonNode> result = ucFeignService.getUserById(ownerId);
					if (BeanUtils.isNotEmpty(result)){
						JsonNode userNode = result.getValue();
						String account = JsonUtil.getString(userNode, "account");
						sysAuthUser.setOwnerCode(account);
					}
				}else if ("org".equals(rightType)){
					ObjectNode orgObjNode = ucFeignService.getOrgByIdOrCode(ownerId);
					if (BeanUtils.isNotEmpty(orgObjNode)){
						String code = orgObjNode.get("code").asText();
						sysAuthUser.setOwnerCode(code);
					}
				}else if ("pos".equals(rightType)){
					CommonResult<ObjectNode> result = ucFeignService.getPostByIdOrCode(ownerId);
					if (BeanUtils.isNotEmpty(result)){
						ObjectNode posObjNode = result.getValue();
						String code = posObjNode.get("code").asText();
						sysAuthUser.setOwnerCode(code);
					}
				}else if ("role".equals(rightType)){
					CommonResult<ObjectNode> result = ucFeignService.getRoleByIdOrCode(ownerId);
					if (BeanUtils.isNotEmpty(result)){
						ObjectNode roleObjNode = result.getValue();
						String code = roleObjNode.get("code").asText();
						sysAuthUser.setOwnerCode(code);
					}
				}
			}
			appModel.setSysAuthUserList(sysAuthUserList);
			appModelList.addAppModel(appModel);
		}
		try {
			String xml = JAXBUtil.marshall(appModelList, AppModelList.class);
			map.put("appModels.portal.xml", xml);
		} catch (Exception e) {
			e.printStackTrace();
			throw new RuntimeException("导出失败！" + e.getMessage(), e);
		}
		return map;
	}

	@Override
	@Transactional
	public void importData(MultipartFile file) {
		String unZipFilePath = null;
		try {
			String rootRealPath = Paths.get(FileUtil.getIoTmpdir(), "attachFiles/unZip").toString();
			FileUtil.createFolder(rootRealPath, true);
			ZipUtil.unZipFile(file, rootRealPath);
			String name = file.getOriginalFilename();
			String fileDir = StringUtil.substringBeforeLast(name, ".");
			unZipFilePath = Paths.get(rootRealPath, fileDir).toString();
			String xml = FileUtil.readFile(Paths.get(unZipFilePath, "appModels.portal.xml").toString());
			if (StringUtil.isEmpty(xml)) {
				throw new BaseException("导入的xml文件没有内容！");
			}
			AppModelList list = (AppModelList) JAXBUtil.unmarshall(xml, AppModelList.class);
			List<AppModel> appModelList = list.getAppModelList();
			for (AppModel appModel : appModelList) {
				importData(appModel);
			}
		} catch (Exception e) {
			throw new BaseException(e.getMessage(), e);
		} finally {
			if (StringUtil.isNotEmpty(unZipFilePath)) {
				File formDir = new File(unZipFilePath);
				if (formDir.exists()) {
					FileUtil.deleteDir(formDir);
				}
			}
		}
	}

	private void importData(AppModel appModel) {
		QueryWrapper wrapper = new QueryWrapper();
		wrapper.eq("name_", appModel.getName());
		List<AppModel> appModelList = baseMapper.selectList(wrapper);
		if (appModelList != null && !appModelList.isEmpty()) {
			ThreadMsgUtil.addMapMsg2("appModelsSkipped", String.format("<div style='margin-left:12px; margin-top:6px;'>%s</div>", appModel.getName() + ""));
			return;
		}
		String appId = UniqueIdUtil.getSuid();
		appModel.setId(appId);
		SysType sysType = sysTypeManager.getByTypeKeyAndGroupKey("appType", appModel.getTypeKey());
		if (sysType == null){
			sysType = sysTypeManager.getByTypeKeyAndGroupKey("appType","mrfl");
			if (sysType == null){
				sysType = sysTypeManager.getByTypeKeyAndGroupKey("appType","default");
			}
		}
		appModel.setTypeId(sysType.getId());
		appModel.setTypeName(sysType.getName());
		List<SysAuthUser> sysAuthUserList = appModel.getSysAuthUserList();
		if (BeanUtils.isEmpty(sysAuthUserList)){
			sysAuthUserManager.createDefault(appId);
		}else{
			for (SysAuthUser sysAuthUser : sysAuthUserList){
				String authId = UniqueIdUtil.getSuid();
				sysAuthUser.setId(authId);
				sysAuthUser.setAuthorizeId(appId);
				String rightType = sysAuthUser.getRightType();
				String ownerCode = sysAuthUser.getOwnerCode();
				if ("user".equals(rightType)){
					JsonNode userNode = ucFeignService.getUser(ownerCode, null);
					if (BeanUtils.isNotEmpty(userNode)){
						String ownerId = JsonUtil.getString(userNode, "id");
						String ownerName = JsonUtil.getString(userNode, "fullname");
						sysAuthUser.setOwnerId(ownerId);
						sysAuthUser.setOwnerName(ownerName);
					}
				}else if ("org".equals(rightType)){
					ObjectNode orgObjNode = ucFeignService.getOrgByIdOrCode(ownerCode);
					if (BeanUtils.isNotEmpty(orgObjNode)){
						String ownerId = orgObjNode.get("id").asText();
						String ownerName = orgObjNode.get("name").asText();
						sysAuthUser.setOwnerId(ownerId);
						sysAuthUser.setOwnerName(ownerName);
					}
				}else if ("pos".equals(rightType)){
					CommonResult<ObjectNode> result = ucFeignService.getPostByIdOrCode(ownerCode);
					if (BeanUtils.isNotEmpty(result)){
						ObjectNode posObjNode = result.getValue();
						String ownerId = posObjNode.get("id").asText();
						String ownerName = posObjNode.get("name").asText();
						sysAuthUser.setOwnerId(ownerId);
						sysAuthUser.setOwnerName(ownerName);
					}
				}else if ("role".equals(rightType)){
					CommonResult<ObjectNode> result = ucFeignService.getRoleByIdOrCode(ownerCode);
					if (BeanUtils.isNotEmpty(result)){
						ObjectNode roleObjNode = result.getValue();
						String ownerId = roleObjNode.get("id").asText();
						String ownerName = roleObjNode.get("name").asText();
						sysAuthUser.setOwnerId(ownerId);
						sysAuthUser.setOwnerName(ownerName);
					}
				}
				sysAuthUserDao.insert(sysAuthUser);
			}
		}
		this.create(appModel);
		ThreadMsgUtil.addMapMsg2("appModelsSaved", String.format("<div style='margin-left:12px; margin-top:6px;'>%s</div>", appModel.getName() + ""));
	}
}
