'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

/**
 * 添加事件监听器
 * @param target 对象
 * @param eventName 事件名称
 * @param handler 回调函数
 * @param opitons
 */
function on(target, eventName, handler, opitons = false) {
    target.addEventListener(eventName, handler, opitons);
}

function uuid4() {
    if (crypto && crypto.randomUUID) {
        return crypto.randomUUID().replace(/-/g, '');
    }
    const getRandomByte = () => {
        if (crypto && crypto.getRandomValues) {
            return crypto.getRandomValues(new Uint8Array(1))[0];
        }
        return Math.random() * 16;
    };
    return ([1e7] + 1e3 + 4e3 + 8e3 + 1e11).replace(/[018]/g, c => (c ^ ((getRandomByte() & 15) >> (c / 4))).toString(16));
}
/**
 * 深拷贝
 * 兼容函数，对象，相互引用场景
 * @param target 需要深拷贝的原对象
 * @return 深拷贝后的对象
 */
function deepCopy(target, map = new Map()) {
    if (target !== null && typeof target === 'object') {
        let res = map.get(target);
        if (res) {
            return res;
        }
        if (target instanceof Array) {
            res = [];
            map.set(target, res);
            target.forEach((item, index) => {
                res[index] = deepCopy(item, map);
            });
        }
        else {
            res = {};
            map.set(target, res);
            Object.keys(target).forEach(key => {
                // @ts-ignore
                res[key] = deepCopy(target[key], map);
            });
        }
        return res;
    }
    return target;
}

const SESSIONID_KEY = '__webtracker_sessionid';
function getSessionId() {
    const sessionIdFromLocalStorage = localStorage.getItem(SESSIONID_KEY);
    if (sessionIdFromLocalStorage) {
        return sessionIdFromLocalStorage;
    }
    const uuid = uuid4();
    localStorage.setItem(SESSIONID_KEY, uuid);
    return uuid;
}

let globalOptions;
function initOptions(options) {
    if (!options) {
        throw new Error('options 不能为空');
    }
    globalOptions = deepCopy(options);
    initSessionOptions();
    initPvOptions();
    if (!globalOptions.beforeReport) {
        globalOptions.beforeReport = () => undefined;
    }
}
function initSessionOptions() {
    const { sessionId } = globalOptions;
    if (typeof sessionId === 'function') {
        // sessionId为function时直接调用
        globalOptions.sessionId = sessionId();
    }
    if (!globalOptions.sessionId) {
        // 默认生成一个uuid为sessionId
        globalOptions.sessionId = getSessionId();
    }
}
function initPvOptions() {
    // 默认开启路由监测
    let route = true;
    // 默认周期性上报为30s一次
    let interval = 30000;
    const opts = globalOptions.pv;
    if (opts === false) {
        // globalOptions.pv 显式指定为false的情况下，不上报pv信息
        route = false;
        interval = 0;
    }
    else if (opts && typeof opts === 'object') {
        // 以对象方式传入配置项时， route或者interval不传视为默认
        if (opts.hasOwnProperty('route')) {
            route = opts.route;
        }
        if (opts.hasOwnProperty('interval')) {
            interval = opts.interval;
        }
    }
    globalOptions.pv = { route, interval };
}

// 上报类型
exports.Type = void 0;
(function (Type) {
    Type["PV"] = "PV";
})(exports.Type || (exports.Type = {}));

/******************************************************************************
Copyright (c) Microsoft Corporation.

Permission to use, copy, modify, and/or distribute this software for any
purpose with or without fee is hereby granted.

THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES WITH
REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT,
INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
PERFORMANCE OF THIS SOFTWARE.
***************************************************************************** */
/* global Reflect, Promise, SuppressedError, Symbol */


function __awaiter(thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
}

typeof SuppressedError === "function" ? SuppressedError : function (error, suppressed, message) {
    var e = new Error(message);
    return e.name = "SuppressedError", e.error = error, e.suppressed = suppressed, e;
};

function report(type, data) {
    return __awaiter(this, void 0, void 0, function* () {
        const { extData, beforeReport } = globalOptions;
        const reportData = {
            type,
            base: getBaseInfo(),
            data,
            extData: extData === null || extData === void 0 ? void 0 : extData()
        };
        if (beforeReport) {
            const res = yield beforeReport(reportData, globalOptions);
            if (res === false) {
                // beforeReport显式返回false或者Promise<false>会取消发送
                return;
            }
        }
        reportToServer([reportData]);
    });
}
function getBaseInfo() {
    const { sessionId, appCode, appName, appVersion, userId } = globalOptions;
    const getUserId = typeof userId === 'function' ? userId : () => userId;
    console.log('typeof userId', typeof userId);
    return {
        sessionId: sessionId,
        appCode,
        appName: appName,
        appVersion: appVersion,
        userId: getUserId() || '',
        url: location.href,
        title: document.title,
        t: Date.now()
    };
}
function reportToServer(data) {
    const { reportUrl } = globalOptions;
    console.log('ready report to server, url:', reportUrl, 'reportData: ', data);
    if ('sendBeacon' in window.navigator) {
        sendByBeacon(reportUrl, data);
    }
    else {
        sendByImage(reportUrl, data);
    }
}
function sendByBeacon(url, data) {
    const formData = new FormData();
    formData.append('v', JSON.stringify(data));
    navigator.sendBeacon(url, formData);
}
function sendByImage(url, data) {
    const img = new Image();
    img.src = `${url}?v=${encodeURIComponent(JSON.stringify(data))}`;
}

/**
 * 页面跳转行为收集
 */
var EventType;
(function (EventType) {
    EventType["LOAD"] = "LOAD";
    EventType["BEFORE_UNLOAD"] = "BEFORE_UNLOAD";
    EventType["HASH_CHANGE"] = "HASH_CHANGE";
    EventType["POP_STATE"] = "POP_STATE";
    EventType["PUSH_STATE"] = "PUSH_STATE";
    EventType["REPLACE_STATE"] = "REPLACE_STATE";
    EventType["PERIOD_PV"] = "PERIOD_PV";
})(EventType || (EventType = {}));
let options;
function initPv() {
    options = globalOptions.pv;
    if (options.route) {
        listenLoad();
        listenHash();
        listenHistory();
    }
    if (options.interval > 0) {
        listenPageView();
    }
}
function listenLoad() {
    reportPV({ eventType: EventType.LOAD });
    on(window, 'load', () => {
        reportPV({ eventType: EventType.LOAD });
    }, false);
    on(window, 'beforeunload', (evt) => {
        console.log('beforeunload', evt);
        reportPV({ eventType: EventType.BEFORE_UNLOAD });
    }, false);
}
function listenHash() {
    on(window, 'hashchange', (evt) => {
        console.log('hashchange', evt);
        reportPV({ eventType: EventType.HASH_CHANGE, oldUrl: evt.oldURL, newUrl: evt.newURL });
    }, false);
}
function listenHistory() {
    if (!window.history.pushState) {
        return;
    }
    on(window, 'popstate', (evt) => {
        console.log('popstate', evt);
        reportPV({ eventType: EventType.POP_STATE, state: evt.state });
    });
    const originPushState = window.history.pushState;
    window.history.pushState = function (...args) {
        console.log('pushState', ...args);
        reportPV({ eventType: EventType.PUSH_STATE, data: args[0] });
        originPushState.call(this, ...args);
    };
    const originReplaceState = window.history.replaceState;
    window.history.replaceState = function (...args) {
        console.log('replaceState', ...args);
        reportPV({ eventType: EventType.REPLACE_STATE, data: args[0] });
        originReplaceState.call(this, ...args);
    };
}
function listenPageView() {
    setInterval(() => {
        reportPV({ eventType: EventType.PERIOD_PV });
    }, options.interval);
}
function reportPV(data) {
    setTimeout(() => {
        report(exports.Type.PV, Object.assign(Object.assign({}, data), { referrer: document.referrer }));
    }, 17);
}

function init(options) {
    initOptions(options);
    initPv();
}

function install(Vue, options) {
    console.log('webtracker vue init', options);
    init(options);
}
var index = {
    install
};

exports.default = index;
exports.init = init;
//# sourceMappingURL=index.cjs.js.map
