package com.artfess.cgpt.material.controller;


import com.artfess.base.controller.BaseController;
import com.artfess.base.enums.ResponseErrorEnums;
import com.artfess.base.exception.BaseException;
import com.artfess.base.model.CommonResult;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.cgpt.material.manager.MaterialHierarchicalManagementManager;
import com.artfess.cgpt.material.manager.MaterialHierarchicalManagementRangeManager;
import com.artfess.cgpt.material.model.MaterialHierarchicalManagement;
import com.artfess.cgpt.material.model.MaterialHierarchicalManagementRange;
import com.artfess.cgpt.material.vo.MaterialHierarchicalManagementQueryVo;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;

/**
 * 物料分级管理表 前端控制器
 *
 * @company artfess.com
 * @author baseli
 * @since 2024-08-15
 */
@RestController
@RequestMapping("/material/bizMaterialHierarchicalManagement/v1/")
public class MaterialHierarchicalManagementController extends BaseController<MaterialHierarchicalManagementManager, MaterialHierarchicalManagement> {

    @Resource
    private MaterialHierarchicalManagementRangeManager materialHierarchicalManagementRangeManager;

    @PostMapping("/")
    @ApiOperation("添加数据的接口")
    public CommonResult<String> create(@ApiParam(name="model", value="数据信息") @RequestBody MaterialHierarchicalManagement t) {

        if(baseService.countByMatId(t) > 0){
            throw new BaseException("物料以存在，请勿重复添加！");
        }
        try {
            boolean result = baseService.save(t);
            List<MaterialHierarchicalManagementRange> materialHierarchicalManagementRangelList = t.getMaterialHierarchicalManagementRangelList();
            if(null != materialHierarchicalManagementRangelList && materialHierarchicalManagementRangelList.size() > 0){
                materialHierarchicalManagementRangelList.forEach(obj -> obj.setMaterialHierarchicalId(t.getId()));
                materialHierarchicalManagementRangeManager.saveBatch(materialHierarchicalManagementRangelList);
            }
        }catch (Exception e){
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, e.getMessage());
        }
        return new CommonResult<>(true, "数据操作成功！");
    }

    @PostMapping("/batch")
    @ApiOperation("批量添加数据的接口")
    public CommonResult<String> createBatch(@ApiParam(name="model", value="数据信息") @RequestBody List<MaterialHierarchicalManagement> list) {
        try {
            // 直接去除已经添加过的
            Iterator<MaterialHierarchicalManagement> iter = list.iterator();
            while(iter.hasNext()){
                MaterialHierarchicalManagement t = iter.next();
                if(baseService.countByMatId(t) > 0){
                    iter.remove();
                }
            }
            boolean result = baseService.saveBatch(list);
            List<MaterialHierarchicalManagementRange> saveMaterialHierarchicalManagementRangeList = new ArrayList<>();
            for(MaterialHierarchicalManagement t : list){
                List<MaterialHierarchicalManagementRange> materialHierarchicalManagementRangelList = t.getMaterialHierarchicalManagementRangelList();
                if(null != materialHierarchicalManagementRangelList && materialHierarchicalManagementRangelList.size() > 0){
                    materialHierarchicalManagementRangelList.forEach(obj -> obj.setMaterialHierarchicalId(t.getId()));
                    saveMaterialHierarchicalManagementRangeList.addAll(materialHierarchicalManagementRangelList);
                }
            }
            if(saveMaterialHierarchicalManagementRangeList.size() > 0){
                materialHierarchicalManagementRangeManager.saveBatch(saveMaterialHierarchicalManagementRangeList);
            }
        }catch (Exception e){
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, e.getMessage());
        }
        return new CommonResult<>(true, "数据操作成功！");
    }

    @PostMapping(value="/queryAllByPage", produces={"application/json; charset=utf-8" })
    @ApiOperation("分页查询分级物料数据")
    public PageList<MaterialHierarchicalManagementQueryVo> queryAllByPage(@ApiParam(name="queryFilter", value="分页查询信息") @RequestBody QueryFilter<MaterialHierarchicalManagement> queryFilter) {
        return baseService.queryAllByPage(queryFilter);
    }

    @PostMapping(value="/queryAllByParentId", produces={"application/json; charset=utf-8" })
    @ApiOperation("不分页查询下级物料数据")
    public List<MaterialHierarchicalManagementQueryVo> queryAllByParentId(@ApiParam(name="parentId", value="不分页查询下级物料数据") @RequestParam String parentId) {
        return baseService.queryAllByParentId(parentId);
    }

    @GetMapping("/{id}")
    @ApiOperation("根据id查询数据")
    public MaterialHierarchicalManagement getById(@ApiParam(name="id", value="数据id") @PathVariable String id) {
        MaterialHierarchicalManagement t = baseService.getById(id);
        if(null != t){
            t.setMaterialHierarchicalManagementRangelList(materialHierarchicalManagementRangeManager.queryListByMaterialHierarchicalId(id));
        }
        return t;
    }

    @GetMapping("/getParentById")
    @ApiOperation("根据id递归查询所有父级数据集合")
    public List<MaterialHierarchicalManagement> getParentById(@ApiParam(name="id", value="数据集合") @RequestParam String id) {
        // 根据id递归查询所有父级ID集（包括自己），可以用于排除物料信息
        List<MaterialHierarchicalManagement> list = baseService.getParentById(id);
        return list;
    }

    @PutMapping("/")
    @ApiOperation("更新数据")
    public CommonResult<String> updateById(@ApiParam(name="model", value="数据信息") @RequestBody MaterialHierarchicalManagement t) {
        try {
            boolean result = baseService.updateById(t);
            materialHierarchicalManagementRangeManager.deleteByMaterialHierarchicalId(t.getId());
            List<MaterialHierarchicalManagementRange> materialHierarchicalManagementRangelList = t.getMaterialHierarchicalManagementRangelList();
            if(null != materialHierarchicalManagementRangelList && materialHierarchicalManagementRangelList.size() > 0){
                materialHierarchicalManagementRangelList.forEach(obj -> obj.setMaterialHierarchicalId(t.getId()));
                materialHierarchicalManagementRangeManager.saveBatch(materialHierarchicalManagementRangelList);
            }
        }catch (Exception e){
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, e.getMessage());
        }
        return new CommonResult<>(true, "数据操作成功！");
    }

    @DeleteMapping("/{id}")
    @ApiOperation("根据id删除")
    public CommonResult<String> deleteById(@ApiParam(name="id", value="数据id") @PathVariable String id) {
        boolean result = baseService.removeById(id);
        materialHierarchicalManagementRangeManager.deleteByMaterialHierarchicalId(id);
        if(!result) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, "删除数据失败");
        }
        return new CommonResult<>(true, "数据操作成功！");
    }

    @DeleteMapping("/")
    @ApiOperation("根据id集合批量删除")
    public CommonResult<String> deleteByIds(@ApiParam(name="ids", value="数据集合") @RequestParam String...ids) {
        boolean result = baseService.removeByIds(Arrays.asList(ids));
        materialHierarchicalManagementRangeManager.deleteByMaterialHierarchicalIds(Arrays.asList(ids));
        if(!result) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, "删除数据失败");
        }
        return new CommonResult<>(true, "数据操作成功！");
    }

    @DeleteMapping("/deleteByIdList")
    @ApiOperation("根据id集合批量删除")
    public CommonResult<String> deleteByIdList(@ApiParam(name="ids", value="数据集合") @RequestParam List<String> ids) {
        boolean result = baseService.removeByIds(ids);
        materialHierarchicalManagementRangeManager.deleteByMaterialHierarchicalIds(ids);
        if(!result) {
            return new CommonResult<>(ResponseErrorEnums.FAIL_OPTION, "删除数据失败");
        }
        return new CommonResult<>(true, "数据操作成功！");
    }

}
