package com.artfess.cgpt.bidding.manager.impl;

import com.artfess.base.exception.BaseException;
import com.artfess.base.manager.impl.BaseManagerImpl;
import com.artfess.base.query.PageList;
import com.artfess.base.query.QueryFilter;
import com.artfess.base.util.BeanUtils;
import com.artfess.cgpt.bidding.dao.BidDocumentsDao;
import com.artfess.cgpt.bidding.manager.BidDocumentsManager;
import com.artfess.cgpt.bidding.model.BidDocuments;
import com.artfess.cgpt.contract.model.BizContract;
import com.artfess.cgpt.contract.model.ContractItem;
import com.artfess.cgpt.universal.model.Accessory;
import com.artfess.cgpt.universal.service.AccessoryService;
import com.artfess.cgpt.utils.BizUtils;
import com.artfess.sysConfig.persistence.param.DictModel;
import com.artfess.uc.api.impl.util.ContextUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import net.qiyuesuo.sdk.SDKClient;
import net.qiyuesuo.sdk.api.CategoryService;
import net.qiyuesuo.sdk.api.ContractService;
import net.qiyuesuo.sdk.bean.company.TenantType;
import net.qiyuesuo.sdk.bean.contract.CreateContractRequest;
import net.qiyuesuo.sdk.bean.contract.SendContractRequest;
import net.qiyuesuo.sdk.bean.contract.ViewUrlRequest;
import net.qiyuesuo.sdk.bean.document.CreateDocumentRequest;
import net.qiyuesuo.sdk.bean.document.CreateDocumentResult;
import net.qiyuesuo.sdk.bean.sign.Signatory;
import net.qiyuesuo.sdk.common.exception.PrivateAppException;
import net.qiyuesuo.sdk.common.http.StreamFile;
import net.qiyuesuo.sdk.common.utils.TimeUtils;
import net.qiyuesuo.sdk.impl.CategoryServiceImpl;
import net.qiyuesuo.sdk.impl.ContractServiceImpl;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.io.*;
import java.math.BigDecimal;
import java.net.URL;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.stream.Collectors;

import static org.apache.logging.log4j.spi.ThreadContextMapFactory.init;

/**
 * @Description:
 * @Author: Rong Tao
 * @Date: 2024/6/17 11:36
 */
@Service
public class BidDocumentsManagerImpl extends BaseManagerImpl<BidDocumentsDao, BidDocuments> implements BidDocumentsManager {

    private final DateTimeFormatter DATE_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd");

    @Autowired
    private AccessoryService accessoryService;

    @Value("${contractSys.serverUrl}")
    private String serverUrl;
    @Value("${contractSys.accessKey}")
    private String accessKey;
    @Value("${contractSys.accessSecret}")
    private String accessSecret;
    @Value("${contractSys.onePartyStreamId}")
    private Long bidDocSignStreamId;
    private CategoryService categoryService;
    private ContractService contractService;

    @Override
    public PageList<BidDocuments> queryAllByPage(QueryFilter<BidDocuments> queryFilter) {
        IPage<BidDocuments> result = baseMapper.queryAllByPage(convert2IPage(queryFilter.getPageBean()),
                convert2Wrapper(queryFilter, currentModelClass()));

        List<BidDocuments> records = result.getRecords();
        if(BeanUtils.isNotEmpty(records) && records.size()>0){
            records.stream().forEach(s->s.setAccessoryList(accessoryService.getAccessoryBySourceId(s.getId())));
        }

        return new PageList<>(result);
    }

    @Override
    @Transactional
    public void saveOrUpdateEntity(BidDocuments entity) {
        List<Accessory> accessoryList = entity.getAccessoryList();
        entity.setSupplierId(ContextUtil.getCurrentOrgId());
        entity.setUploadTime(LocalDateTime.now());
        if(BeanUtils.isEmpty(entity.getBidType())){
            entity.setBidType(2);
        }

        //根据供应商和立项轮数查询是否存在数据
        LambdaQueryWrapper<BidDocuments> bidDocumentsQW = new LambdaQueryWrapper<>();
        bidDocumentsQW.eq(BidDocuments::getProjectApprovalType,entity.getProjectApprovalType())
                .eq(BidDocuments::getProjectApprovalId,entity.getProjectApprovalId())
                .eq(BidDocuments::getSupplierId,entity.getSupplierId())
                .eq(BidDocuments::getQuotationRounds,entity.getQuotationRounds())
                .orderByDesc(BidDocuments::getUploadTime)
                .last("limit 1");
        BidDocuments history = baseMapper.selectOne(bidDocumentsQW);

        if(BeanUtils.isEmpty(entity.getId())){
            if(BeanUtils.isNotEmpty(history)){
                entity.setId(history.getId());
            }
        }

        if(entity.getProjectApprovalType()==2 && BeanUtils.isNotEmpty(history)){
            //项目，如果上传过投标文件，判断投标文件类型是否一致
            if(entity.getBidType()!=history.getBidType()){
                List<Accessory> hisFile = accessoryService.getAccessoryBySourceId(history.getId());
                if(BeanUtils.isNotEmpty(hisFile) && hisFile.size()>0){
                    //如果已经上传附件
                    throw new BaseException("上传失败，只能选择一种投标文件类型上传");
                }
            }
        }

        super.saveOrUpdate(entity);

//        accessoryService.removeBySourceId(entity.getId());

        if(BeanUtils.isNotEmpty(accessoryList) && accessoryList.size()>0){
            accessoryList.stream().forEach(s-> {
                s.setSourceId(entity.getId());
                if(BeanUtils.isNotEmpty(entity.getBidType())){
                    s.setBidType(entity.getBidType());
                    if(s.getBidType()==1){
                        s.setBidSignStatus(1);
                    }else {
                        s.setBidSignStatus(3);
                    }
                }

            });
            accessoryService.saveBatch(accessoryList);
        }

    }

    @Override
    public Integer getAllSignStatus(BidDocuments entity) {
        //根据供应商和立项轮数查询是否存在数据
        LambdaQueryWrapper<BidDocuments> bidDocumentsQW = new LambdaQueryWrapper<>();
        bidDocumentsQW.eq(BidDocuments::getProjectApprovalType,entity.getProjectApprovalType())
                .eq(BidDocuments::getProjectApprovalId,entity.getProjectApprovalId())
                .eq(BidDocuments::getSupplierId,entity.getSupplierId())
                .eq(BidDocuments::getQuotationRounds,entity.getQuotationRounds())
                .orderByDesc(BidDocuments::getUploadTime)
                .last("limit 1");
        BidDocuments data = baseMapper.selectOne(bidDocumentsQW);
        if(BeanUtils.isNotEmpty(data)){
            if(BeanUtils.isNotEmpty(data.getBidType())){
                List<Accessory> fileList = accessoryService.getAccessoryBySourceId(data.getId());
                if(BeanUtils.isNotEmpty(fileList) && fileList.size()>0){
                    List<Accessory> collect = fileList.stream().filter(s -> s.getBidSignStatus() != 3).collect(Collectors.toList());
                    if(BeanUtils.isEmpty(collect) || collect.size()<=0){
                        return 1;
                    }
                }

            }

        }

        return 0;
    }

    @Override
    @Transactional
    public void removeByIds(String ids) {
        List<String> idList = Arrays.asList(ids.split(","));
        super.removeByIds(idList);
        accessoryService.removeBySourceIds(idList);
    }

    public void init() {
        SDKClient sdkClient = new SDKClient(serverUrl, accessKey, accessSecret);

        contractService = new ContractServiceImpl(sdkClient);
        categoryService = new CategoryServiceImpl(sdkClient);
    }

    @Override
    public String sendDocSign(String fileId) throws Exception {
        Accessory data = accessoryService.getById(fileId);

        if(BeanUtils.isEmpty(data)){
            throw new BaseException("请刷新后重试");
        }

        if(data.getBidType()!=1){
            throw new BaseException("发起失败，当前投标文件类型无法发起");
        }

        if(data.getBidSignStatus()!=1){
            throw new BaseException("发起失败：未签署状态下才可以发起签署");
        }

        //初始化
        init();

        Long contrStreamId = bidDocSignStreamId;

        List<Long> documentIdList = new ArrayList<>();

        //获取合同附件
        String url = data.getUrl();
        //对本地文件命名
//                String fileName = url.substring(url.lastIndexOf("."),url.length());
        File file = null;
        URL urlfile;
        InputStream inStream = null;
        OutputStream os = null;
        try {
            file = File.createTempFile("net_url", data.getSuffix());
            //下载
            urlfile = new URL(url);
            inStream = urlfile.openStream();
            os = new FileOutputStream(file);
            int bytesRead = 0;
            byte[] buffer = new byte[8192];
            while ((bytesRead = inStream.read(buffer, 0, 8192)) != -1) {
                os.write(buffer, 0, bytesRead);
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            try {
                if (null != os) {
                    os.close();
                }
                if (null != inStream) {
                    inStream.close();
                }

            } catch (Exception e) {
                e.printStackTrace();
            }
        }

        String contrTitle = BeanUtils.isNotEmpty(data.getName()) ? data.getName() : ContextUtil.getCurrentOrgName() + "投标文件";

        //为附件创建合同文档
        CreateDocumentRequest request = new CreateDocumentRequest();
        request.setFile(new StreamFile(new FileInputStream(file)));
//          request.setFile(new StreamFile(new FileInputStream(new File("E:\\OneDrive\\桌面\\artfess\\cs.docx"))));
        request.setFileType(data.getSuffix());
        request.setTitle(contrTitle);
        //  request.setTitle("其他文档");
        CreateDocumentResult result = contractService.createByFile(request);
        Long documentId = result.getDocumentId();
        documentIdList.add(documentId);

        Map<String, String> params = new HashMap<>();
        // 设置参数：
        CreateContractRequest createContractRequest = new CreateContractRequest();
        //合同终止时间；格式：yyyy-MM-dd 。时分秒默认为当天的23时59分59秒（不支持修改）。
        //默认合同结束日期
        createContractRequest.setEndTime(TimeUtils.format(TimeUtils.after(new Date(), 7), TimeUtils.STANDARD_PATTERN));
//        createContractRequest.setEndTime(DATE_FORMATTER.format(localContract.getContractEndDate()));
        //是否立即发起合同，默认true。（true：立即发起；false：保存为草稿）
        createContractRequest.setSend(true);
        //允许指定位置签署，默认使用业务分类中的「签署位置规则」配置
        createContractRequest.setMustSign(true);
        //是否允许在指定的签署位置之外签署，默认使用业务分类中的「签署位置规则」配置
        createContractRequest.setExtraSign(true);
        //是否自动生成会签节点 （用于签署动作中传入多个印章并且未指定签署人）默认false
        createContractRequest.setAutoCreateCounterSign(true);
        //设置接收方必须签署全部合同文档
        createContractRequest.setSignAll("RECEIVER");
        //0 发送全部通知；1 仅发送短信通知；2 仅发送邮件通知
        createContractRequest.setMsgCode(0);

        //为合同添加其他文档
        if(documentIdList.size()>0){
            createContractRequest.setDocuments(documentIdList);
        }

        //合同名称
        createContractRequest.setSubject(contrTitle);
        //合同创建人姓名，不传创建人的相关信息默认为发起方的名称
//        createContractRequest.setCreatorName(localContract.getFirstPartyUserName());
        //合同创建人联系方式，联系方式允许使用大陆、台湾和香港的手机号，台湾和香港的手机号格式为：区号+空格+手机号，例：852 9xxxxxxx
//        createContractRequest.setCreatorContact(localContract.getFirstPartyUserPhone());
        //发起方名称
//        createContractRequest.setTenantName(localContract.getFirstPartyCompanyName());
        createContractRequest.setTenantName("重庆盛普物资有限公司");

        //签署方信息
        List<Signatory> signatories = new ArrayList<Signatory>();
        //转让方
        Signatory signatory1 = new Signatory();
        //接收人联系方式：手机号、电子邮箱号（签署方为个人时联系方式（contact）或编号（receiverNumber）至少传递一个）
//        signatory1.setContact();
        //	签署方类型：COMPANY（企业），PERSONAL（个人）
        signatory1.setTenantType(TenantType.COMPANY);
        //签署方名称
        signatory1.setTenantName(ContextUtil.getCurrentOrgName());
        signatory1.setSerialNo(1); // 签署顺序，从1开始
        //接收人姓名
//        signatory1.setReceiverName(localContract.getVendorUserName());
//
//        //受让方
//        Signatory signatory2 = new Signatory();
//        //接收人联系方式：手机号、电子邮箱号（签署方为个人时联系方式（contact）或编号（receiverNumber）至少传递一个）
//        signatory2.setContact(localContract.getMatCompanyUserPhone());
//        //	签署方类型：COMPANY（企业），PERSONAL（个人）
//        signatory2.setTenantType(TenantType.COMPANY);
//        //签署方名称
//        signatory2.setTenantName(localContract.getMatCompanyName());
//        signatory2.setSerialNo(1); // 签署顺序，从1开始
//        //接收人姓名
//        signatory2.setReceiverName(localContract.getMatCompanyUserName());
//
//        // 添加签署方
        signatories.add(signatory1);
//        signatories.add(signatory2);
        createContractRequest.setSignatories(signatories);

        //用印流程id
        createContractRequest.setCategoryId(contrStreamId);

        //文档参数
        createContractRequest.setDocumentParams(params);

        Long contractId = null;
        try {
            //创建合同
            contractId = contractService.createContractByCategory(createContractRequest);
        } catch (PrivateAppException e) {
            System.out.println(LocalDateTime.now()+"发起签署失败:"+e.getMessage());
            throw new BaseException("发起签署失败:"+e.getMessage());
        }

        System.out.println("创建合同成功:"+contractId);

        //状态改为签订中【2】
//        localContract.setStatus(2);
        data.setBidSignStatus(2);
        data.setBidContractId(String.valueOf(contractId));

        //合同浏览页面方法调用
        ViewUrlRequest viewRequest = new ViewUrlRequest();
        viewRequest.setContractId(contractId);
        viewRequest.setPageType(ViewUrlRequest.PageType.CONTENT);
        String viewUrl = null;
        try {
            viewUrl = contractService.viewUrl(viewRequest);
        } catch (PrivateAppException e) {
            System.out.println(LocalDateTime.now()+"签署预览失败:"+e.getMessage());
            throw new BaseException("签署预览失败:"+e.getMessage());
        }
        System.out.println("合同查看链接:"+viewUrl);

        data.setBidContractUrl(viewUrl);
        accessoryService.updateById(data);

        return viewUrl;
    }


}
