/*
 * ✨ 请求封装器工具类函数 ✨
 * */

/**
 * 检查并处理分页请求体
 * @param {Object} [body = {}] 分页请求对象
 * @param {Object} [body.pageBean = {}] 分片配置对象
 * @param {Array} [body.querys = []] 查询配置数组
 * @param {Array} [body.sorter = []] 排序配置数组
 * @return {Object} 处理后的分页请求配置对象
 * @example
 * refineRequestBody({pageBean:{},querys:[],sorter:[]})
 * */
export function refineRequestBody(body = {}) {
    // 如果请求体为空对象，则返回空对象
    if (!Object.keys(body).length) return { pageBean: { pageSize: -1 } }

    // 辅助函数：判断值是否为空（空字符串、空数组、null 或 undefined）
    const _isEmpty = (value) =>
        value === '' ||
        (Array.isArray(value) && !value.length) ||
        value === null ||
        value === undefined

    const { pageBean = { pageSize: -1 }, querys = [], sorter = [] } = body

    return {
        pageBean,
        querys: querys.flatMap((item) => {
            const { defaultValue, ...i } = item
            let value = item.value

            if (_isEmpty(value)) {
                // 如果 value 为空，则检查 defaultValue
                value = defaultValue
                // 如果 defaultValue 也为空，则移除该查询对象
                if (_isEmpty(value)) return []
            }

            // 如果为LIKE查询时，处理为字符串格式
            if (i.operation === 'LIKE' && Array.isArray(value)) {
                value = value.join()
            }

            // 返回处理后的查询对象，加入默认属性
            return {
                group: 'main',
                operation: 'EQUAL',
                relation: 'AND',
                ...i,
                value
            }
        }),
        // 过滤掉没有 direction 属性的排序对象
        sorter: sorter.filter((i) => i.direction)
    }
}

/**
 * 用于更改后端分页查询配置项中的查询字段值
 * @param {Object} params 分页请求对象
 * @param {Array} params.querys 查询配置数组
 * @param {String} props 字段名
 * @param {any} value 字段值
 * @return void
 * */
export function setQueryValue(params = {}, props = '', value = '', operation = '') {
    params?.querys?.forEach((i) => {
        if (i.property.trim() !== props.trim()) return
        if (this?.$set) {
            this?.$set(i, 'value', value)
            operation ? this?.$set(i, 'operation', operation) : ''
        } else {
            i.value = value
            operation ? (i.operation = operation) : ''
        }
    })
}

/**
 * 用于查詢后端分页查询配置项中的字段值
 * @param {Object} params 分页请求对象
 * @param {Array} params.querys 查询配置数组
 * @param {String} props 字段名
 * @return {any} 字段值
 * */
export function getQueryValue(params = {}, props = '') {
    const { value, defaultValue } = params?.querys?.find((i) => i.property.trim() === props.trim())
    // 辅助函数：判断值是否为空（空字符串、空数组、null 或 undefined）
    const isEmpty = (value) =>
        value === '' ||
        (Array.isArray(value) && !value.length) ||
        value === null ||
        value === undefined
    let result
    if (isEmpty(value)) {
        if (!isEmpty(defaultValue)) result = defaultValue
    } else {
        result = value
    }
    return result
}

/**
 * 用于查詢后端分页查询配置项中的query绑定元素对象
 * @deprecated 自 ”2024-6-24“ 起废弃，不建议继续使用。
 * @param {Object} params 分页请求对象
 * @param {Array} params.querys 查询配置数组
 * @param {String} props 字段名
 * @return {Object} query绑定元素对象
 * */
export function getQueryItem(params = {}, props = '') {
    const index = params?.querys?.findIndex((i) => i.property.trim() === props.trim())
    return index !== -1 ? params?.querys[index] : {}
}

/**
 * 用于更改后端分页排序配置项中的排序规则
 * @param {Object} params 分页请求对象
 * @param {Array} [params.sorter = []] 排序配置数组
 * @param {String} props 字段名
 * @param {String} direction 排序方式
 * @return void
 * */
export function setSorterDirection(params = {}, props = '', direction = '') {
    params?.sorter?.forEach((i) => {
        if (i.property.trim() !== props.trim()) {
            i.direction = ''
        } else {
            i.direction = direction
        }
    })
}

/**
 * 用于查詢后端分页排序配置项中的排序规则
 * @param {Object} params 分页请求对象
 * @param {Array} [params.sorter = []] 排序配置数组
 * @param {Boolean} isNotFullName 排序方式规则字符串是否全称
 * @return {Object} sorter绑定排序对象
 * */
export function getSorterDirection(params = {}, isNotFullName = false) {
    const { sorter = [] } = params || {}
    if (!(Array.isArray(sorter) && sorter.length)) return {}
    const { property: prop = '', direction: order = '' } = sorter.find((i) => i.direction) || {}
    if (!prop) return {}
    const orderOpt = { ASC: 'ascending', DESC: 'descending' }
    return { prop, order: isNotFullName ? orderOpt[order] || '' : order }
}
